/*
 * Decompiled with CFR 0.152.
 */
package org.mule.extension.redis.internal.connection;

import java.io.Serializable;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import org.mule.extension.redis.internal.connection.ExecutableRedisConnection;
import org.mule.extension.redis.internal.connection.RedisObjectStoreExpiryTask;
import org.mule.extension.redis.internal.error.exceptions.InvalidDataException;
import org.mule.extension.redis.internal.service.RedisAPIService;
import org.mule.extension.redis.internal.service.factory.ServiceFactory;
import org.mule.runtime.api.i18n.I18nMessageFactory;
import org.mule.runtime.api.scheduler.Scheduler;
import org.mule.runtime.api.serialization.ObjectSerializer;
import org.mule.runtime.api.store.ObjectStoreException;
import org.mule.runtime.api.store.ObjectStoreSettings;
import org.mule.runtime.api.store.TemplateObjectStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RedisObjectStore
extends TemplateObjectStore<Serializable> {
    private static final Logger logger = LoggerFactory.getLogger(RedisObjectStore.class);
    private final RedisAPIService service;
    private final Integer entryTTL;
    private final Integer maxEntries;
    private final long expirationInterval;
    private ObjectSerializer objectSerializer;
    private Scheduler scheduler;
    private ScheduledFuture<?> scheduledTask;
    private String name;

    public RedisObjectStore(ExecutableRedisConnection redisConnection, ObjectSerializer objectSerializer, Integer connectionTtl, ObjectStoreSettings settings, String name, Scheduler scheduler) {
        this.service = ServiceFactory.create().getService(redisConnection);
        Integer ttl = settings.getEntryTTL().map(Long::intValue).orElse(connectionTtl);
        if (ttl == null) {
            this.entryTTL = null;
        } else {
            this.entryTTL = ttl / 1000;
            if (connectionTtl != null && this.entryTTL > connectionTtl) {
                throw new InvalidDataException("The entry TTL set for the custom object store is greater than the one set in the Connection configuration");
            }
        }
        this.maxEntries = settings.getMaxEntries().orElse(null);
        if (!settings.isPersistent()) {
            logger.warn("You are trying to configure a transient object store on a system that is persistent. The PERSISTENT setting of the Object Store will be ignored");
        }
        this.name = name;
        this.expirationInterval = settings.getExpirationInterval();
        this.objectSerializer = objectSerializer;
        this.scheduler = scheduler;
    }

    protected boolean doContains(String key) throws ObjectStoreException {
        try {
            return this.service.exists(key);
        }
        catch (Exception e) {
            throw new ObjectStoreException((Throwable)e);
        }
    }

    public boolean isPersistent() {
        return true;
    }

    public void clear() throws ObjectStoreException {
        try {
            String[] keys = (String[])this.service.keys("*").stream().map(String::new).distinct().toArray(String[]::new);
            if (keys.length != 0) {
                this.service.del(keys);
            }
        }
        catch (Exception e) {
            throw new ObjectStoreException((Throwable)e);
        }
    }

    public void open() throws ObjectStoreException {
        if (this.expirationInterval > 0L) {
            try {
                this.scheduledTask = this.scheduler.scheduleWithFixedDelay((Runnable)new RedisObjectStoreExpiryTask(this), 0L, this.expirationInterval, TimeUnit.MILLISECONDS);
            }
            catch (Exception e) {
                throw new ObjectStoreException(I18nMessageFactory.createStaticMessage((String)("ObjectStore expiry task could not be scheduled for object store: " + this.name)), (Throwable)e);
            }
        }
    }

    public void close() {
        if (this.scheduledTask != null) {
            this.scheduledTask.cancel(true);
        }
    }

    public List<String> allKeys() throws ObjectStoreException {
        try {
            return this.service.keys("*").stream().map(String::new).collect(Collectors.toList());
        }
        catch (Exception e) {
            throw new ObjectStoreException((Throwable)e);
        }
    }

    public Map<String, Serializable> retrieveAll() throws ObjectStoreException {
        try {
            return this.service.keys("*").stream().map(String::new).collect(Collectors.toMap(Function.identity(), k -> this.fromByteArray(this.service.get((String)k))));
        }
        catch (Exception e) {
            throw new ObjectStoreException((Throwable)e);
        }
    }

    protected void doStore(String key, Serializable value) throws ObjectStoreException {
        try {
            this.service.set(key, this.toByteArray(value), this.entryTTL, false);
        }
        catch (Exception e) {
            throw new ObjectStoreException((Throwable)e);
        }
    }

    public Serializable doRetrieve(String key) throws ObjectStoreException {
        try {
            return this.fromByteArray(this.service.get(key));
        }
        catch (Exception e) {
            throw new ObjectStoreException((Throwable)e);
        }
    }

    protected Serializable doRemove(String key) throws ObjectStoreException {
        try {
            return this.service.del(key);
        }
        catch (Exception e) {
            throw new ObjectStoreException((Throwable)e);
        }
    }

    public RedisAPIService getService() {
        return this.service;
    }

    public Integer getMaxEntries() {
        return this.maxEntries;
    }

    private byte[] toByteArray(Serializable serializable) {
        return this.objectSerializer.getInternalProtocol().serialize((Object)serializable);
    }

    private Serializable fromByteArray(byte[] bytes) {
        return (Serializable)this.objectSerializer.getInternalProtocol().deserialize(bytes);
    }
}

