/*
 * Decompiled with CFR 0.152.
 */
package org.mule.extension.ldap.internal.connection.jndi;

import com.google.common.base.Strings;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.URL;
import java.security.KeyStore;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import javax.naming.Binding;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.BasicAttributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.ModificationItem;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.InitialLdapContext;
import javax.naming.ldap.LdapContext;
import javax.naming.ldap.StartTlsRequest;
import javax.naming.ldap.StartTlsResponse;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.TrustManagerFactory;
import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.mule.extension.ldap.api.parameters.Referral;
import org.mule.extension.ldap.internal.connection.LDAPConnection;
import org.mule.extension.ldap.internal.error.LDAPErrorType;
import org.mule.extension.ldap.internal.error.exception.LDAPException;
import org.mule.extension.ldap.internal.error.exception.util.ExceptionUtils;
import org.mule.extension.ldap.internal.model.LDAPEntry;
import org.mule.extension.ldap.internal.model.LDAPEntryAttribute;
import org.mule.extension.ldap.internal.model.LDAPEntryAttributeTypeDefinition;
import org.mule.extension.ldap.internal.model.LDAPEntryAttributes;
import org.mule.extension.ldap.internal.model.LDAPSearchControls;
import org.mule.extension.ldap.internal.model.datasense.LDAPEntryObjectClassDefinition;
import org.mule.extension.ldap.internal.util.ActiveDirectoryUUIDByteParser;
import org.mule.extension.ldap.internal.util.LDAPJNDIUtils;
import org.mule.extension.ldap.internal.util.LDAPResultSet;
import org.mule.extension.ldap.internal.util.LDAPResultSetFactory;
import org.mule.extension.ldap.internal.util.LDAPSSLSocketFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LDAPJNDIConnection
extends LDAPConnection {
    private static final Logger logger = LoggerFactory.getLogger(LDAPJNDIConnection.class);
    public static final int DEFAULT_MAX_POOL_CONNECTIONS = 0;
    public static final int DEFAULT_INITIAL_POOL_CONNECTIONS = 0;
    public static final long DEFAULT_POOL_TIMEOUT = 0L;
    public static final String DEFAULT_INITIAL_CONTEXT_FACTORY = "com.sun.jndi.ldap.LdapCtxFactory";
    public static final String DEFAULT_REFERRAL = Referral.IGNORE.toString();
    private static final boolean IGNORE_CASE = true;
    private static final String INITIAL_CONTEXT_FACTORY_ATTR = "initialContextFactory";
    private static final String POOL_ENABLED_ENV_PARAM = "com.sun.jndi.ldap.connect.pool";
    private static final String MAX_POOL_SIZE_ENV_PARAM = "com.sun.jndi.ldap.connect.pool.maxsize";
    private static final String INIT_POOL_SIZE_ENV_PARAM = "com.sun.jndi.ldap.connect.pool.initsize";
    private static final String TIME_OUT_ENV_PARAM = "com.sun.jndi.ldap.connect.pool.timeout";
    private static final String AUTHENTICATION_ENV_PARAM = "com.sun.jndi.ldap.pool.authentication";
    private static final String BINARY_ATTRIBUTE = "java.naming.ldap.attributes.binary";
    private String providerUrl = null;
    private int maxPoolConnections = 0;
    private int initialPoolSizeConnections = 0;
    private long poolTimeout = 0L;
    private String authentication = "none";
    private String initialContextFactory = "com.sun.jndi.ldap.LdapCtxFactory";
    private String referral = DEFAULT_REFERRAL;
    private Map<String, String> extendedEnvironment = null;
    private LoadingCache<String, LDAPEntryAttributeTypeDefinition> schemaCache = null;
    private LdapContext conn = null;
    private StartTlsResponse tls = null;
    private static final String TRUST_STORE_PATH_EXT_CONF = "org.mule.module.ldap.trustStorePath";
    private static final String TRUST_STORE_PASSWORD_EXT_CONF = "org.mule.module.ldap.trustStorePassword";
    private String trustStorePath;
    private String trustStorePassword;

    public LDAPJNDIConnection() {
    }

    public LDAPJNDIConnection(String providerUrl) {
        this(providerUrl, DEFAULT_INITIAL_CONTEXT_FACTORY);
    }

    public LDAPJNDIConnection(String providerUrl, String initialContextFactory) {
        this(providerUrl, initialContextFactory, "none");
    }

    public LDAPJNDIConnection(String providerUrl, String initialContextFactory, String authentication, int maxPoolConnections, int initialPoolSizeConnections, long poolTimeout) {
        this(providerUrl, initialContextFactory, authentication, maxPoolConnections, initialPoolSizeConnections, poolTimeout, false);
    }

    public LDAPJNDIConnection(String providerUrl, String initialContextFactory, String authentication) {
        this(providerUrl, initialContextFactory, authentication, 0, 0, 0L);
    }

    public LDAPJNDIConnection(String providerUrl, String initialContextFactory, String authentication, int maxPoolConnections, int initialPoolSizeConnections, long poolTimeout, boolean schemaEnabled) {
        this();
        this.setProviderUrl(providerUrl);
        this.setInitialContextFactory(initialContextFactory);
        this.setAuthentication(authentication);
        this.setMaxPoolConnections(maxPoolConnections);
        this.setInitialPoolSizeConnections(initialPoolSizeConnections);
        this.setPoolTimeout(poolTimeout);
        this.setSchemaEnabled(schemaEnabled);
        this.initializeCache();
    }

    private synchronized void initializeCache() {
        if (this.isSchemaEnabled() && this.schemaCache == null) {
            this.schemaCache = CacheBuilder.newBuilder().maximumSize(1000L).build((CacheLoader)new CacheLoader<String, LDAPEntryAttributeTypeDefinition>(){

                public LDAPEntryAttributeTypeDefinition load(String attributeName) {
                    return LDAPJNDIConnection.this.retrieveAttributeTypeDefinition(attributeName);
                }
            });
        }
    }

    @Override
    protected void initialize(Map<String, String> conf) {
        if (conf != null) {
            this.setInitialConfig(conf);
        }
    }

    private void setInitialConfig(Map<String, String> conf) {
        logger.debug("Initial config");
        this.extendedEnvironment = new HashMap<String, String>(conf);
        this.extendedEnvironment.remove("type");
        this.setAuthentication(this.getConfValue(conf, "authentication", "none"));
        this.extendedEnvironment.remove("authentication");
        this.setInitialContextFactory(this.getConfValue(conf, INITIAL_CONTEXT_FACTORY_ATTR, DEFAULT_INITIAL_CONTEXT_FACTORY));
        this.extendedEnvironment.remove(INITIAL_CONTEXT_FACTORY_ATTR);
        this.setInitialPoolSizeConnections(this.getConfValue(conf, "initialPoolSize", 0));
        this.extendedEnvironment.remove("initialPoolSize");
        this.setMaxPoolConnections(this.getConfValue(conf, "maxPoolSize", 0));
        this.extendedEnvironment.remove("maxPoolSize");
        this.setPoolTimeout(this.getConfValue(conf, "poolTimeout", 0L));
        this.extendedEnvironment.remove("poolTimeout");
        this.setProviderUrl(this.getConfValue(conf, "url", null));
        this.extendedEnvironment.remove("url");
        this.setReferral(this.getConfValue(conf, "referral", DEFAULT_REFERRAL));
        this.extendedEnvironment.remove("referral");
        this.setSchemaEnabled("true".equals(this.getConfValue(conf, "schema-enabled", String.valueOf(false))));
        this.initializeCache();
        this.extendedEnvironment.remove("schema-enabled");
        this.setTlsEnabled("true".equals(this.getConfValue(conf, "tls-enabled", String.valueOf(false))));
        this.extendedEnvironment.remove("tls-enabled");
        this.setSslEnabled("true".equals(this.getConfValue(conf, "ssl-enabled", String.valueOf(false))));
        this.extendedEnvironment.remove("ssl-enabled");
        this.setTrustStore(this.getConfValue(conf, TRUST_STORE_PATH_EXT_CONF, this.getConfValue(conf, "truststore-path", null)));
        this.extendedEnvironment.remove(TRUST_STORE_PATH_EXT_CONF);
        this.extendedEnvironment.remove("truststore-path");
        this.setTrustStorePassword(this.getConfValue(conf, TRUST_STORE_PASSWORD_EXT_CONF, this.getConfValue(conf, "truststore-password", null)));
        this.extendedEnvironment.remove(TRUST_STORE_PASSWORD_EXT_CONF);
        this.extendedEnvironment.remove("truststore-password");
        logger.debug("Initial config completed");
    }

    private String getConfValue(Map<String, String> conf, String key, String defaultValue) {
        String value = conf.get(key);
        return StringUtils.isNotEmpty((String)value) && !value.equals("null") ? value : defaultValue;
    }

    private int getConfValue(Map<String, String> conf, String key, int defaultValue) {
        String value = conf.get(key);
        return StringUtils.isNotEmpty((String)value) ? Integer.parseInt(value) : defaultValue;
    }

    private long getConfValue(Map<String, String> conf, String key, long defaultValue) {
        String value = conf.get(key);
        return StringUtils.isNotEmpty((String)value) ? Long.parseLong(value) : defaultValue;
    }

    private void logConfiguration(String bindDn) {
        StringBuilder conf = new StringBuilder();
        conf.append("{");
        conf.append("tls: ").append(this.isTlsEnabled()).append(", ");
        conf.append("ssl: ").append(this.isSslEnabled()).append(", ");
        conf.append("url: ").append(this.getProviderUrl()).append(", ");
        conf.append("authentication: ").append(this.getAuthentication()).append(", ");
        if (!this.isNoAuthentication() && StringUtils.isNotEmpty((String)bindDn)) {
            conf.append("authDn: ").append(bindDn).append(", ");
        } else {
            conf.append("authDn: {anonymous}, ");
        }
        conf.append("initialContextFactory: ").append(this.getInitialContextFactory()).append(", ");
        conf.append("referral: ").append(this.getReferral()).append(", ");
        if (this.isConnectionPoolEnabled()) {
            conf.append("initialPoolSize: ").append(this.getInitialPoolSizeConnections()).append(", ");
            conf.append("maxPoolSize: ").append(this.getMaxPoolConnections()).append(", ");
            conf.append("poolTimeout: ").append(this.getPoolTimeout());
        } else {
            conf.append("pool: disabled");
        }
        if (this.extendedEnvironment != null && !this.extendedEnvironment.isEmpty()) {
            conf.append(", extended: ").append(this.extendedEnvironment);
        }
        conf.append("}");
        if (logger.isDebugEnabled()) {
            logger.debug("{}", (Object)conf);
        }
    }

    public boolean isNoAuthentication() {
        return "none".equalsIgnoreCase(this.getAuthentication());
    }

    @Override
    public boolean isClosed() {
        return this.conn == null;
    }

    @Override
    public void close() {
        logger.debug("Closing connection");
        if (!this.isClosed()) {
            String connectionId = this.toString();
            try {
                this.getConn().close();
                logger.info("Connection {} closed.", (Object)connectionId);
            }
            catch (NamingException nex) {
                logger.error("Close connection {} failed. {}", (Object)connectionId, (Object)nex);
            }
            finally {
                this.setConn(null);
                this.tls = null;
                logger.info("TLS has been set to null after connection is closed.");
            }
        } else {
            logger.warn("Connection already closed.");
        }
    }

    private Map<String, String> buildEnvironment(String dn, String password) {
        HashMap<String, String> env = new HashMap<String, String>();
        if (this.getReferral() != null) {
            env.put("java.naming.referral", this.getReferral());
        }
        env.put("java.naming.security.authentication", this.getAuthentication());
        if (!this.isNoAuthentication()) {
            env.put("java.naming.security.principal", dn);
            env.put("java.naming.security.credentials", password);
        }
        env.put("java.naming.factory.initial", this.getInitialContextFactory());
        env.put("java.naming.provider.url", this.getProviderUrl());
        env.put(BINARY_ATTRIBUTE, "objectGUID objectSid");
        if (this.isConnectionPoolEnabled()) {
            env.put(POOL_ENABLED_ENV_PARAM, "true");
            env.put(AUTHENTICATION_ENV_PARAM, this.getAuthentication());
            if (this.getMaxPoolConnections() > 0) {
                env.put(MAX_POOL_SIZE_ENV_PARAM, String.valueOf(this.getMaxPoolConnections()));
            }
            if (this.getInitialPoolSizeConnections() > 0) {
                env.put(INIT_POOL_SIZE_ENV_PARAM, String.valueOf(this.getInitialPoolSizeConnections()));
            }
            if (this.getPoolTimeout() > 0L) {
                env.put(TIME_OUT_ENV_PARAM, String.valueOf(this.getPoolTimeout()));
            }
        } else {
            env.put(POOL_ENABLED_ENV_PARAM, "false");
        }
        if (this.extendedEnvironment != null && !this.extendedEnvironment.isEmpty()) {
            env.putAll(this.extendedEnvironment);
        }
        if (this.isSslEnabled()) {
            env.put("java.naming.security.protocol", "ssl");
        }
        logger.debug("Created environment without authentication credentials: {}", env);
        return env;
    }

    @Override
    public void rebind() {
        logger.debug("Rebinding");
        if (this.isClosed()) {
            throw new LDAPException("Cannot rebind a close connection. You must first bind.", LDAPErrorType.CONNECTIVITY);
        }
        String dn = this.getBindedUserDn();
        String password = this.getBindedUserPassword();
        this.bind(dn, password);
    }

    @Override
    public void bind(String dn, String password) {
        LdapContext ldapContext = null;
        logger.debug("Binding - DN: {}", (Object)dn);
        try {
            logger.debug("Binding - Is connection closed: {}", (Object)this.isClosed());
            if (!this.isClosed()) {
                String currentUrl = (String)this.getConn().getEnvironment().get("java.naming.provider.url");
                String currentAuth = (String)this.getConn().getEnvironment().get("java.naming.security.authentication");
                String currentDn = this.getBindedUserDn();
                logger.info("Already binded to {} with {} authentication as {}. Closing connection first.", new Object[]{currentUrl, currentAuth, currentDn != null ? currentDn : "anonymous"});
                this.close();
                logger.info("Re-binding to {} with {} authentication as {}", new Object[]{this.getProviderUrl(), this.getAuthentication(), dn});
            }
            this.logConfiguration(dn);
            if (!"none".equals(this.getAuthentication()) && dn == null) {
                throw new LDAPException("Invalid Credentials: dn cannot be null.", LDAPErrorType.INVALID_ATTRIBUTE);
            }
            dn = LDAPJNDIUtils.escapeMetaCharacters(dn);
            ldapContext = this.getLDAPContext(dn, password);
            this.setConn(ldapContext);
            logger.info("Binded to {} with {} authentication as {}", new Object[]{this.getProviderUrl(), this.getAuthentication(), dn});
        }
        catch (NamingException nex) {
            this.silentyCloseDirContext(ldapContext);
            throw this.handleNamingException(nex, "Bind failed.");
        }
        catch (Exception ex) {
            this.silentyCloseDirContext(ldapContext);
            throw new LDAPException(LDAPErrorType.UNKNOWN, ex);
        }
        logger.debug("Binding - Complete");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private LdapContext getLDAPContext(String dn, String password) throws NamingException {
        InitialLdapContext ldapContext = null;
        Hashtable<String, String> env = new Hashtable<String, String>();
        env.put(BINARY_ATTRIBUTE, "objectGUID objectSid");
        Map<String, String> buildEnv = this.buildEnvironment(dn, password);
        Set<String> keys = buildEnv.keySet();
        for (String key : keys) {
            String value;
            if (key == null || (value = buildEnv.get(key)) == null) continue;
            env.put(key, value);
        }
        Thread currentThread = Thread.currentThread();
        ClassLoader currentClassLoader = currentThread.getContextClassLoader();
        currentThread.setContextClassLoader(LDAPJNDIConnection.class.getClassLoader());
        try {
            logger.debug("Binding - TLS: {} | SSL: {}", (Object)this.isTlsEnabled(), (Object)this.isSslEnabled());
            if (this.isTlsEnabled() || this.isSslEnabled()) {
                Map<String, String> postEncryptEnv = this.removeAuthenticationConfigurationFromEnvironment(env);
                if (this.isSslEnabled()) {
                    this.initSsl();
                }
                ldapContext = new InitialLdapContext(env, null);
                if (this.isTlsEnabled()) {
                    this.initTls(ldapContext);
                }
                this.applyAuthenticationConfiguration(postEncryptEnv, ldapContext);
            } else {
                ldapContext = new InitialLdapContext(env, null);
            }
        }
        finally {
            currentThread.setContextClassLoader(currentClassLoader);
        }
        return ldapContext;
    }

    private void validateTrustStore() {
        logger.debug("TrustStore validation - Path: {} | Is Password set; {}", (Object)this.trustStorePath, (Object)(!Strings.isNullOrEmpty((String)this.trustStorePassword) ? 1 : 0));
        if (!Strings.isNullOrEmpty((String)this.trustStorePath) || !Strings.isNullOrEmpty((String)this.trustStorePassword)) {
            Optional<File> trustStoreFileOptional = this.getTrustStoreFile();
            if (!trustStoreFileOptional.isPresent()) {
                logger.debug("TrustStore validation - Error: Invalid TrustStore path {}", (Object)this.trustStorePath);
                throw new LDAPException(LDAPErrorType.CONNECTIVITY, new Exception("Wrong TrustStore path: " + this.trustStorePath));
            }
            try {
                KeyStore ks = KeyStore.getInstance(KeyStore.getDefaultType());
                try (FileInputStream kos = new FileInputStream(trustStoreFileOptional.get());){
                    ks.load(kos, this.trustStorePassword.toCharArray());
                }
            }
            catch (Exception e) {
                logger.debug("TrustStore validation | Error: TrustStore file exists in {} but the password is incorrect", (Object)this.trustStorePath);
                throw new LDAPException(LDAPErrorType.CONNECTIVITY, new Exception("Failed to validate password for " + this.trustStorePath + ": " + e.getMessage(), e));
            }
        }
    }

    private void initSsl() {
        logger.debug("SSL initialization");
        this.validateTrustStore();
        try {
            KeyStore ks = KeyStore.getInstance(KeyStore.getDefaultType());
            try (FileInputStream kos = new FileInputStream(this.getTrustStoreFile().get());){
                ks.load(kos, this.trustStorePassword.toCharArray());
            }
            SSLContext sslContext = SSLContext.getInstance("SSL");
            TrustManagerFactory factory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
            factory.init(ks);
            KeyManagerFactory keyManagerFactory = KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
            keyManagerFactory.init(ks, this.trustStorePassword.toCharArray());
            sslContext.init(keyManagerFactory.getKeyManagers(), factory.getTrustManagers(), new SecureRandom());
            SSLContext.setDefault(sslContext);
        }
        catch (Exception e) {
            throw new LDAPException(LDAPErrorType.CONNECTIVITY, new Exception("Failed to set custom TrustStore", e));
        }
    }

    private void applyAuthenticationConfiguration(Map<String, String> postEncryptEnv, LdapContext conn) throws NamingException {
        logger.debug("Applying authentication configuration");
        for (Map.Entry<String, String> entry : postEncryptEnv.entrySet()) {
            logger.debug("Applying authentication configuration. Key: {} | Value: {}", (Object)entry.getKey(), (Object)entry.getValue());
            conn.addToEnvironment(entry.getKey(), entry.getValue());
        }
    }

    private Map<String, String> removeAuthenticationConfigurationFromEnvironment(Map<String, String> env) {
        HashMap<String, String> authEnv = new HashMap<String, String>();
        logger.debug("Removing authentication configuration from environment");
        for (String key : Arrays.asList("java.naming.security.authentication", "java.naming.security.credentials", "java.naming.security.principal", "java.naming.security.protocol")) {
            String value = env.remove(key);
            if (value != null) {
                authEnv.put(key, value);
            }
            logger.debug("Removing authentication configuration from environment. Key {} | Value {}", (Object)key, (Object)value);
        }
        return authEnv;
    }

    private void initTls(LdapContext conn) {
        try {
            logger.debug("Enabling TLS");
            this.tls = (StartTlsResponse)conn.extendedOperation(new StartTlsRequest());
            Optional<File> trustStoreFileOptional = this.getTrustStoreFile();
            if (trustStoreFileOptional.isPresent()) {
                this.validateTrustStore();
                LDAPSSLSocketFactory ssf = new LDAPSSLSocketFactory(trustStoreFileOptional.get(), this.trustStorePassword);
                SSLSession negotiate = this.tls.negotiate(ssf);
                logger.info("TLS enabled successfully using protocol {}  with clientkeystore.jks", (Object)negotiate.getProtocol());
            } else {
                SSLSession negotiate = this.tls.negotiate();
                logger.info("TLS enabled successfully using protocol {} ", (Object)negotiate.getProtocol());
            }
        }
        catch (NamingException nex) {
            this.silentCloseTls();
            throw this.handleNamingException(nex, "TLS initialization failed.");
        }
        catch (IOException ex) {
            this.silentCloseTls();
            throw this.handleException(ex, "TLS negotiation failed.");
        }
    }

    private Optional<File> getTrustStoreFile() {
        logger.debug("Getting trustStore file {}", (Object)this.trustStorePath);
        if (Objects.isNull(this.trustStorePath)) {
            logger.debug("Getting trustStore file - Undefined path, returning empty file");
            return Optional.empty();
        }
        File trustStoreFile = new File(this.trustStorePath);
        if (!trustStoreFile.exists()) {
            URL resource = this.getClass().getClassLoader().getResource(this.trustStorePath);
            logger.debug("Failed to get file, retrying on absolute path.");
            if (Objects.isNull(resource)) {
                logger.error("Failed to get file on absolute path");
                return Optional.empty();
            }
            trustStoreFile = new File(resource.getPath());
            logger.debug("Getting trustStore file - Found file in {}", (Object)resource.getPath());
        }
        return Optional.of(trustStoreFile);
    }

    private void silentCloseTls() {
        logger.debug("Silently closing TLS");
        try {
            if (this.tls != null) {
                this.tls.close();
            }
        }
        catch (Exception ex) {
            logger.info("Closing TLS connection failed.", (Throwable)ex);
        }
        finally {
            this.tls = null;
        }
    }

    private String getBindedUserPassword() {
        try {
            return (String)this.getConn().getEnvironment().get("java.naming.security.credentials");
        }
        catch (NamingException nex) {
            throw this.handleNamingException(nex, "Cannot get binded user password.");
        }
    }

    @Override
    public String getBindedUserDn() {
        if (!this.isClosed()) {
            try {
                return (String)this.getConn().getEnvironment().get("java.naming.security.principal");
            }
            catch (NamingException nex) {
                throw this.handleNamingException(nex, "Cannot get binded user DN.");
            }
        }
        return null;
    }

    @Override
    public LDAPResultSet search(String baseDn, String filter, LDAPSearchControls controls) {
        return this.doSearch(baseDn, filter, null, controls, this.buildLDAPContext(controls));
    }

    @Override
    public LDAPResultSet search(String baseDn, String filter, Object[] filterArgs, LDAPSearchControls controls) {
        return this.doSearch(baseDn, filter, filterArgs, controls, this.buildLDAPContext(controls));
    }

    public LdapContext buildLDAPContext(LDAPSearchControls controls) {
        try {
            return controls.isPagingEnabled() ? this.getLDAPContext(this.getBindedUserDn(), this.getBindedUserPassword()) : this.getConn();
        }
        catch (NamingException nex) {
            throw this.handleNamingException(nex, "Unable to construct a valid LDAP Context.");
        }
    }

    private LDAPResultSet doSearch(String baseDn, String filter, Object[] filterArgs, LDAPSearchControls controls, LdapContext searchConn) {
        try {
            logger.debug("Searching. BaseDN: {} | Filter: {}", (Object)baseDn, (Object)filter);
            if (filter.toLowerCase().contains("objectguid=")) {
                filter = ActiveDirectoryUUIDByteParser.replaceFilterGUID(filter);
            }
            NamingEnumeration<SearchResult> entries = filterArgs != null && filterArgs.length > 0 ? searchConn.search(baseDn, filter, filterArgs, LDAPJNDIUtils.buildSearchControls(controls)) : searchConn.search(baseDn, filter, LDAPJNDIUtils.buildSearchControls(controls));
            return LDAPResultSetFactory.create(baseDn, filter, filterArgs, searchConn, controls, entries, this.isSchemaEnabled() ? this : null);
        }
        catch (NamingException nex) {
            throw this.handleNamingException(nex, "Search failed.");
        }
    }

    @Override
    public LDAPEntry lookup(String dn) {
        try {
            logger.debug("LookUp. DN: {}", (Object)dn);
            return LDAPJNDIUtils.buildEntry(dn, this.getConn().getAttributes(dn), this.isSchemaEnabled() ? this : null);
        }
        catch (NamingException nex) {
            throw this.handleNamingException(nex, "Lookup of entry " + dn + " failed.");
        }
    }

    @Override
    public LDAPEntry lookup(@NotNull String dn, String[] attributes) {
        try {
            logger.debug("LookUp. DN: {}", (Object)dn);
            return LDAPJNDIUtils.buildEntry(dn, this.getConn().getAttributes(dn, attributes), this.isSchemaEnabled() ? this : null);
        }
        catch (NamingException nex) {
            throw this.handleNamingException(nex, "Lookup of entry " + dn + " failed.");
        }
    }

    @Override
    public void addEntry(@NotNull LDAPEntry entry) {
        try {
            this.getConn().bind(entry.getDn(), null, this.buildAttributes(entry));
        }
        catch (NamingException nex) {
            throw this.handleNamingException(nex, "Add entry " + entry.getDn() + " failed.");
        }
    }

    private LDAPException handleNamingException(NamingException nex, String logMessage) {
        logger.warn(logMessage, (Throwable)nex);
        return ExceptionUtils.buildLdapException(nex);
    }

    private LDAPException handleException(Exception ex, String logMessage) {
        logger.warn(logMessage, (Throwable)ex);
        return new LDAPException(ex.getMessage(), LDAPErrorType.UNKNOWN, ex);
    }

    @Override
    public void updateEntry(@NotNull LDAPEntry entry) {
        try {
            ModificationItem[] mods = new ModificationItem[entry.getAttributeCount()];
            Iterator<LDAPEntryAttribute> it = entry.attributes();
            for (int i = 0; it.hasNext() && i < mods.length; ++i) {
                mods[i] = new ModificationItem(2, this.buildBasicAttribute(it.next()));
            }
            this.getConn().modifyAttributes(LDAPJNDIUtils.escapeMetaCharacters(entry.getDn()), mods);
        }
        catch (NamingException nex) {
            throw this.handleNamingException(nex, "Update entry " + entry.getDn() + " failed.");
        }
    }

    @Override
    public void deleteEntry(LDAPEntry entry) {
        this.deleteEntry(entry.getDn());
    }

    @Override
    public void deleteEntry(String dn) {
        try {
            logger.debug("About to delete entry {} ", (Object)dn);
            this.getConn().unbind(LDAPJNDIUtils.escapeMetaCharacters(dn));
            logger.info("Deleted entry {} ", (Object)dn);
        }
        catch (NamingException nex) {
            throw this.handleNamingException(nex, "Delete entry failed.");
        }
    }

    @Override
    public void renameEntry(String oldDn, String newDn) {
        try {
            logger.debug("About to rename entry {} to {}", (Object)oldDn, (Object)newDn);
            this.getConn().rename(LDAPJNDIUtils.escapeMetaCharacters(oldDn), LDAPJNDIUtils.escapeMetaCharacters(newDn));
            logger.info("Renamed entry {} to {} ", (Object)oldDn, (Object)newDn);
        }
        catch (NamingException nex) {
            throw this.handleNamingException(nex, "Rename entry " + oldDn + " to " + newDn + " failed.");
        }
    }

    @Override
    public void addAttribute(@NotNull String dn, @NotNull LDAPEntryAttribute attribute) {
        try {
            ModificationItem[] mods = new ModificationItem[]{new ModificationItem(1, this.buildBasicAttribute(attribute))};
            this.getConn().modifyAttributes(LDAPJNDIUtils.escapeMetaCharacters(dn), mods);
        }
        catch (NamingException nex) {
            throw this.handleNamingException(nex, "Add attribute " + attribute.getName() + " to entry " + dn + " failed.");
        }
    }

    @Override
    public void updateAttribute(@NotNull String dn, @NotNull LDAPEntryAttribute attribute) {
        try {
            ModificationItem[] mods = new ModificationItem[]{new ModificationItem(2, this.buildBasicAttribute(attribute))};
            this.getConn().modifyAttributes(LDAPJNDIUtils.escapeMetaCharacters(dn), mods);
        }
        catch (NamingException nex) {
            throw this.handleNamingException(nex, "Update attribute " + attribute.getName() + " from entry " + dn + " failed.");
        }
    }

    @Override
    public void deleteAttribute(@NotNull String dn, @NotNull LDAPEntryAttribute attribute) {
        try {
            ModificationItem[] mods = new ModificationItem[]{new ModificationItem(3, this.buildBasicAttribute(attribute))};
            this.getConn().modifyAttributes(LDAPJNDIUtils.escapeMetaCharacters(dn), mods);
        }
        catch (NamingException nex) {
            throw this.handleNamingException(nex, "Delete attribute " + attribute.getName() + " from entry " + dn + " failed.");
        }
    }

    private void silentyCloseDirContext(DirContext ctx) {
        if (ctx != null) {
            try {
                ctx.close();
            }
            catch (NamingException nex) {
                logger.warn("Cannot close directory context", (Throwable)nex);
            }
        }
    }

    @Override
    public synchronized LDAPEntryAttributeTypeDefinition getAttributeTypeDefinition(@NotNull String attributeName) {
        if (this.schemaCache != null) {
            try {
                return (LDAPEntryAttributeTypeDefinition)this.schemaCache.get((Object)attributeName);
            }
            catch (ExecutionException ex) {
                logger.error("Could not retrieve attribute type definition for attribute " + attributeName + " from cache. Trying to retrieve directly.", (Throwable)ex);
                return this.retrieveAttributeTypeDefinition(attributeName);
            }
        }
        logger.info("Schema cache disabled. Retrieving attribute type definition directly.");
        return this.retrieveAttributeTypeDefinition(attributeName);
    }

    protected LDAPEntryAttributeTypeDefinition retrieveAttributeTypeDefinition(@NotNull String attributeName) {
        DirContext schema = null;
        DirContext attrSchema = null;
        try {
            schema = this.getConn().getSchema("");
            logger.debug("About to retrieve attribute definition for attribute {}", (Object)attributeName);
            attrSchema = (DirContext)schema.lookup("AttributeDefinition/" + attributeName);
            LDAPEntryAttributeTypeDefinition lDAPEntryAttributeTypeDefinition = LDAPJNDIUtils.buildAttributeTypeDefinition(attrSchema.getAttributes(""));
            this.silentyCloseDirContext(attrSchema);
            this.silentyCloseDirContext(schema);
            return lDAPEntryAttributeTypeDefinition;
        }
        catch (NamingException nex) {
            try {
                throw this.handleNamingException(nex, "Get attribute type definition for attribute " + attributeName + " failed.");
            }
            catch (Throwable throwable) {
                this.silentyCloseDirContext(attrSchema);
                this.silentyCloseDirContext(schema);
                throw throwable;
            }
        }
    }

    @Override
    public List<String> getAllObjectClasses() {
        DirContext schema = null;
        try {
            schema = this.getConn().getSchema("");
            logger.debug("About to retrieve all object classes");
            NamingEnumeration<Binding> bindings = schema.listBindings("ClassDefinition");
            ArrayList<String> objectClasses = new ArrayList<String>(200);
            while (bindings.hasMore()) {
                Binding binding = bindings.next();
                objectClasses.add(binding.getName());
            }
            ArrayList<String> arrayList = objectClasses;
            return arrayList;
        }
        catch (NamingException nex) {
            throw this.handleNamingException(nex, "Get all object classes failed.");
        }
        finally {
            this.silentyCloseDirContext(schema);
        }
    }

    @Override
    public LDAPEntryObjectClassDefinition getObjectClassDefinition(String objectClassName) {
        DirContext schema = null;
        DirContext attrSchema = null;
        try {
            schema = this.getConn().getSchema("");
            logger.debug("About to retrieve class definition for objectClass {}", (Object)objectClassName);
            attrSchema = (DirContext)schema.lookup("ClassDefinition/" + objectClassName);
            LDAPEntryObjectClassDefinition lDAPEntryObjectClassDefinition = LDAPJNDIUtils.buildObjectClassDefinition(attrSchema.getAttributes(""));
            this.silentyCloseDirContext(attrSchema);
            this.silentyCloseDirContext(schema);
            return lDAPEntryObjectClassDefinition;
        }
        catch (NamingException nex) {
            try {
                throw this.handleNamingException(nex, "Get class definition for objectClass " + objectClassName + " failed.");
            }
            catch (Throwable throwable) {
                this.silentyCloseDirContext(attrSchema);
                this.silentyCloseDirContext(schema);
                throw throwable;
            }
        }
    }

    @Override
    public String getAuthentication() {
        return this.authentication;
    }

    @Override
    public void setAuthentication(String authentication) {
        this.authentication = authentication;
    }

    public int getInitialPoolSizeConnections() {
        return this.initialPoolSizeConnections;
    }

    public void setInitialPoolSizeConnections(int initialPoolSizeConnections) {
        this.initialPoolSizeConnections = initialPoolSizeConnections;
    }

    public int getMaxPoolConnections() {
        return this.maxPoolConnections;
    }

    public void setMaxPoolConnections(int maxPoolConnections) {
        this.maxPoolConnections = maxPoolConnections;
    }

    public long getPoolTimeout() {
        return this.poolTimeout;
    }

    public void setPoolTimeout(long poolTimeout) {
        this.poolTimeout = poolTimeout;
    }

    public String getProviderUrl() {
        return this.providerUrl;
    }

    public void setProviderUrl(String provider) {
        this.providerUrl = provider;
    }

    public boolean isConnectionPoolEnabled() {
        return this.getInitialPoolSizeConnections() > 0;
    }

    public String getInitialContextFactory() {
        return this.initialContextFactory;
    }

    public void setInitialContextFactory(String initialContextFactory) {
        this.initialContextFactory = initialContextFactory;
    }

    private LdapContext getConn() {
        if (this.conn == null) {
            logger.debug("Tried to get a closed connection");
            throw new LDAPException("Connection is closed. Call bind method first.", LDAPErrorType.CONNECTIVITY);
        }
        return this.conn;
    }

    private void setConn(LdapContext conn) {
        this.conn = conn;
    }

    private Attributes buildAttributes(LDAPEntryAttributes attrs) {
        BasicAttributes attributes = new BasicAttributes(true);
        Iterator<LDAPEntryAttribute> it = attrs.attributes();
        while (it.hasNext()) {
            attributes.put(this.buildBasicAttribute(it.next()));
        }
        return attributes;
    }

    private Attributes buildAttributes(@NotNull LDAPEntry entry) {
        return this.buildAttributes(entry.getAttributes());
    }

    private BasicAttribute buildBasicAttribute(@NotNull LDAPEntryAttribute attribute) {
        if (attribute.isMultiValued()) {
            BasicAttribute basicAttribute = new BasicAttribute(attribute.getName());
            for (Object o : attribute.getValues()) {
                basicAttribute.add(o);
            }
            return basicAttribute;
        }
        return new BasicAttribute(attribute.getName(), attribute.getValue());
    }

    public String getReferral() {
        return this.referral;
    }

    public void setReferral(String referral) {
        this.referral = referral;
    }

    public String toString() {
        try {
            String user = this.getBindedUserDn();
            return (user != null ? user : "anonymous") + "@" + this.getProviderUrl();
        }
        catch (Exception exception) {
            return "{unknown}@" + this.getProviderUrl();
        }
    }

    @Override
    public void setTrustStore(String trustStorePath) {
        this.trustStorePath = trustStorePath;
    }

    @Override
    public void setTrustStorePassword(String trustStorePassword) {
        this.trustStorePassword = trustStorePassword;
    }

    @Override
    public boolean isValid() {
        return this.conn != null;
    }
}

