/*
 * Copyright (c) 2025, Salesforce, Inc.,
 * All rights reserved.
 * For full license text, see the LICENSE.txt file
 */
%dw 2.7
/**
* This module has serialization helpers.
*
* Formats supported:
* - Query parameters (as described by OAS)
* - URI parameters (as described by OAS)
* - Header parameters (as described by OAS)
* - Cookie parameters (as described by OAS)
*/
import encodeURIComponent from dw::core::URL
import HttpRequestCookies from dw::io::http::Types

/**
* A type representing _empty_ values for a given parameter.
*
* See [`allowEmptyValue`] on the OAS Spec for more info.
*
* [`allowEmptyValue`]: https://spec.openapis.org/oas/latest.html#fixed-fields-9
*/
type SerializableEmpty = Null

/**
* A type representing _primitive_ values.
*
* There's no normative definition of _primitive_:
* - OAS differentiates between `primitive`, `array` and `object` [1]
* - JSONSchema says `array` and `object` are primitive types [2]
*
* [1]: https://spec.openapis.org/oas/latest.html#style-values
* [2]: https://datatracker.ietf.org/doc/html/draft-bhutton-json-schema-00#section-4.2.1
*/
type SerializableScalar =  StringCoerceable

/**
* A type representing an object with _primitive_ values.
*/
type SerializableObject =  { _?: SerializableScalar }

/**
* A type representing an array with _primitive_ elements.
*/
type SerializableArray = Array<SerializableScalar>

/**
* A type representing the union of all serializable values
*/
type SerializableAny = SerializableScalar | SerializableObject | SerializableArray | SerializableEmpty

/**
* A type representing a parameter dictionary ready for serialization.
*
* This means an object whose values are all serializable according
* to `SerializableAny`
*/
type SerializableParams = {
    _?: SerializableAny
}

/**
* The serialization [styles](https://spec.openapis.org/oas/latest.html#style-values)
* supported for query parameters.
*/
type QueryParamStyle = "form" | "spaceDelimited" | "pipeDelimited" | "deepObject" | "asJSON"

type BodyParamStyle = "form" | "spaceDelimited" | "pipeDelimited" | "deepObject"

/**
* The serialization [styles](https://spec.openapis.org/oas/latest.html#style-values)
* supported for URI parameters.
*/
type UriParamStyle = "matrix" | "label" | "simple" | "asJSON"

/**
* The serialization [styles](https://spec.openapis.org/oas/latest.html#style-values)
* supported for headers parameters
*/
type HeaderStyle = "simple" | "asJSON"

/**
* The serialization [styles](https://spec.openapis.org/oas/latest.html#style-values)
* supported for cookie parameters
*/
type CookieStyle = "form" | "asJSON"

/**
* Serialization configuration for a given parameter.
*/
type ParamConfig<Style <: String> = {
    /**
    * Which serialization style to use
    */
    style?: Style,
    /**
    * On non-primitive parameters: whether to serialize everything as a single
    * parameter or to split it into multiple ones.
    */
    explode?: Boolean
}

/**
* Serialization configuration for a given query parameter.
*/
type QueryParamConfig = ParamConfig<QueryParamStyle>

/**
* Serialization configuration for a given URI parameter.
*/
type UriParamConfig = ParamConfig<UriParamStyle>

/**
* Serialization configuration for a given headers parameter.
*/
type HeaderConfig = ParamConfig<HeaderStyle>

/**
* Serialization configuration for a given cookies parameter.
*/
type CookieConfig = ParamConfig<CookieStyle>

/**
* An object whose entries specify the serialization configuration for each
* parameter. Missing entries will be interpreted as their corresponding default.
*/
type SerializationConfig<Style <: String> = {
    _?: ParamConfig<Style>
}

@Internal(permits=[])
fun encodeEntries(value: SerializableObject, separator: String, joiner: String): String =
    value
        pluck "$(encodeURIComponent($$))$(separator)$(encodeURIComponent($ as String))"
        joinBy joiner

@Internal(permits=[])
fun encodeItems(value: SerializableArray, joiner: String): String =
    value
        map "$(encodeURIComponent($ as String))"
        joinBy joiner

@Internal(permits=[])
fun serializeQueryParam(key: String, value: SerializableObject, style: 'form', explode: false): String =
    "$(encodeURIComponent(key))=$(encodeEntries(value, ',', ','))"

@Internal(permits=[])
fun serializeBodyParam(key: String, value: SerializableObject, style: 'form', explode: false): Object =
        (key) : encodeEntries(value as SerializableObject, ',', ',')

@Internal(permits=[])
fun serializeQueryParam(key: String, value: SerializableObject, style: 'form', explode: true): String =
    encodeEntries(value, '=', '&')

@Internal(permits=[])
fun serializeBodyParam(key: String, value: SerializableObject, style: 'form', explode: true): Object =
    value mapObject ((v, k, index) -> {
        (k) : v
    })

@Internal(permits=[])
fun serializeQueryParam(key: String, value: SerializableArray, style: 'form', explode: true): String = do {
    var encodedKey = "$(encodeURIComponent(key))"
    ---
    value
        map "$(encodedKey)=$(encodeURIComponent($ as String))"
        joinBy '&'
}

@Internal(permits=[])
fun serializeBodyParam(key: String, value: SerializableArray, style: 'form', explode: true): Object =
    (key) : value

@Internal(permits=[])
fun serializeQueryParam(key: String, value: SerializableArray, style: 'form', explode: false): String =
    "$(encodeURIComponent(key))=$(encodeItems(value, ','))"

@Internal(permits=[])
fun serializeBodyParam(key: String, value: SerializableArray, style: 'form', explode: false): Object =
    (key) : encodeItems(value, ',')

@Internal(permits=[])
fun serializeQueryParam(key: String, value: SerializableScalar, style: 'form', explode: Boolean): String =
    "$(encodeURIComponent(key))=$(encodeURIComponent(value as String))"

@Internal(permits=[])
fun serializeBodyParam(key: String, value: SerializableScalar, style: 'form', explode: Boolean): Object =
    (key) : value

@Internal(permits=[])
fun serializeQueryParam(key: String, value: SerializableEmpty, style: 'form', explode: Boolean): String =
    "$(encodeURIComponent(key))="

@Internal(permits=[])
fun serializeBodyParam(key: String, value: SerializableEmpty, style: 'form', explode: Boolean): Object =
    (key) : value

@Internal(permits=[])
fun serializeQueryParam(key: String, value: SerializableArray, style: 'spaceDelimited', explode: false): String =
    "$(encodeURIComponent(key))=$(encodeURIComponent(encodeItems(value, ' ')))"

@Internal(permits=[])
fun serializeBodyParam(key: String, value: SerializableArray, style: 'spaceDelimited', explode: false): Object =
    (key) : encodeItems(value, ' ')

@Internal(permits=[])
fun serializeBodyParam(key: String, value: SerializableArray, style: 'spaceDelimited', explode: true): Object =
    (key) : value

@Internal(permits=[])
fun serializeQueryParam(key: String, value: SerializableObject, style: 'spaceDelimited', explode: false): String =
    "$(encodeURIComponent(key))=$(encodeURIComponent(encodeEntries(value, ' ', ' ')))"

@Internal(permits=[])
fun serializeBodyParam(key: String, value: SerializableObject, style: 'spaceDelimited', explode: false): Object =
    (key) : value

@Internal(permits=[])
fun serializeQueryParam(key: String, value: SerializableArray, style: 'pipeDelimited', explode: false): String =
    "$(encodeURIComponent(key))=$(encodeItems(value, '|'))"

@Internal(permits=[])
fun serializeBodyParam(key: String, value: SerializableArray, style: 'pipeDelimited', explode: false): Object =
    (key) : encodeItems(value, '|')

@Internal(permits=[])
fun serializeBodyParam(key: String, value: SerializableArray, style: 'pipeDelimited', explode: true): Object =
    (key) : value

@Internal(permits=[])
fun serializeQueryParam(key: String, value: SerializableObject, style: 'pipeDelimited', explode: false): String =
    "$(encodeURIComponent(key))=$(encodeEntries(value, '|', '|'))"


@Internal(permits=[])
fun serializeBodyParam(key: String, value: SerializableObject, style: 'pipeDelimited', explode: false): Object =
    (key) : value

// True nested objects are not yet supported, see: https://github.com/OAI/OpenAPI-Specification/issues/1706
@Internal(permits=[])
fun serializeQueryParam(key: String, value: SerializableObject, style: 'deepObject', explode: true): String = do {
    var encodedKey = encodeURIComponent(key)
    ---
    value
        pluck "$(encodedKey)[$(encodeURIComponent($$))]=$(encodeURIComponent($ as String))"
        joinBy '&'
}

@Internal(permits=[])
fun serializeBodyParam(key: String, value: SerializableObject, style: 'deepObject', explode: true): Object =
    value mapObject ((v, k, index) ->  {
        "$(key)[$(k)]" : v
    })

@Internal(permits=[])
fun serializeQueryParam(key: String, value: SerializableAny, style: 'asJSON', explode: false): String =
    "$(encodeURIComponent(key))=$(encodeURIComponent((write(value, 'application/json') as String)))"

@Internal(permits=[])
fun serializeUriParam(key: String, value: SerializableArray, style: 'matrix', explode: false): String =
    ";$(encodeURIComponent(key))=$(encodeItems(value, ','))"

@Internal(permits=[])
fun serializeUriParam(key: String, value: SerializableArray, style: 'matrix', explode: true): String =
    ";" ++ (value map "$(encodeURIComponent(key))=$(encodeURIComponent($ as String))" joinBy ";")

@Internal(permits=[])
fun serializeUriParam(key: String, value: SerializableObject, style: 'matrix', explode: false): String =
    ";$(encodeURIComponent(key))=$(encodeEntries(value, ',', ','))"

@Internal(permits=[])
fun serializeUriParam(key: String, value: SerializableObject, style: 'matrix', explode: true): String =
    ";" ++ "$(encodeEntries(value, '=', ';'))"

@Internal(permits=[])
fun serializeUriParam(key: String, value: SerializableScalar, style: 'matrix', explode: Boolean): String =
    ";$(encodeURIComponent(key))=$(encodeURIComponent(value as String))"

@Internal(permits=[])
fun serializeUriParam(key: String, value: SerializableEmpty, style: 'matrix', explode: Boolean): String =
    ";$(encodeURIComponent(key))"

@Internal(permits=[])
fun serializeUriParam(key: String, value: SerializableObject, style: 'label', explode: false): String =
    ".$(encodeEntries(value, ',', ','))"

@Internal(permits=[])
fun serializeUriParam(key: String, value: SerializableObject, style: 'label', explode: true): String =
    ".$(encodeEntries(value, '=', '.'))"

@Internal(permits=[])
fun serializeUriParam(key: String, value: SerializableArray, style: 'label', explode: false): String =
    ".$(encodeItems(value, ','))"

@Internal(permits=[])
fun serializeUriParam(key: String, value: SerializableArray, style: 'label', explode: true): String =
    ".$(encodeItems(value, '.'))"

@Internal(permits=[])
fun serializeUriParam(key: String, value: SerializableScalar, style: 'label', explode: Boolean): String =
    ".$(encodeURIComponent(value as String))"

@Internal(permits=[])
fun serializeUriParam(key: String, value: SerializableEmpty, style: 'label', explode: Boolean): String =
    "."

@Internal(permits=[])
fun serializeUriParam(key: String, value: SerializableObject, style: 'simple', explode: false): String =
    "$(encodeEntries(value, ',', ','))"

@Internal(permits=[])
fun serializeUriParam(key: String, value: SerializableObject, style: 'simple', explode: true): String =
    "$(encodeEntries(value, '=', ','))"

@Internal(permits=[])
fun serializeUriParam(key: String, value: SerializableArray, style: 'simple', explode: Boolean): String =
    "$(encodeItems(value, ','))"

@Internal(permits=[])
fun serializeUriParam(key: String, value: SerializableScalar, style: 'simple', explode: Boolean): String =
    "$(encodeURIComponent(value as String))"

@Internal(permits=[])
fun serializeUriParam(key: String, value: SerializableAny, style: 'asJSON', explode: false): String =
    encodeURIComponent("$(write(value, 'application/json') as String)")

@Internal(permits=[])
fun serializeHeader(value: SerializableScalar, style: 'simple', explode: Boolean): String =
    "$(value)"

@Internal(permits=[])
fun serializeHeader(value: SerializableArray, style: 'simple', explode: Boolean): String =
    "$(value map ($ as String) joinBy ',')"

@Internal(permits=[])
fun serializeHeader(value: SerializableObject, style: 'simple', explode: false): String =
    value
        pluck "$($$)$(',')$($ as String)"
        joinBy ','

@Internal(permits=[])
fun serializeHeader(value: SerializableObject, style: 'simple', explode: true): String =
    value
        pluck "$($$)$('=')$($ as String)"
        joinBy ','

@Internal(permits=[])
fun serializeHeader(value: SerializableAny, style: 'asJSON', explode: false): String =
    "$(write(value, 'application/json') as String)"

@Internal(permits=[])
fun serializeCookie(key: String, value: SerializableScalar, style: 'form', explode: Boolean): HttpRequestCookies =
    {"$(key)":"$(value as String)"}

@Internal(permits=[])
fun serializeCookie(key: String, value: SerializableArray, style: 'form', explode: false): HttpRequestCookies =
    {"$(key)":"$(value map ($ as String) joinBy ',')"}

@Internal(permits=[])
fun serializeCookie(key: String, value: SerializableObject, style: 'form', explode: false): HttpRequestCookies =
    {"$(key)" :  (value
                    pluck "$($$)$(',')$($ as String)"
                    joinBy ',')}

@Internal(permits=[])
fun serializeCookie(key: String, value: SerializableArray, style: 'form', explode: true): HttpRequestCookies =
    {"$(key)":"$(value map ($ as String) joinBy '&')"}

@Internal(permits=[])
fun serializeCookie(key: String, value: SerializableObject, style: 'form', explode: true): HttpRequestCookies =
    value mapObject (value,key,index) -> {"$(key)":"$(value as String)"}

@Internal(permits=[])
fun serializeCookie(key: String, value: SerializableEmpty, style: 'form', explode: Boolean): HttpRequestCookies =
    {"$(key)=":""}

@Internal(permits=[])
fun serializeCookie(key: String, value: SerializableAny, style: 'asJSON', explode: false): HttpRequestCookies =
    {"$(key)":"$(write(value, 'application/json') as String)"}

@Internal(permits=[])
fun effectiveConfig<Style <: String>(key: String, value: Any, config: SerializationConfig<Style>, defaultStyle: Style): { style: Style, explode: Boolean } = do {
    // Comments from: https://spec.openapis.org/oas/latest.html#fixed-fields-9
    // Default values (based on value of `in`): for `query` - `form`; for `path` - `simple`; for `header` - `simple`; for `cookie` - `form`.
    var style: Style = value.^style as Style
        default config[key].style
        default defaultStyle
    // When style is form, the default value is true. For all other styles, the default value is false.
    var explode: Boolean = value.^explode as Boolean
        default config[key].explode
        default style == 'form'
    ---
    {
        style: style,
        explode: explode
    }
}

/**
* Encodes a list of query parameters into a suitable string representation.
*
* Accepts an optional `config` parameter where the `style` and `explode`
* characteristics of the serialization can be configured per-parameter.
* An additional way to configure serialization is to include `style` and
* `explode` fields on the metadata of parameter values.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `queryParams` | Object | An object where each entry is one of the query parameters, all parameters should be Serializable as specified by `SerializableAny`
* | `config` | SerializationConfig<QueryParamStyle> | An object where each entry is the default configuration of a given header. Missing entries will receive the default configuration as per OAS spec.
* |===
*
* === Example
*
* This example shows how the `serializeQueryParam` function behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.7
* import serializeQueryParams from com::mulesoft::connectivity::transport::Serialization
* output application/json
* ---
* serializeQueryParams({
*   lightColors: ["red", "green", "blue"],
*   inkColors: ["cyan", "magenta", "yellow", "black"],
*   flagColors: ["sky blue", "white", "yellow"] <~ { style: "pipeDelimited" }
* }, { inkColors: { explode: false } })
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* "lightColors=red&lightColors=green&lightColors=blue&inkColors=cyan,magenta,yellow,black&flagColors=sky%20blue|white|yellow"
* ----
*/
fun serializeQueryParams(queryParams: Object, config: SerializationConfig<QueryParamStyle> = {}) =
    queryParams pluck ((value, key) -> do {
        var conf = effectiveConfig(key, value, config, 'form')
        var style: QueryParamStyle = conf.style
        ---
        serializeQueryParam(key as String, value as SerializableAny, style, conf.explode) as String
    })
    joinBy '&'

/**
* Encodes each field on a body object into its suitable string representation.
* This is specially useful when encoding bodies with an
* `application/x-www-form-urlencoded` MIME.
*
* Accepts an optional `config` parameter where the `style` and `explode`
* characteristics of the serialization can be configured per-parameter.
* An additional way to configure serialization is to include `style` and
* `explode` fields on the metadata of parameter values.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `bodyParams` | Object | An object where each entry is one of the body parameters, all parameters should be Serializable as specified by `SerializableAny`
* | `config` | SerializationConfig<BodyParamStyle> | An object where each entry is the default configuration of a given param. Missing entries will receive the default configuration as per OAS spec.
* |===
*
* === Example
*
* This example shows how the `serializeBodyParams` function behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.7
* import serializeBodyParams from com::mulesoft::connectivity::transport::Serialization
* output application/json
* ---
* serializeBodyParams({
*   uParam: {
*     role: "dev",
*     firstName: "Maxi"
*   }
* }, { uParam: { style: "deepObject", explode: true } })
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "uParam[role]": "dev",
*   "uParam[firstName]": "Maxi"
* }
* ----
*/
fun serializeBodyParams(bodyParams: Object, config: SerializationConfig<BodyParamStyle> = {}) =
    bodyParams mapObject ((value, key) -> do {
        var conf = effectiveConfig(key, value, config, 'form')
        var style: BodyParamStyle = conf.style
        ---
        if (value is Object)
            serializeBodyParam(key as String, value as SerializableObject, style, conf.explode)
        else if (value is Array)
            serializeBodyParam(key as String, value as SerializableArray, style, conf.explode)
        else
            (key) : value
    })

/**
* Encodes each URI parameter entry from a given object into its string
* representation.
*
* Accepts an optional `config` parameter where the `style` and `explode`
* characteristics of the serialization can be configured per-parameter.
* An additional way to configure serialization is to include `style` and
* `explode` fields on the metadata of parameter values.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `uriParams` | Object | An object where each entry is one of the URI parameters, all parameters should be Serializable as specified by `SerializableAny`.
* | `config` | SerializationConfig<UriParamStyle> | An object where each entry is the default configuration of a given header. Missing entries will receive the default configuration as per OAS spec.
* |===
*
* === Example
*
* This example shows how the `serializeUriParam` function behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.7
* import serializeUriParam from com::mulesoft::connectivity::transport::Serialization
* output application/json
* ---
* serializeUriParams({
*   member: { role: "dev", firstName: "Maxi" }
* }, {
*   member: { style: "simple", explode: true }
* })
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "member": "role=dev,firstName=Maxi"
* }
* ----
*/
fun serializeUriParams(uriParams: Object, config: SerializationConfig<UriParamStyle> = {}): { _?: String } =
    uriParams mapObject (value, key) -> do {
        var conf = effectiveConfig(key, value, config, 'simple')
        var style: UriParamStyle = conf.style
        ---
        (key): serializeUriParam(key as String, value as SerializableAny, style, conf.explode) as String
    }

/**
* Encodes each header parameter entry from a given object into its string
* representation.
*
* Accepts an optional `config` parameter where the `style` and `explode`
* characteristics of the serialization can be configured per-parameter.
* An additional way to configure serialization is to include `style` and
* `explode` fields on the metadata of parameter values.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `headers` | Object | An object where each entry is one of the header parameters, all parameters should be Serializable as specified by `SerializableAny`
* | `config` | SerializationConfig<HeaderStyle> | An object where each entry is the default configuration of a given header. Missing entries will receive the default configuration as per OAS spec.
* |===
*
* === Example
*
* This example shows how the `serializeHeaders` function behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.7
* import serializeHeaders from com::mulesoft::connectivity::transport::Serialization
* output application/json
* ---
* serializeHeaders({
*   "X-Header1": [ "cyan", "magenta", "yellow" ],
*   "X-Header2": [ "cyan", "magenta", "yellow" ] <~ { style: "simple", explode: false },
* }, {
*   "X-Header1": {style: "simple", explode: false}
* })
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "X-Header1": "cyan,magenta,yellow",
*   "X-Header2": "cyan,magenta,yellow"
* }
* ----
*/
fun serializeHeaders(headers: Object, config: SerializationConfig<HeaderStyle> = {}): { _?: String } =
    headers mapObject (value, key) -> do {
        var conf = effectiveConfig(key, value, config, 'simple')
        var style: HeaderStyle = conf.style
        ---
        (key): serializeHeader(value as SerializableAny, style, conf.explode) as String
    }

/**
* Encodes a collection of cookie parameters into a suitable string representation.
*
* Serialization configurations can be provided on a per-parameter basis using
* the `config` parameter. When set the `style` and `explode` fields of a given
* parameter value are prioritized over the serialization configuration from
* `conf`.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `cookieParams` | Object | An object where each entry is one of the cookie parameters, all parameters should be Serializable as specified by `SerializableAny`
* | `config` | SerializationConfig<CookiesStyle> | An object where each entry is the default configuration of a given cookie. Missing entries will receive the default configuration as per OAS spec.
* |===
*
* === Example
*
* This example shows how the `serializeCookies` function behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.7
* import serializeCookies from com::mulesoft::connectivity::transport::Serialization
* output application/json
* ---
* serializeCookies({
*   cookieId: [ "cyan", "magenta", "yellow" ],
*   anotherCookie: [ "cyan", "magenta", "yellow" ] <~ { style: "form", explode: true }
* }, {
*   cookieId: { style: "form", explode: false }
* })
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* "cookieId=cyan,magenta,yellow; anotherCookie=cyan&anotherCookie=magenta&anotherCookie=yellow"
* ----
*/
fun serializeCookies(cookieParams: Object, config: SerializationConfig<CookieStyle> = {}): HttpRequestCookies =
    cookieParams mapObject ((value, key) -> do {
        var conf = effectiveConfig(key, value, config, 'form')
        var style: CookieStyle = conf.style
        ---
        serializeCookie(key as String, value as SerializableAny, style, conf.explode)
    })

/**
* Describes the `withSerializationConfig` function purpose.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `v` | T | The type of the input object
* | `config` | SerializationConfig | The serialization config to apply
* |===
*
* === Example
*
* This example shows how to use `withSerializationConfig` in order to configure serialization easily.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.7
* import withSerializationConfig from com::mulesoft::connectivity::transport::Serialization
* output application/dw
* var query = {
*     tags: ["argentina", "technology"]
* }
* ---
* query withSerializationConfig {
*     tags: { style: "form", explode: false }
* }
* ----
*
* ==== Output
*
* [source,DataWeave,linenums]
* ----
* {
*   tags: [
*     "argentina",
*     "technology"
*   ] as Array {style: "form", explode: false}
* }
* ----
*
*/
fun withSerializationConfig<T <: Object>(v: T, config: SerializationConfig): T = do {
    var result: Any = v mapObject (value, key) ->
        if (config[key]?) do {
            var style = value.^style default config[key].style
            var explode = value.^explode default config[key].explode
            ---
            (key): value <~ (value.^ update {
                case .style! if (style != null) -> style
                case .explode! if (explode != null) -> explode
            })
        } else
            (key): value
    ---
    result as T
}
