/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.discovery.core.version.comparator;

import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * The total order defined for sorting versions is: AGW|MULE|EE < SNAPSHOT < EAx < RCx < RELEASE < RELEASE-YYYYMMDD.
 * <p>
 * NOTE: EAx < EAy if and only if x < y. Same logic applies to RC and HF.
 * <p>
 * This comparator checks ONLY the 'modifier' part for a pair of artifact versions.
 */
public class ModifierComparator implements Comparator<String> {

  public static final String SNAPSHOT = "-";
  public static final String RELEASE = "";
  public static final String DATE = "-DATE";
  public static final String HF = "-hf";
  public static final String HFx = "-hfX";
  private final Pattern numberPattern = Pattern.compile("\\d+");
  private final Pattern chPattern = Pattern.compile("-[0-9]{8}");
  private final Map<String, Integer> indexes;

  public ModifierComparator() {
    this.indexes = new HashMap<>();

    indexes.put(SNAPSHOT, 0);
    indexes.put("-ea", 1);
    indexes.put("-rc", 2);
    indexes.put(RELEASE, 3);
    indexes.put(HFx, 4);
    indexes.put(DATE, 5);
    indexes.put(HF, 6);
  }

  @Override
  public int compare(String aModifier, String anotherModifier) {
    String aCleanModifier = cleanUp(aModifier);
    String anotherCleanModifier = cleanUp(anotherModifier);

    int keyAvailables = keyAvailables(aCleanModifier, anotherCleanModifier);
    if (keyAvailables != 0) {
      return keyAvailables;
    }

    int modifierComparison = indexes.get(aCleanModifier) - indexes.get(anotherCleanModifier);

    if (modifierComparison == 0) {
      modifierComparison = number(aModifier) - number(anotherModifier);
    }

    return modifierComparison;
  }

  private String cleanUp(String aModifier) {
    boolean isDate = chPattern.matcher(aModifier).matches();
    if (isDate) {
      return DATE;
    }
    String modSubString = aModifier.substring(0, numberIndex(aModifier));

    if (modSubString.equals(HF)) {
      return modSubString.length() == aModifier.length() ? HF : HFx;
    }

    return modSubString;
  }

  private int number(String aModifier) {
    String stringNumber = numberAsString(aModifier);
    return stringNumber != null ? Integer.parseInt(stringNumber) : 0;
  }

  private int numberIndex(String aModifier) {
    String numberAsString = numberAsString(aModifier);
    int numberLength = numberAsString != null ? numberAsString.length() : 0;
    return aModifier.length() - numberLength;
  }

  private String numberAsString(String s) {
    Matcher m = numberPattern.matcher(s);
    return m.find() ? m.group() : null;
  }

  private int keyAvailables(String aCleanModifier, String anotherCleanModifier) {
    if (indexes.containsKey(aCleanModifier)) {
      if (!indexes.containsKey(anotherCleanModifier)) {
        return 1;
      }
    } else {
      if (indexes.containsKey(anotherCleanModifier)) {
        return -1;
      }
      return 1;
    }
    return 0;
  }
}
