/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.discovery.core.version;

import static java.util.regex.Pattern.CASE_INSENSITIVE;

import com.mulesoft.anypoint.discovery.api.exception.NotParseableVersionException;
import com.mulesoft.anypoint.discovery.api.version.ArtifactVersion;
import com.mulesoft.anypoint.discovery.api.version.ArtifactVersionFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.function.BiFunction;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class MuleArtifactVersionFactory implements ArtifactVersionFactory {

  private final Map<Pattern, BiFunction<Matcher, String, ArtifactVersion>> versionPatternMatcher;

  public static final String majorMinorPatch = "([0-9]+)\\.([0-9]+)\\.([0-9]+)";
  private static final Pattern snapshotPattern = Pattern.compile("^" + majorMinorPatch + "(-SNAPSHOT)$", CASE_INSENSITIVE);
  private static final Pattern monthlySnapshotPattern = Pattern.compile("^" + majorMinorPatch + "(-HF-SNAPSHOT)$", CASE_INSENSITIVE);
  private static final Pattern monthlyPattern = Pattern.compile("^" + majorMinorPatch + "(-)([0-9]{8})$", CASE_INSENSITIVE);
  private static final Pattern eaPattern = Pattern.compile("^" + majorMinorPatch + "(-ea)([0-9]+)$", CASE_INSENSITIVE);
  private static final Pattern rcPattern = Pattern.compile("^" + majorMinorPatch + "(-rc)([0-9]+)$", CASE_INSENSITIVE);
  private static final Pattern releasePattern = Pattern.compile("^" + majorMinorPatch + "$", CASE_INSENSITIVE);

  public MuleArtifactVersionFactory() {
    versionPatternMatcher = new HashMap<>();

    versionPatternMatcher.put(snapshotPattern, (matcher, version) -> new SnapshotVersion(version));
    versionPatternMatcher.put(monthlyPattern, (matcher, version) -> new MonthlyVersion(mmp(matcher), matcher.group(5)));
    versionPatternMatcher.put(monthlySnapshotPattern, (matcher, version) -> new MonthlySnapshotVersion(mmp(matcher)));
    versionPatternMatcher.put(eaPattern, (matcher, version) -> new EarlyAccessVersion(version));
    versionPatternMatcher.put(rcPattern, (matcher, version) -> new ReleaseCandidateVersion(version));
    versionPatternMatcher.put(releasePattern, (matcher, version) -> new ReleaseVersion(version));
  }

  @Override
  public ArtifactVersion create(String version) {
    return versionPatternMatcher.keySet().stream()
        .filter(pattern -> pattern.matcher(version).matches())
        .map(pattern -> versionPatternMatcher.get(pattern).apply(pattern.matcher(version), version))
        .findFirst()
        .orElse(new UnknownArtifactVersion(version));
  }

  private String mmp(Matcher matcher) {
    if (matcher.matches()) {
      return matcher.group(1) + "." + matcher.group(2) + "." + matcher.group(3);
    }
    throw new NotParseableVersionException("Matcher fail to extract groups. " + matcher);
  }

  @Override
  public ArtifactVersion create(int major, int minor, int patch, boolean isSnapshot) {
    if (isSnapshot) {
      return new SnapshotVersion(major, minor, patch);
    }
    return new ReleaseVersion(major, minor, patch);
  }
}
