/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.discovery.core.version;

import static java.lang.Integer.valueOf;
import static java.lang.String.format;
import static java.util.regex.Pattern.CASE_INSENSITIVE;

import com.mulesoft.anypoint.discovery.api.exception.InvalidHotFixVersionException;
import com.mulesoft.anypoint.discovery.api.exception.NotParseableVersionException;
import com.mulesoft.anypoint.discovery.api.version.ArtifactVersion;
import com.mulesoft.anypoint.discovery.api.version.VersionVisitor;
import com.mulesoft.anypoint.discovery.core.version.comparator.MajorMinorPatchComparator;
import com.mulesoft.anypoint.discovery.core.version.comparator.ModifierComparator;
import com.mulesoft.anypoint.discovery.core.version.comparator.SameVersionSnapshotComparator;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

//TODO AGW-2110: Reuse MuleVersion object or org.semver.api
abstract class InternalArtifactVersion implements ArtifactVersion {

  private static final Pattern versionRegex = Pattern.compile("^([0-9]+)\\.([0-9]+)\\.([0-9]+)(-(([a-zA-Z]+[0-9]+((-SNAPSHOT)?))|(SNAPSHOT)|([0-9]{8})|(HF-SNAPSHOT)))?$", CASE_INSENSITIVE);

  private final int major;
  private final int minor;
  private final int patch;
  private final boolean isSnapshot;
  private final String modifier;

  InternalArtifactVersion(String version) {
    version = version.toUpperCase();
    if (!isValid(version)) {
      throw new NotParseableVersionException("Cannot create version for '" + version + "'.");
    }

    Matcher matcher = versionRegex.matcher(version);
    matcher.matches();

    this.major = valueOf(matcher.group(1));
    this.minor = valueOf(matcher.group(2));
    this.patch = valueOf(matcher.group(3));
    String modifier = matcher.group(4);
    this.modifier = modifier != null ? withoutSnapshot(modifier).toLowerCase() : "";
    this.isSnapshot = modifier != null && modifier.toUpperCase().endsWith("-SNAPSHOT");
  }

  public InternalArtifactVersion(int major, int minor, int patch, boolean isSnapshot) {
    this(major + "." + minor + "." + patch + (isSnapshot ? "-SNAPSHOT" : ""));
  }

  public static boolean isValid(String possibleVersion) {
    Matcher matcher = versionRegex.matcher(possibleVersion);
    return matcher.matches();
  }

  @Override
  public int major() {
    return major;
  }

  @Override
  public int minor() {
    return minor;
  }

  @Override
  public int patch() {
    return patch;
  }

  @Override
  public String value() {
    return formattedVersion();
  }

  @Override
  public int hotFixNumber() {
    try {
      return Integer.parseInt(modifier.substring(3));
    } catch (Exception e) {
      throw new InvalidHotFixVersionException("Version " + value() + " is not a valid hot fix version: format must be -hfX where 'X' is an integer.", e);
    }
  }

  @Override
  public boolean olderThan(ArtifactVersion anotherVersion) {
    if (anotherVersion instanceof NullArtifactVersion) {
      return false;
    }

    InternalArtifactVersion other = (InternalArtifactVersion) anotherVersion;

    int mmpComparison = new MajorMinorPatchComparator().compare(this, anotherVersion);
    if (mmpComparison != 0) {
      return mmpComparison < 0;
    }

    String thisModifier = toComparableModifier(this);
    String otherModifier = toComparableModifier(other);
    int modifierComparison = new ModifierComparator().compare(thisModifier, otherModifier);
    if (modifierComparison != 0) {
      return modifierComparison < 0;
    }

    int releaseComparison = new SameVersionSnapshotComparator().compare(isSnapshot, other.isSnapshot);
    return releaseComparison < 0;
  }

  private String toComparableModifier(InternalArtifactVersion version) {
    return version.isSnapshot && version.modifier.equals("") ? "-" : version.modifier;
  }

  @Override
  public boolean isSnapshot() {
    return isSnapshot;
  }

  @Override
  public boolean isHotFix() {
    return modifier.toUpperCase().startsWith("-HF");
  }

  @Override
  public boolean isHotFixOf(ArtifactVersion anotherVersion) {
    return modifier.startsWith("-hf") && modifier.length() > 3 ? new ReleaseVersion(major, minor, patch).equals(anotherVersion) : false;
  }

  @Override
  public boolean isReleaseCandidate() {
    return modifier.startsWith("-rc");
  }

  @Override
  public boolean isChPatch() {
    return modifier.matches("-[0-9]{8}") || upperForChPatch(modifier).equals("-HF");
  }

  @Override
  public boolean isEarlyAccess() {
    return modifier.startsWith("-ea");
  }

  @Override
  public ArtifactVersion release() {
    if(isChPatch()){
      throw new InvalidHotFixVersionException("Cannot create release version '"+toString()+"' as I'm not sure the date. #whatYearIsIt");
    }

    String newModifier = isEarlyAccess() || isReleaseCandidate() ? "" : modifier;
    return new ReleaseVersion(major + "." + minor + "." + patch + newModifier);
  }

  public abstract  <T> T accept(VersionVisitor<T> visitor);

  @Override
  public String toString() {
    return formattedVersion();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o)
      return true;
    if (!(o instanceof InternalArtifactVersion))
      return false;

    InternalArtifactVersion that = (InternalArtifactVersion) o;

    return formattedVersion().equals(that.formattedVersion());
  }

  @Override
  public int hashCode() {
    return formattedVersion().hashCode();
  }

  private String formattedVersion() {
    return format("%d.%d.%d%s%s", major, minor, patch, upperForChPatch(modifier), isSnapshot ? "-SNAPSHOT" : "");
  }

  private String upperForChPatch(String modifier) {
    if(modifier.equals("-hf")){
      modifier = modifier.toUpperCase();
    }
    return modifier;
  }

  private String withoutSnapshot(String modifier) {
    String substring = modifier;
    int index = snapshotIndex(modifier);
    if (index > -1) {
      substring = modifier.substring(0, index);
    }
    return substring != null ? substring : "";
  }

  private int snapshotIndex(String modifier) {
    return modifier != null ? modifier.toUpperCase().indexOf("-SNAPSHOT") : -1;
  }
}
