/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright
 * law. All use of this software is subject to MuleSoft's Master Subscription Agreement
 * (or other master license agreement) separately entered into in writing between you and
 * MuleSoft. If such an agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.discovery.core.aether;

import static java.util.stream.Collectors.toList;

import com.mulesoft.anypoint.discovery.api.RuntimeVersionProvider;
import com.mulesoft.anypoint.discovery.api.exception.NoLatestVersionFoundException;
import com.mulesoft.anypoint.discovery.api.exception.VersionDiscoveryException;
import com.mulesoft.anypoint.discovery.api.version.ArtifactVersion;
import com.mulesoft.anypoint.discovery.core.version.comparator.ArtifactVersionComparator;
import com.mulesoft.anypoint.discovery.api.version.ArtifactVersionFactory;
import com.mulesoft.anypoint.discovery.api.version.ArtifactVersionValidator;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Stream;

import org.eclipse.aether.RepositorySystem;
import org.eclipse.aether.RepositorySystemSession;
import org.eclipse.aether.artifact.Artifact;
import org.eclipse.aether.artifact.DefaultArtifact;
import org.eclipse.aether.repository.RemoteRepository;
import org.eclipse.aether.resolution.VersionRangeResolutionException;
import org.eclipse.aether.resolution.VersionRangeResult;
import org.eclipse.aether.version.Version;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * {@inheritDoc}
 * <p>
 * {@link ArtifactVersion}s are obtained from Nexus repository using repository credentials found in the settings.xml.
 */
public class AetherRuntimeVersionProvider implements RuntimeVersionProvider {

  private static final Logger LOGGER = LoggerFactory.getLogger(AetherRuntimeVersionProvider.class);

  protected final List<ArtifactVersion> allVersions;
  protected final ArtifactVersion latest;
  protected final ArtifactVersionValidator validator;
  protected final ArtifactVersionFactory factory;
  protected final Artifact artifact;

  public AetherRuntimeVersionProvider(RepositorySystem repositorySystem, RepositorySystemSession repositorySystemSession,
                                      List<RemoteRepository> repositories, Product product, ArtifactVersionFactory factory, ArtifactVersionValidator validator)
      throws VersionDiscoveryException {
    this.factory = factory;
    this.artifact = artifact(product);

    this.validator = validator;

    try {
      VersionRangeResult versionRange =
          new AetherVersionProvider(repositorySystem, repositorySystemSession, repositories).findVersions(artifact);

      List<ArtifactVersion> releases = allVersions(versionRange)
          .filter(version -> !version.isSnapshot())
          .collect(toList());
      List<ArtifactVersion> snapshots = allVersions(versionRange)
          .filter(version -> version.isSnapshot() && (version.isChPatch() || !releases.contains(version.release())))
          .collect(toList());
      this.allVersions = new ArrayList<>(releases);
      this.allVersions.addAll(snapshots);
      this.allVersions.sort(new ArtifactVersionComparator());

      LOGGER.debug("All available versions are: " + allVersions);

      if (allVersions.size() > 0) {
        latest = allVersions.get(allVersions.size() - 1);
        LOGGER.debug("Latest version is: " + latest);
      } else {
        throw new NoLatestVersionFoundException("No highest version has been found in Nexus, VersionRangeResult is: "
            + versionRange);
      }
    } catch (VersionRangeResolutionException e) {
      throw new VersionDiscoveryException("An error occurred while resolving versions for product " + product, e);
    }
  }

  /**
   * @return {@inheritDoc}.
   * <p>
   * NOTE: {@link ArtifactVersion} lower than 4.1.1 are filtered.
   */
  @Override
  public List<ArtifactVersion> all() {
    return allVersions;
  }

  @Override
  public ArtifactVersion latest() {
    return latest;
  }

  protected Stream<ArtifactVersion> allVersions(VersionRangeResult versionRange) {
    return versionRange.getVersions().stream()
        .filter(onlyValids())
        .map(toArtifactVersion())
        .filter(greaterThan411())
        .distinct();
  }

  public Predicate<Version> onlyValids() {
    return version -> validator.isValid(version.toString());
  }

  private Function<Version, ArtifactVersion> toArtifactVersion() {
    return version -> factory.create(version.toString());
  }

  public Artifact artifact(Product product) {
    Artifact artifact;
    switch (product) {
      case EE_BOM:
        artifact = muleEEBom();
        break;
      case CE_BOM:
        artifact = muleCEBom();
        break;
      case EE_DISTRO:
        artifact = muleEEDistro();
        break;
      default:
        artifact = muleEEBom();
    }
    return artifact;
  }

  private Predicate<ArtifactVersion> greaterThan411() {
    return version -> !version.olderThan(factory.create("4.1.1"));
  }

  private Artifact muleEEBom() {
    return new DefaultArtifact("com.mulesoft.mule.distributions:mule-ee-bom:(0,]");
  }

  private Artifact muleCEBom() {
    return new DefaultArtifact("org.mule.distributions:mule-bom:(0,]");
  }

  private Artifact muleEEDistro() {
    return new DefaultArtifact("com.mulesoft.mule.distributions:mule-ee-distribution-standalone:(0,]");
  }
}
