/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.pushbase.repository;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.net.Uri;
import android.os.Bundle;
import com.moe.pushlibrary.providers.MoEDataContract;
import com.moe.pushlibrary.providers.MoEDataContract.CampaignListEntity;
import com.moe.pushlibrary.providers.MoEDataContract.MessageEntity;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.core.RemoteConfig;
import com.moengage.pushbase.model.NotificationPayload;
import com.moengage.pushbase.push.MoEngageNotificationUtils;

public class PushDAO {

  private static final String TAG = "PushDAO";

  private PushDAO() {

  }

  private static PushDAO instance;

  public static PushDAO getInstance() {
    if (instance == null) {
      synchronized (PushDAO.class) {
        if (instance == null) instance = new PushDAO();
      }
    }
    return instance;
  }

  /**
   * Save campaign-id for Push-Amp duplicate check.
   *
   * @param context instance of {@link Context}
   * @param campaignId Unique identifier for a campaign.
   */
  public void saveCampaignId(Context context, String campaignId) {
    try {
      ContentValues values = new ContentValues();
      values.put(CampaignListEntity.CAMPAIGN_ID, campaignId);
      values.put(CampaignListEntity.CAMPAIGN_ID_TTL,
          MoEUtils.currentMillis() + RemoteConfig.getConfig().pushAmpCampaignExpiryTime);
      context.getContentResolver().insert(CampaignListEntity.getContentUri(context), values);
    } catch (Exception e) {
      Logger.e(TAG + " saveCampaignId() ", e);
    }
  }

  /**
   * Checks whether the given campaign has been shown before or not.
   *
   * @param context instance of {@link Context}
   * @param campaignId Unique identifier for a campaign.
   * @return true if campaign is already shown, else false
   */
  public boolean doesCampaignExists(Context context, String campaignId) {
    Cursor cursor = null;
    try {
      if (MoEUtils.isEmptyString(campaignId)) return false;
      cursor = context.getContentResolver()
          .query(CampaignListEntity.getContentUri(context),
              new String[] { CampaignListEntity.CAMPAIGN_ID },
              CampaignListEntity.CAMPAIGN_ID + "=?", new String[] { campaignId }, null);
      if (cursor != null && cursor.moveToFirst()) return true;
    } catch (Exception e) {
      Logger.e(TAG + " doesCampaignExists() ", e);
    } finally {
      closeCursor(cursor);
    }
    return false;
  }

  /**
   * Save campaign for showing in app-inbox.
   *
   * @param context instance of {@link Context}
   * @param campaignPayload Push Campaign Payload.
   */
  public void saveCampaign(Context context, NotificationPayload campaignPayload) {
    try {
      String msgDetails =
          MoEngageNotificationUtils.convertBundleToJsonString(campaignPayload.payload);
      ContentValues values = new ContentValues();
      values.put(MoEDataContract.MessageEntity.MSG_DETAILS, msgDetails);
      long gTime =
          campaignPayload.payload.getLong(MoEHelperConstants.EXTRA_MSG_RECEIVED_TIME);
      values.put(MessageEntity.GTIME, gTime);
      values.put(MessageEntity.MSG_CLICKED, 0);
      values.put(MessageEntity.MSG_TTL, campaignPayload.inboxExpiry);
      values.put(MessageEntity.MSG_TAG, campaignPayload.campaignTag);
      values.put(MessageEntity.CAMPAIGN_ID, campaignPayload.campaignId);
      //adding to inbox
      Uri newRecord = context.getContentResolver()
          .insert(MoEDataContract.MessageEntity.getContentUri(context), values);
      if (newRecord != null) {
        Logger.v(TAG + " saveCampaign() : Add a new record with entry: " + newRecord);
      } else {
        Logger.v(TAG + " saveCampaign() : Failed to add notification to inbox.");
      }
    } catch (Exception e) {
      Logger.e(TAG + " saveCampaign() : ", e);
    }
  }

  /**
   * Mark notification as clicked.
   *
   * @param context instance of {@link Context}
   * @param pushPayload Push Campaign payload.
   * @return Number of rows updated.
   */
  public int updateNotificationClick(Context context, Bundle pushPayload) {
    int count = -1;
    try {
      ContentValues values = new ContentValues();
      values.put(MessageEntity.MSG_CLICKED, true);
      String campaignId = pushPayload.getString(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID);
      if (doesCampaignExistInInbox(context, campaignId)) {
        count = context.getContentResolver()
            .update(MessageEntity.getContentUri(context), values,
                MessageEntity.CAMPAIGN_ID + " = ? ", new String[] { campaignId });
      } else {
        long receivedTime = pushPayload.getLong(MoEHelperConstants.EXTRA_MSG_RECEIVED_TIME, -1);
        if (receivedTime == -1) {
          Logger.v(TAG + " updateNotificationClick() : Cannot update click, received time not "
              + "present.");
          return count;
        }
        count = context.getContentResolver()
            .update(MessageEntity.getContentUri(context), values, MessageEntity.GTIME + " = ? ",
                new String[] { String.valueOf(receivedTime) });
      }
      if (count > 0) {
        context.getContentResolver().notifyChange(MessageEntity.getContentUri(context), null);
      }
    } catch (Exception e) {
      Logger.e(TAG + " updateNotificationClick() : ", e);
    }
    return count;
  }

  private boolean doesCampaignExistInInbox(Context context, String campaignId) {
    Cursor cursor = null;
    try {
      if (MoEUtils.isEmptyString(campaignId)) return false;
      cursor = context.getContentResolver()
          .query(MessageEntity.getContentUri(context),
              new String[] { MessageEntity.CAMPAIGN_ID },
              MessageEntity.CAMPAIGN_ID + " = ? ", new String[] { campaignId }, null);
      if (cursor != null && cursor.moveToFirst()) return true;
    } catch (Exception e) {
      Logger.e(TAG + " doesCampaignExistInInbox() : ", e);
    } finally {
      closeCursor(cursor);
    }
    return false;
  }

  private void closeCursor(Cursor cursor) {
    if (cursor != null) cursor.close();
  }
}