/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.pushbase.repository;

import android.os.Bundle;
import androidx.annotation.Nullable;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.ActionMapperConstants;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEUtils;
import com.moengage.core.SdkConfig;
import com.moengage.pushbase.PushActionMapperConstants;
import com.moengage.pushbase.PushConstants;
import com.moengage.pushbase.model.ActionButton;
import com.moengage.pushbase.model.NotificationPayload;
import com.moengage.pushbase.model.NotificationText;
import com.moengage.pushbase.model.action.Action;
import com.moengage.pushbase.model.action.CallAction;
import com.moengage.pushbase.model.action.CopyAction;
import com.moengage.pushbase.model.action.CustomAction;
import com.moengage.pushbase.model.action.NavigationAction;
import com.moengage.pushbase.model.action.RemindLaterAction;
import com.moengage.pushbase.model.action.ShareAction;
import com.moengage.pushbase.model.action.SnoozeAction;
import com.moengage.pushbase.model.action.TrackAction;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 * Date: 24/12/19
 */
public class PayloadParser {

  private static final String TAG = PushConstants.MODULE_TAG + "PayloadParser";

  public NotificationPayload parsePayload(Bundle payload) {
    NotificationPayload notificationPayload = new NotificationPayload(payload);
    boolean hasTemplate = hasTemplate(payload);
    notificationPayload.channelId =
        payload.getString(PushConstants.PUSH_PAYLOAD_ATTR_NOTIFICATION_CHANNEL_ID,
            PushConstants.NOTIFICATION_FALLBACK_CHANNEL_ID);
    notificationPayload.notificationType = payload.getString(
        MoEHelperConstants.NOTIFICATION_TYPE);
    notificationPayload.text = getText(payload, hasTemplate);
    notificationPayload.campaignId = payload.getString(NOTIFICATION_CAMPAIGN_ID);
    notificationPayload.imageUrl = payload.getString(MoEConstants.PUSH_NOTIFICATION_IMAGE_URL);
    notificationPayload.inboxExpiry =
        Long.parseLong(payload.getString(PushConstants.MOE_NOTIFICATION_EXPIRY, String.valueOf(
            MoEUtils.currentSeconds() + PushConstants.DEFAULT_INBOX_TTL))) * 1000;
    notificationPayload.actionButtonList = actionButtonsFromJson(payload);
    notificationPayload.enableDebugLogs =
        Boolean.parseBoolean(payload.getString(PushConstants.ENABLE_DEBUG_LOGS, "false"));
    notificationPayload.sound = payload.getString(NOTIFICATION_TONE,
        SdkConfig.getConfig().pushConfig.tone);
    parseKVFeatures(notificationPayload);
    return notificationPayload;
  }

  private NotificationText getText(Bundle payload, boolean hasTemplate) {
    try {
      if (hasTemplate) {
        NotificationText richText = getRichText(payload);
        if (!MoEUtils.isEmptyString(richText.title) && !MoEUtils.isEmptyString(richText.message)) {
          return richText;
        }
      }
      return getDefaultText(payload);
    } catch (Exception e) {
      Logger.e(TAG + " getText() : ", e);
    }
    return getDefaultText(payload);
  }

  private NotificationText getRichText(Bundle payload) throws JSONException {
    String moeFeatures = payload.getString(PushConstants.PAYLOAD_ATTRIBUTE_MOE_FEATURES);
    JSONObject featuresJson = new JSONObject(moeFeatures);
    JSONObject richPush = featuresJson.getJSONObject(PushConstants.PAYLOAD_ATTRIBUTE_RICH_PUSH);
    return new NotificationText(richPush.optString(PushConstants.NOTIFICATION_TITLE, ""),
        richPush.optString(PushConstants.NOTIFICATION_MESSAGE, ""),
        richPush.optString(PushConstants.NOTIFICATION_SUMMARY, ""));
  }

  private NotificationText getDefaultText(Bundle payload) {
    return new NotificationText(payload.getString(MoEConstants.PUSH_NOTIFICATION_TITLE),
        payload.getString(MoEConstants.PUSH_NOTIFICATION_MESSAGE),
        payload.getString(MoEConstants.PUSH_NOTIFICATION_SUMMARY, ""));
  }

  private boolean hasTemplate(Bundle payloadBundle) {
    try {
      if (!payloadBundle.containsKey(PushConstants.PAYLOAD_ATTRIBUTE_MOE_FEATURES)) return false;
      String moeFeatures =
          payloadBundle.getString(PushConstants.PAYLOAD_ATTRIBUTE_MOE_FEATURES);
      if (MoEUtils.isEmptyString(moeFeatures)) return false;
      JSONObject featuresJson = new JSONObject(moeFeatures);
      return featuresJson.has(PushConstants.PAYLOAD_ATTRIBUTE_RICH_PUSH);
    } catch (JSONException e) {
      Logger.e(TAG + " hasTemplate() : ", e);
    }
    return false;
  }

  private void parseKVFeatures(NotificationPayload notificationPayload) {
    try {
      if (!notificationPayload.payload.containsKey(PushConstants.PAYLOAD_ATTRIBUTE_MOE_FEATURES)) {
        return;
      }
      String moeFeatures =
          notificationPayload.payload.getString(PushConstants.PAYLOAD_ATTRIBUTE_MOE_FEATURES);
      JSONObject featuresJson = new JSONObject(moeFeatures);
      notificationPayload.campaignTag =
          featuresJson.optString(MESSAGE_TAG, PushConstants.DEFAULT_NOTIFICATION_TAG);
      notificationPayload.shouldIgnoreInbox = featuresJson.optBoolean(IGNORE_INBOX, false);
      notificationPayload.pushToInbox = featuresJson.optBoolean(PUSH_TO_INBOX, false);
      notificationPayload.isRichPush = featuresJson.has(PushConstants.PAYLOAD_ATTRIBUTE_RICH_PUSH);
      JSONObject androidJson = featuresJson.optJSONObject(ANDROID);
      if (androidJson == null) return;
      notificationPayload.isPersistent = androidJson.optBoolean(IS_PERSISTENT, false);
      notificationPayload.shouldDismissOnClick =
          androidJson.optBoolean(DO_NOT_CANCEL, true);
      notificationPayload.autoDismissTime = androidJson.optLong(AUTO_DISMISS, -1);
      notificationPayload.shouldShowMultipleNotification =
          androidJson.has(SHOW_MULTIPLE_NOTIFICATIONS)
              ? androidJson.getBoolean(SHOW_MULTIPLE_NOTIFICATIONS)
              : SdkConfig.getConfig().pushConfig.shouldShowMultiplePushInDrawer;
      notificationPayload.largeIconUrl = androidJson.optString(NOTIFICATION_LARGE_ICON, "");
    } catch (Exception e) {
      Logger.e(TAG + " parseAndAddMoEngageFeatures() : ", e);
    }
  }

  @Nullable
  private List<ActionButton> actionButtonsFromJson(Bundle payload) {
    try {
      if (!payload.containsKey(NOTIFICATION_ACTION_BUTTONS)) return null;
      String actionButtonString = payload.getString(NOTIFICATION_ACTION_BUTTONS);
      JSONArray buttonArray = new JSONArray(actionButtonString);
      int buttonCount = Math.min(buttonArray.length(), 3);
      List<ActionButton> actionButtonList = new ArrayList<>(buttonCount);
      for (int index = 0; index < buttonCount; index++) {
        ActionButton actionButton = buttonFromJson(buttonArray.getJSONObject(index));
        if (actionButton != null) {
          actionButtonList.add(actionButton);
        }
      }
      return actionButtonList;
    } catch (Exception e) {
      Logger.e(TAG + " actionButtonsFromJson() : ", e);
    }
    return null;
  }

  @Nullable
  private ActionButton buttonFromJson(JSONObject jsonObject) {
    try {
      ActionButton actionButton = new ActionButton(
          jsonObject.getString(KEY_ACTION_TITLE),
          jsonObject.optString(KEY_ACTION_ICON, ""),
          jsonObject.getString(KEY_ACTION_ID),
          actionFromJson(jsonObject)
      );
      if (MoEUtils.isEmptyString(actionButton.title)) return null;
      return actionButton;
    } catch (Exception e) {
      Logger.e(TAG + " buttonFromJson() : ", e);
    }
    return null;
  }

  private Action actionFromJson(JSONObject actionJson) {
    try {
      String type = actionJson.getString(KEY_ACTION_TAG);
      if (!MAPPER.containsKey(type)) return null;
      String actionType = MAPPER.get(type);
      if (MoEUtils.isEmptyString(actionType)) return null;
      switch (actionType) {
        case PushConstants.ACTION_NAVIGATE:
          return navigationActionFromJson(actionJson);
        case PushConstants.ACTION_TRACK_ATTR:
          return trackActionFromJson(actionJson);
        case PushConstants.ACTION_SHARE:
          return shareActionFromJson(actionJson);
        case PushConstants.ACTION_CALL:
          return callActionFromJson(actionJson);
        case PushConstants.ACTION_COPY:
          return copyActionFromJson(actionJson);
        case PushConstants.ACTION_REMIND_ME_LATER:
          return remindLaterActionFromJson(actionJson);
        case PushConstants.ACTION_SNOOZE:
          return snoozeActionFromJson(actionJson);
        case PushConstants.ACTION_CUSTOM:
          return customActionFromJson(actionJson);
        default:
          Logger.e(TAG + " actionFromJson() : Not a supported action.");
      }
    } catch (Exception e) {
      Logger.e(TAG + " actionFromJson() : ", e);
    }
    return null;
  }

  private SnoozeAction snoozeActionFromJson(JSONObject actionJson) throws JSONException {
    return new SnoozeAction(MAPPER.get(actionJson.getString(KEY_ACTION_TAG)),
        Integer.parseInt(actionJson.getString(KEY_ACTION_VALUE).trim()));
  }

  private CallAction callActionFromJson(JSONObject actionJson) throws JSONException {
    return new CallAction(MAPPER.get(actionJson.getString(KEY_ACTION_TAG)),
        actionJson.getString(KEY_ACTION_VALUE));
  }

  private CopyAction copyActionFromJson(JSONObject actionJson) throws JSONException {
    return new CopyAction(MAPPER.get(actionJson.getString(KEY_ACTION_TAG)),
        actionJson.getString(KEY_ACTION_VALUE));
  }

  private ShareAction shareActionFromJson(JSONObject actionJson) throws JSONException {
    return new ShareAction(MAPPER.get(actionJson.getString(KEY_ACTION_TAG)),
        actionJson.getString(KEY_ACTION_CONTENT));
  }

  private NavigationAction navigationActionFromJson(JSONObject actionJson) throws JSONException {
    String navigationType = "";
    navigationType = getNavigationType(actionJson);
    if (navigationType == null) return null;
    if (MoEUtils.isEmptyString(navigationType)) return null;
    String url = "";
    switch (navigationType) {
      case PushConstants.NAVIGATION_TYPE_SCREEN_NAME:
        url = actionJson.getString(KEY_ACTION_SCREEN);
        break;
      case PushConstants.NAVIGATION_TYPE_RICH_LANDING:
        url = actionJson.getJSONObject(KEY_ACTION_EXTRAS)
            .getString(MoEHelperConstants.GCM_EXTRA_WEB_URL);
        break;
      case PushConstants.NAVIGATION_TYPE_DEEP_LINK:
        url = actionJson.getString(KEY_ACTION_URI);
        break;
      default:
        Logger.e(TAG + " navigationActionFromJson() : Not a valid navigation type");
    }
    if (MoEUtils.isEmptyString(url)) return null;
    return new NavigationAction(MAPPER.get(actionJson.getString(KEY_ACTION_TAG)),
        navigationType, url,
        actionJson.has(KEY_ACTION_EXTRAS) ?
            MoEUtils.jsonToBundle(actionJson.getJSONObject(KEY_ACTION_EXTRAS)) : null);
  }

  @Nullable private String getNavigationType(JSONObject actionJson) throws JSONException {
    String navigationType;
    if (actionJson.has(KEY_ACTION_URI)) {
      navigationType = PushConstants.NAVIGATION_TYPE_DEEP_LINK;
    } else if (actionJson.has(KEY_ACTION_SCREEN)) {
      if (!actionJson.has(KEY_ACTION_EXTRAS)) {
        navigationType = PushConstants.NAVIGATION_TYPE_SCREEN_NAME;
      } else {
        JSONObject extras = actionJson.getJSONObject(KEY_ACTION_EXTRAS);
        if (extras.length() == 1 && extras.has(MoEHelperConstants.GCM_EXTRA_WEB_URL)) {
          navigationType = PushConstants.NAVIGATION_TYPE_RICH_LANDING;
        } else {
          navigationType = PushConstants.NAVIGATION_TYPE_SCREEN_NAME;
        }
      }
    } else {
      return null;
    }
    return navigationType;
  }

  private RemindLaterAction remindLaterActionFromJson(JSONObject actionJson) throws JSONException {
    return new RemindLaterAction(MAPPER.get(actionJson.getString(KEY_ACTION_TAG)),
        actionJson.optInt(KEY_ACTION_VALUE_TODAY, -1),
        actionJson.optInt(KEY_ACTION_VALUE_TOMORROW, -1));
  }

  private TrackAction trackActionFromJson(JSONObject actionJson) throws JSONException {
    String actionTag = actionJson.getString(KEY_ACTION_TAG);
    String actionType = MAPPER.get(actionTag);
    if (MoEUtils.isEmptyString(actionType)) return null;
    String trackType = "";
    if (ActionMapperConstants.ACTION_TRACK_ATTR.equals(actionTag)) {
      trackType = PushConstants.TRACK_TYPE_EVENT;
    } else if (ActionMapperConstants.ACTION_SET_ATTRIBUTE.equals(actionTag)) {
      trackType = PushConstants.TRACK_TYPE_USER_ATTRIBUTE;
    }
    if (MoEUtils.isEmptyString(trackType)) return null;
    switch (trackType) {
      case PushConstants.TRACK_TYPE_EVENT:
        return new TrackAction(actionType, trackType,
            actionJson.getString(KEY_ACTION_VALUE_OF), actionJson.getString(KEY_ACTION_TRACK));
      case PushConstants.TRACK_TYPE_USER_ATTRIBUTE:
        return new TrackAction(actionType, trackType,
            actionJson.getString(KEY_ACTION_VALUE), actionJson.getString(KEY_ACTION_SET));
      default:
        return null;
    }
  }

  private CustomAction customActionFromJson(JSONObject actionJson) throws JSONException {
    return new CustomAction(MAPPER.get(actionJson.getString(KEY_ACTION_TAG)),
        actionJson.getString(KEY_ACTION_CUSTOM_ACTION));
  }

  private static final Map<String, String> MAPPER;

  static {
    Map<String, String> mapper = new LinkedHashMap<>();
    mapper.put(ActionMapperConstants.ACTION_NAVIGATE, PushConstants.ACTION_NAVIGATE);
    mapper.put(ActionMapperConstants.ACTION_TRACK_ATTR, PushConstants.ACTION_TRACK_ATTR);
    mapper.put(ActionMapperConstants.ACTION_SHARE, PushConstants.ACTION_SHARE);
    mapper.put(ActionMapperConstants.ACTION_CALL, PushConstants.ACTION_CALL);
    mapper.put(ActionMapperConstants.ACTION_COPY, PushConstants.ACTION_COPY);
    mapper.put(ActionMapperConstants.ACTION_SET_ATTRIBUTE, PushConstants.ACTION_TRACK_ATTR);
    mapper.put(PushActionMapperConstants.ACTION_REMIND_EXACT, PushConstants.ACTION_SNOOZE);
    mapper.put(PushActionMapperConstants.ACTION_REMIND_INEXACT,
        PushConstants.ACTION_REMIND_ME_LATER);
    mapper.put(PushActionMapperConstants.ACTION_CUSTOM, PushConstants.ACTION_CUSTOM);
    MAPPER = Collections.unmodifiableMap(mapper);
  }

  private static final String NOTIFICATION_ACTION_BUTTONS = "gcm_actions";
  private static final String NOTIFICATION_CAMPAIGN_ID = "gcm_campaign_id";
  private static final String NOTIFICATION_TONE = "gcm_tone";

  private static final String AUTO_DISMISS = "autoDismiss";
  private static final String IS_PERSISTENT = "isPersistent";
  private static final String PUSH_TO_INBOX = "pushToInbox";
  private static final String IGNORE_INBOX = "ignoreInbox";
  private static final String MESSAGE_TAG = "msgTag";
  private static final String SHOW_MULTIPLE_NOTIFICATIONS = "showMultipleNotification";
  private static final String NOTIFICATION_LARGE_ICON = "largeIcon";
  private static final String DO_NOT_CANCEL = "dismissOnClick";
  private static final String ANDROID = "android";
  // action button
  private static final String KEY_ACTION_TAG = "action_tag";
  private static final String KEY_ACTION_TITLE = "action_title";
  private static final String KEY_ACTION_ICON = "action_icon";
  private static final String KEY_ACTION_ID = "action_id";
  private static final String KEY_ACTION_VALUE = "value";
  private static final String KEY_ACTION_MESSAGE = "msg";
  private static final String KEY_ACTION_CONTENT = "content";
  private static final String KEY_ACTION_URI = "uri";
  private static final String KEY_ACTION_SCREEN = "screen";
  private static final String KEY_ACTION_EXTRAS = "extras";
  private static final String KEY_ACTION_VALUE_TODAY = "value_today";
  private static final String KEY_ACTION_VALUE_TOMORROW = "value_tomorrow";
  private static final String KEY_ACTION_VALUE_OF = "valueOf";
  private static final String KEY_ACTION_TRACK = "track";
  private static final String KEY_ACTION_SET = "set";
  private static final String KEY_ACTION_CUSTOM_ACTION = "custom_payload";
}
