/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.pushbase.push;

import android.app.Activity;
import android.app.Notification;
import android.app.NotificationManager;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Build.VERSION;
import android.os.Build.VERSION_CODES;
import android.os.Bundle;
import android.os.Parcelable;
import androidx.annotation.RestrictTo;
import androidx.annotation.RestrictTo.Scope;
import androidx.core.app.NotificationCompat;
import androidx.core.app.TaskStackBuilder;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.activities.MoEActivity;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEDispatcher;
import com.moengage.core.MoEUtils;
import com.moengage.core.Properties;
import com.moengage.core.SdkConfig;
import com.moengage.core.executor.TaskManager;
import com.moengage.core.internal.location.GeoManager;
import com.moengage.core.LogLevel;
import com.moengage.pushbase.MoEPushHelper;
import com.moengage.pushbase.PushConstants;
import com.moengage.pushbase.activities.PushTracker;
import com.moengage.pushbase.model.NotificationMetaData;
import com.moengage.pushbase.model.NotificationPayload;
import com.moengage.pushbase.model.action.NavigationAction;
import com.moengage.pushbase.push.task.LogNotificationClickTask;
import com.moengage.pushbase.repository.PayloadParser;
import com.moengage.pushbase.repository.PushDAO;
import com.moengage.pushbase.richnotification.RichNotificationManager;
import java.util.Map;
import org.json.JSONObject;

import static com.moe.pushlibrary.utils.MoEHelperConstants.EXTRA_MSG_RECEIVED_TIME;

/**
 * This class is a callback for the push message delivery. In order to modify the behavior
 * simply extend this class and override methods.<br>
 *
 * @author MoEngage (abhishek@moengage.com)
 * @version 1.0
 * @since 5.3
 */
public class PushMessageListener {

  private final Object lock = new Object();
  private static final String TAG = PushConstants.MODULE_TAG + "PushMessageListener";
  private boolean isOnCreateNotificationCalled = false;

  private NotificationPayload notificationPayload;
  private NotificationBuilder notificationBuilder;

  /**
   * Callback which is triggered when a push payload is received. This method is responsible for
   * creating the notification object and posting the notification.
   *
   * @param context instance of {@link Context}
   * @param payload push payload
   */
  public final void onMessageReceived(Context context, Bundle payload) {
    synchronized (lock) {
      try {
        Logger.v(TAG + " onMessageReceived() : Push Payload received will try to show "
            + "notification.");
        if (payload == null || context == null) return;
        if (ConfigurationProvider.getInstance(context).isPushNotificationOptedOut()) {
          Logger.w(TAG + " onMessageReceived() : push notification opted out "
              + "cannot show push");
          return;
        }
        MoEUtils.dumpIntentExtras(TAG, payload);
        if (!MoEPushHelper.getInstance().isFromMoEngagePlatform(payload)) {
          Logger.w(TAG + " onMessageReceived() : Non-MoEngage push received, passing callback.");
          onNonMoEngageMessageReceived(context, payload);
          return;
        }
        notificationPayload = new PayloadParser().parsePayload(payload);
        if (notificationPayload == null) {
          Logger.v(TAG + " onMessageReceived() : Payload parsing failed. Cannot show "
              + "notification.");
          return;
        }
        serverSyncIfRequired(context, payload);
        enableLogsIfRequired(context, notificationPayload);
        Logger.v(TAG + " onMessageReceived() : Payload: " + notificationPayload);
        if (isSilentNotification(notificationPayload)) {
          Logger.v(TAG + " onMessageReceived() : Silent push, returning.");
          return;
        }
        if (!isValidPayload(notificationPayload)) {
          Logger.v(TAG + " onMessageReceived() : Not a valid push payload. ignoring payload");
          return;
        }
        ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
        //duplicate campaign check
        if (PushDAO.getInstance().doesCampaignExists(context, notificationPayload.campaignId)
            && !MoEngageNotificationUtils.isReNotification(payload)) {
          Logger.v(TAG + " onMessageReceived() : Received notification is already shown, will be"
              + " ignored. Campaign id - " + notificationPayload.campaignId);
          return;
        }
        if (!isSilentNotification(notificationPayload)
            && !MoEngageNotificationUtils.isReNotification(payload)) {
          onNotificationReceived(context, payload);
        }

        if (!isNotificationRequired(context, payload)
            && !MoEngageNotificationUtils.isReNotification(payload)) {
          Logger.v(TAG + " onMessageReceived() : Notification not required. Discarding message.");
          if (!isSilentNotification(notificationPayload)) {
            Logger.v(TAG + " onMessageReceived() : Notification not required");
            onNotificationNotRequired(context, payload);
          }
          return;
        }
        if (!hasMetaDataForShowingPush(SdkConfig.getConfig())){
          Logger.w(TAG + " onMessageReceived() : Not enough meta data for showing push "
              + "notification. Check if the SDK is initialised correctly.");
          return;
        }
        //set the last campaign id
        PushDAO.getInstance().saveCampaignId(context, notificationPayload.campaignId);
        //set campaign received time
        payload.putLong(EXTRA_MSG_RECEIVED_TIME, MoEUtils.currentMillis());
        // push to inbox only
        if (notificationPayload.pushToInbox && !MoEngageNotificationUtils.isReNotification(
            payload)) {
          Logger.v(TAG + " onMessageReceived() : Campaign need not be shown in notification "
              + "drawer. Will be saved in inbox.");
          logCampaignImpression(context, payload);
          MoEngageNotificationUtils.addNotificationToInboxIfRequired(context, payload);
          return;
        }

        Logger.i(TAG + " onMessageReceived() Will try to show notification");
        //get the redirection intent
        Intent finalIntent = getRedirectIntent(context);
        finalIntent.putExtras(payload);

        //get the boilerplate notification
        int notificationId =
            payload.getInt(PushConstants.MOE_NOTIFICATION_ID, -1);
        if (notificationId == -1) {
          notificationId = getNotificationId(provider,
              notificationPayload.shouldShowMultipleNotification);
        }
        // notification id to extras
        finalIntent.putExtra(PushConstants.MOE_NOTIFICATION_ID, notificationId);
        notificationBuilder = new NotificationBuilder(context, notificationPayload,
            notificationId, finalIntent);
        NotificationCompat.Builder notificationCompatBuilder;
        if (MoEngageNotificationUtils.isReNotification(payload)) {
          notificationCompatBuilder = onCreateNotificationInternal(context, payload, provider);
        } else {
          notificationCompatBuilder = onCreateNotification(context, payload, provider);
        }
        //sets the auto dismiss if it is required by the notification
        notificationBuilder.addAutoDismissIfAny();
        //required for logging the notification cleared event
        notificationBuilder.addClickAndClearCallbacks(notificationCompatBuilder);
        //build the notification
        Notification notification = notificationCompatBuilder.build();
        //customize it
        customizeNotification(notification, context, payload);
        //app has to call isNotificationRequired by all means if this class is being
        // overridden
        if (!isNotificationRequiredCalled) {
          throw new IllegalStateException(
              "super.isNotificationRequired(context, extras) not called.");
        }
        //post the created notification
        NotificationManager notificationManager =
            (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
        if (!(notificationPayload.isRichPush && MoEngageNotificationUtils.isReNotification(
            payload))) {
          notificationManager.notify(notificationId, notification);
        }

        if (!isSilentNotification(notificationPayload)
            && !MoEUtils.isEmptyString(notificationPayload.campaignId)
            && !MoEngageNotificationUtils.isReNotification(payload)) {
          MoEngageNotificationUtils.addNotificationToInboxIfRequired(context, payload);
          logCampaignImpression(context, payload);
          onPostNotificationReceived(context, payload);
        }
        //resetting the state so that this can be reused
        isNotificationRequiredCalled = false;
        if (!isOnCreateNotificationCalled) {
          Logger.v(TAG + " onMessageReceived() : onCreateNotification is not called. Client has "
              + "overridden and customised the display will not add rich content.");
          return;
        }
        boolean isUpdateRequired = false;
        if (notificationPayload.isRichPush && RichNotificationManager.getInstance()
            .hasRichNotificationModule() && RichNotificationManager.getInstance()
            .isTemplateSupported(notificationPayload)) {
          Logger.v(TAG + " onMessageReceived() : Will try to build rich notification.");
          // build and show rich push
          isUpdateRequired = RichNotificationManager.getInstance()
              .buildTemplate(context,
                  new NotificationMetaData(notificationPayload, notificationCompatBuilder,
                      finalIntent, notificationId));
          if (isUpdateRequired && !MoEngageNotificationUtils.isReNotification(
              notificationPayload.payload)) {
            Properties properties = new Properties();
            properties.addAttribute(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID,
                notificationPayload.campaignId);
            MoEngageNotificationUtils.addAttributesToProperties(notificationPayload.payload,
                properties);
            properties.setNonInteractive();
            MoEHelper.getInstance(context).trackEvent(MoEConstants.NOTIFICATION_SHOWN, properties);
          }
        } else if (!MoEUtils.isEmptyString(notificationPayload.imageUrl)) {
          // build and show image notification.
          notificationCompatBuilder =
              notificationBuilder.buildImageNotification(notificationCompatBuilder);
          isUpdateRequired = true;
        } else {
          notificationPayload = null;
          return;
        }
        if (!isUpdateRequired) {
          Logger.v(TAG + " onMessageReceived() : Re-posting not required.");
          return;
        }
        // making notification persistent
        if (notificationPayload.isPersistent) {
          notificationCompatBuilder.setOngoing(true);
        }
        notificationCompatBuilder.setChannelId(PushConstants.NOTIFICATION_RICH_CHANNEL_ID);
        notification = notificationCompatBuilder.build();
        if (notificationManager != null) {
          notificationManager.notify(notificationId, notification);
        }
      } catch (Exception e) {
        Logger.e(TAG + " onMessageReceived() ", e);
      }
    }
  }

  private void enableLogsIfRequired(Context context, NotificationPayload notificationPayload) {
    ConfigurationProvider.getInstance(context)
        .setDebugLogStatus(notificationPayload.enableDebugLogs);
    if (notificationPayload.enableDebugLogs) {
      SdkConfig.getConfig().logConfig.level = LogLevel.VERBOSE;
      SdkConfig.getConfig().logConfig.isEnabledForSignedBuild = true;
    }
  }

  /**
   * Callback triggered after the notification is shown.
   *
   * @param context Application context.
   * @param payload Push Payload
   */
  protected void onPostNotificationReceived(final Context context, final Bundle payload) {
    Logger.v(TAG + " onPostNotificationReceived() : Callback after notification is shown.");
  }

  /**
   * To get a callback for push not sent via MoEngage Platform.
   *
   * @param context instance of {@link Context}
   * @param payload Push Payload
   */
  public void onNonMoEngageMessageReceived(final Context context, final Bundle payload) {
    Logger.v(TAG + " onNonMoEngageMessageReceived() : Callback for non-moengage push received.");
  }

  /**
   * Callback triggered whenever notification is not shown by the SDK.
   *
   * @param context instance of {@link Context}
   * @param payload Push Payload
   */
  public void onNotificationNotRequired(final Context context, final Bundle payload) {
    Logger.v(TAG + " onNotificationNotRequired() : Callback for discarded notification.");
  }

  /**
   * Callback triggered when notification is cleared from the device's notification drawer.
   *
   * @param context instance of {@link Context}
   * @param payload Push Payload
   * @since 9.4.04
   */
  public void onNotificationCleared(final Context context, final Bundle payload) {
    Logger.v(TAG + " onNotificationCleared() : Callback for notification cleared.");
  }

  /**
   * Callback triggered whenever notification is received by the application.
   *
   * @param context instance of {@link Context}
   * @param payload Push Payload
   * @since 9.4.04
   */
  public void onNotificationReceived(final Context context, final Bundle payload) {
    Logger.v(TAG + " onNotificationReceived() : Callback for notification received.");
  }

  /**
   * Callback triggered for custom action.
   *
   * @param context Application Context
   * @param payload Action payload
   */
  public void handleCustomAction(final Context context, final String payload) {
    Logger.v(
        TAG + " handleCustomAction() : Custom Action on notification click. Payload: " + payload);
  }

  /**
   * Return value of this method decides whether a notification should be shown or not.
   * In case you are overriding this method ensure you call <code>super(context, payload)</code>,
   * notifications will not be shown if super is not called.
   * <b>Note:</b> If the <code>super(context, payload)</code> returns <code>false</code> ensure that
   * the method always returns false.
   *
   * @param context instance of {@link Context}
   * @param payload push payload
   * @return true if the notification should be shown to the user else false.
   */
  public boolean isNotificationRequired(final Context context, final Bundle payload) {
    isNotificationRequiredCalled = true;
    Logger.v(TAG + " isNotificationRequired() : ");
    //uninstall check - using silent push mechanism
    return !isSilentNotification(notificationPayload);
  }

  /**
   * Build the notification using {@link NotificationCompat.Builder}.<br>
   * Override this method to customise the {@link NotificationCompat.Builder}. If you want to just
   * add a feature to the notification do not forget to call super()
   *
   * @param context instance of {@link Context}
   * @param payload push payload
   * @param provider Instance of {@link ConfigurationProvider} of MoEngage SDK
   */
  public NotificationCompat.Builder onCreateNotification(final Context context,
      final Bundle payload,
      final ConfigurationProvider provider) {
    return onCreateNotificationInternal(context, payload, provider);
  }

  private NotificationCompat.Builder onCreateNotificationInternal(final Context context,
      final Bundle payload,
      final ConfigurationProvider provider) {
    isOnCreateNotificationCalled = true;
    return notificationBuilder.buildTextNotification();
  }

  /**
   * A boolean which is set when {@link #isNotificationRequired(Context, Bundle)} is called
   */
  private boolean isNotificationRequiredCalled = false;

  /**
   * Returns the redirectIntent which will be started from the pending intent of the
   * notification<br>
   * <b>Note:<b/> Overriding this method and returning a custom intent will affect click tracking.
   *
   * @param context instance of {@link Context}
   */
  @RestrictTo(Scope.LIBRARY)
  public Intent getRedirectIntent(final Context context) {
    Intent pushTrackerIntent = new Intent(context, PushTracker.class);
    pushTrackerIntent.setAction("" + System.currentTimeMillis());
    pushTrackerIntent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
    return pushTrackerIntent;
  }

  /**
   * Logs impression for the notification if it was shown
   *
   * @param context instance of {@link Context}
   * @param payload the {@link Bundle} which was passed with the GCM payload
   */
  final public void logCampaignImpression(final Context context, final Bundle payload) {
    //intentionally done this way. Do not change
    MoEngageNotificationUtils.logNotificationImpression(context, payload);
  }

  /**
   * @param configProvider instance of {@link ConfigurationProvider} of MoEngage SDK.
   * @param update true is the notification display type is set to multiple else false.
   * @return identifier for {@link Notification} which is being posted.
   */
  public final int getNotificationId(final ConfigurationProvider configProvider,
      final boolean update) {
    if (!update) {
      return configProvider.getNotificationId();
    } else {
      int id = configProvider.getNotificationId();
      configProvider.updateNotificationId(++id);
      return id;
    }
  }

  /**
   * Callback to customise the notification object if required. SDK uses this method to
   * add/customise a few settings like vibration, led color etc. Make sure you call super before
   * adding your own customisation.
   *
   * @param notification instance of {@link Notification} which will be posted.
   * @param context instance of {@link Context}.
   * @param payload push payload.
   */
  public void customizeNotification(final Notification notification, final Context context,
      final Bundle payload) {
  }

  /**
   * This method is used to redirect the user to a specific screen or URL on click of a
   * notification.<br>
   * {@link PushConstants#IS_DEFAULT_ACTION} is a key in the payload which can be used to
   * identify whether the user clicked on the default content(or Default Action on MoEngage
   * Platform).<br>
   * If true the user has clicked on the default action and the key-value pair entered on the
   * MoEngage platform would be present at the root level of the {@link Bundle}
   * If false the user has clicked on the Action Button in the notification or clicked on a push
   * template. In this case the associated can be fetched from {@link PushConstants#NAV_ACTION}.
   * The value of this key is a {@link Parcelable}. You get the action payload using the below code.
   * <pre>
   *   <code>
   *     NavigationAction action = payload.getParcelable(PushConstants.NAV_ACTION);
   *   </code>
   * </pre>
   * Refer to {@link NavigationAction} to know more about the action payload.
   * @param activity instance of {@link Activity}
   * @param payload push payload
   */
  public void onHandleRedirection(Activity activity, Bundle payload) {
    Logger.v(TAG + ": onHandleRedirection() Will try to redirect user.");
    Intent launcherIntent = MoEHelperUtils.getLauncherActivityIntent(activity);
    Intent redirectIntent;
    try {
      boolean isDefaultAction = payload.getBoolean(PushConstants.IS_DEFAULT_ACTION, true);
      if (!isDefaultAction) {
        handleActionButtonNavigation(activity, payload);
        return;
      }
      // default action
      Logger.v(TAG + " onHandleRedirection() : Processing default notification action click.");
      String notificationType = payload.getString(MoEHelperConstants.NOTIFICATION_TYPE);
      if (MoEUtils.isEmptyString(notificationType)) {
        activity.startActivity(launcherIntent);
        return;
      }
      if (MoEHelperConstants.GCM_EXTRA_WEB_NOTIFICATION.equals(notificationType)) {
        Logger.v(TAG + " onHandleRedirection() : Will try to open url.");
        Uri finalLink = getDeepLink(payload);
        //remove web notification tags, type
        payload.remove(MoEHelperConstants.GCM_EXTRA_WEB_NOTIFICATION);
        payload.remove(MoEHelperConstants.NOTIFICATION_TYPE);

        if (finalLink == null) return;
        Logger.v(
            "PushMessagingListener:onHandleRedirection : Final URI : " + finalLink.toString());
        redirectIntent = new Intent(Intent.ACTION_VIEW, finalLink);
        redirectIntent.putExtras(payload);
        redirectIntent.addFlags(getIntentFlags(payload));
        activity.startActivity(redirectIntent);
        return;
      }
      String activityName = payload.getString(MoEHelperConstants.GCM_EXTRA_ACTIVITY_NAME);
      if (!MoEUtils.isEmptyString(activityName)) {
        redirectIntent = new Intent(activity, Class.forName(activityName));
      } else {
        redirectIntent = launcherIntent;
      }
      payload.putBoolean(MoEHelperConstants.EXTRA_IS_FROM_BACKGROUND,
          !MoEHelper.isAppInForeground());
      redirectIntent.putExtras(payload);
      redirectIntent.addFlags(getIntentFlags(payload));
      if (!SdkConfig.getConfig().pushConfig.isBackStackBuilderOptedOut) {
        Logger.v(TAG + " onHandleRedirection() : synthesizing back-stack");
        TaskStackBuilder builder = TaskStackBuilder.create(activity);
        builder.addNextIntentWithParentStack(redirectIntent).startActivities();
      } else {
        activity.startActivity(redirectIntent);
      }
      return;
    } catch (Exception e) {
      Logger.e(TAG + " onHandleRedirection() ", e);
    }
    activity.startActivity(launcherIntent);
  }

  private Uri getDeepLink(Bundle pushPayload) {
    if (pushPayload.containsKey(PushConstants.MOE_WEB_URL)) {
      return Uri.parse(pushPayload.getString(MoEHelperConstants.MOE_WEB_URL));
    }
    Uri link = Uri.parse(pushPayload.getString(MoEHelperConstants.GCM_EXTRA_WEB_URL));
    //append extras to URI
    Uri.Builder builder = link.buildUpon();
    //add extras to URI
    MoEngageNotificationUtils.addPayloadToUri(pushPayload, builder);
    return builder.build();
  }

  private void handleActionButtonNavigation(Activity activity, Bundle payload) {
    try {
      Context context = activity.getApplicationContext();
      NavigationAction action = payload.getParcelable(PushConstants.NAV_ACTION);
      if (action == null || MoEUtils.isEmptyString(action.navigationType) || MoEUtils.isEmptyString(
          action.navigationUrl)) {
        Logger.v(TAG + " handleActionButtonNavigation() : Not a valid action.");
        return;
      }
      Logger.v(TAG + " handleActionButtonNavigation() : Action: " + action);
      Intent intent = null;
      switch (action.navigationType) {
        case PushConstants.NAVIGATION_TYPE_SCREEN_NAME:
          intent = new Intent(context, Class.forName(action.navigationUrl));
          break;
        case PushConstants.NAVIGATION_TYPE_DEEP_LINK:
          intent = new Intent(Intent.ACTION_VIEW, Uri.parse(action.navigationUrl));
          break;
        case PushConstants.NAVIGATION_TYPE_RICH_LANDING:
          intent = new Intent(activity, MoEActivity.class);
          intent.putExtra(MoEHelperConstants.GCM_EXTRA_WEB_URL, action.navigationUrl);
          break;
        default:
          Logger.v(TAG + " handleActionButtonNavigation() : Not a valid navigation type.");
      }
      if (intent == null) return;
      if (action.keyValuePair != null && !action.keyValuePair.isEmpty()) {
        intent.putExtras(action.keyValuePair);
      }
      if (!SdkConfig.getConfig().pushConfig.isBackStackBuilderOptedOut) {
        Logger.v(TAG + " handleActionButtonNavigation() : synthesizing back-stack.");
        TaskStackBuilder builder = TaskStackBuilder.create(context);
        builder.addNextIntentWithParentStack(intent).startActivities();
      } else {
        activity.startActivity(intent);
      }
    } catch (Exception e) {
      Logger.e(TAG + " handleActionButtonNavigation() : ", e);
    }
  }

  /**
   * Tracks notification clicks on MoEngage Platform.<br>
   * <b>Note:<b/> If super() is not called when overriding this method click tracking will be
   * affected.
   *
   * @param context instance of Application {@link Context}.
   * @param intent Notification click intent.
   */
  @RestrictTo(Scope.LIBRARY)
  public final void logNotificationClicked(final Context context, final Intent intent) {
    //intentionally done this way. Do not change
    Logger.v(TAG + " logNotificationClicked() : Will track notification click.");
    TaskManager.getInstance()
        .addTaskToQueueBeginning(
            new LogNotificationClickTask(context, intent));
  }

  /**
   * Forcefully dismisses a {@link Notification} posted to the status bar.
   * This is required when action buttons are clicked since autoCancel
   * does not work with action button
   *
   * @param context An instance of the Application {@link Context}
   * @param payload The {@link Bundle} which is passed with the intent
   */
  public final void dismissNotificationAfterClick(final Context context, final Bundle payload) {
    Logger.v(TAG + " dismissNotificationAfterClick() : Will attempt to dismiss notification.");
    int notificationId = payload.getInt(PushConstants.MOE_NOTIFICATION_ID, -1);
    NotificationPayload notificationPayload = new PayloadParser().parsePayload(payload);
    Logger.v(TAG
        + " dismissNotificationAfterClick() : Should dismiss notification: "
        + notificationPayload.shouldDismissOnClick
        + " Notification id: "
        + notificationId);
    if (notificationPayload.isPersistent) return;
    if (notificationId != -1 && notificationPayload.shouldDismissOnClick) {
      NotificationManager manager =
          (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
      manager.cancel(notificationId);
    }
  }

  /**
   * Get the intent flags for the redirection intent
   *
   * @param payload The {@link Bundle} which is passed with the intent
   * @return {@link Intent} flags to be set to the redirection activity
   */
  public int getIntentFlags(final Bundle payload) {
    return Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_SINGLE_TOP;
  }

  /**
   * Callback which is triggered when a push payload is received. This method is responsible for
   * creating the notification object and posting the notification.
   *
   * @param context instance of application {@link Context}
   * @param payloadMap push payload.
   */
  public final void onMessageReceived(Context context, Map<String, String> payloadMap) {
    Bundle bundle = MoEUtils.convertMapToBundle(payloadMap);
    if (bundle != null) onMessageReceived(context, bundle);
  }

  private boolean shouldMakeGeoFenceCall(Bundle bundle) {
    return bundle.getString(PushConstants.PUSH_PAYLOAD_EXTRA, "false").equals("true");
  }

  private boolean isSilentNotification(NotificationPayload payload) {
    return PushConstants.NOTIFICATION_TYPE_SILENT.equals(payload.notificationType);
  }

  private void serverSyncIfRequired(Context context, Bundle pushPayload) {
    try {
      Logger.v(TAG + " serverSyncIfRequired() : Sync APIs if required.");
      if (shouldMakeGeoFenceCall(pushPayload)) {
        GeoManager.getInstance().updateFenceAndLocation(context);
      }
      if (!pushPayload.containsKey(PushConstants.PAYLOAD_EXTRA_SERVER_SYNC)) return;
      String syncRequestString = pushPayload.getString(PushConstants.PAYLOAD_EXTRA_SERVER_SYNC);
      JSONObject syncJson = new JSONObject(syncRequestString);
      if (syncJson.length() == 0 || !syncJson.has(PushConstants.PAYLOAD_EXTRA_TYPE)) return;
      String syncType = syncJson.getString(PushConstants.PAYLOAD_EXTRA_TYPE);
      Logger.v(TAG + " serverSyncIfRequired() : Request type: " + syncType);
      switch (syncType) {
        case PushConstants.PAYLOAD_EXTRA_SYNC_TYPE_CONFIG:
          MoEDispatcher.getInstance(context).syncConfigIfRequired();
          break;
        case PushConstants.PAYLOAD_EXTRA_SYNC_TYPE_REPORT_ADD:
          MoEDispatcher.getInstance(context).sendInteractionData();
          break;
      }
    } catch (Exception e) {
      Logger.e(TAG + " serverSyncIfRequired() : ", e);
    }
  }

  private boolean isValidPayload(NotificationPayload payload) {
    return !MoEUtils.isEmptyString(payload.campaignId)
        && !MoEUtils.isEmptyString(payload.text.title)
        && !MoEUtils.isEmptyString(payload.text.message);
  }

  private boolean hasMetaDataForShowingPush(SdkConfig config){
    if (VERSION_CODES.LOLLIPOP > VERSION.SDK_INT && config.pushConfig.largeIcon == -1) return false;
    return config.pushConfig.smallIcon != -1;
  }
}
