/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.pushbase.push;

import android.app.AlarmManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.net.Uri;
import android.os.Build;
import android.os.Build.VERSION;
import android.os.Build.VERSION_CODES;
import androidx.core.app.NotificationCompat;
import androidx.core.app.NotificationCompat.BigPictureStyle;
import androidx.core.app.NotificationCompat.BigTextStyle;
import androidx.core.app.NotificationCompat.Builder;
import androidx.core.text.HtmlCompat;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.core.SdkConfig;
import com.moengage.pushbase.MoEPushReceiver;
import com.moengage.pushbase.PushActionMapperConstants;
import com.moengage.pushbase.PushConstants;
import com.moengage.pushbase.model.ActionButton;
import com.moengage.pushbase.model.NotificationPayload;
import com.moengage.pushbase.model.action.Action;

/**
 * @author Umang Chamaria
 * Date: 27/12/19
 */
public class NotificationBuilder {

  private static final String TAG = PushConstants.MODULE_TAG + "NotificationBuilder";

  private NotificationPayload notificationPayload;
  private Context context;
  private int notificationId;
  private Intent actionIntent;

  NotificationBuilder(Context context, NotificationPayload notificationPayload,
      int notificationId, Intent actionIntent) {
    this.context = context;
    this.notificationPayload = notificationPayload;
    this.notificationId = notificationId;
    this.actionIntent = actionIntent;
  }

  /**
   * Builds a text only notification object.
   *
   * @return instance of {@link NotificationCompat.Builder}
   */
  NotificationCompat.Builder buildTextNotification() {
    setUpNotificationChannel();
    NotificationCompat.Builder builder = new Builder(context, notificationPayload.channelId);
    builder.setContentTitle(
        HtmlCompat.fromHtml(notificationPayload.text.title, HtmlCompat.FROM_HTML_MODE_COMPACT))
        .setContentText(
            HtmlCompat.fromHtml(notificationPayload.text.message,
                HtmlCompat.FROM_HTML_MODE_COMPACT));
    if (!MoEUtils.isEmptyString(notificationPayload.text.summary)) {
      builder.setSubText(
          HtmlCompat.fromHtml(notificationPayload.text.summary, HtmlCompat.FROM_HTML_MODE_COMPACT));
    }
    setNotificationSmallIcon(builder);
    setNotificationLargeIcon(builder);
    if (SdkConfig.getConfig().pushConfig.notificationColor != -1) {
      builder.setColor(
          context.getResources().getColor(SdkConfig.getConfig().pushConfig.notificationColor));
    }

    BigTextStyle bigTextStyle = new BigTextStyle()
        .setBigContentTitle(
            HtmlCompat.fromHtml(notificationPayload.text.title, HtmlCompat.FROM_HTML_MODE_COMPACT))
        .bigText(HtmlCompat.fromHtml(notificationPayload.text.message,
            HtmlCompat.FROM_HTML_MODE_COMPACT));
    if (!MoEUtils.isEmptyString(notificationPayload.text.summary)) {
      bigTextStyle.setSummaryText(HtmlCompat.fromHtml(notificationPayload.text.summary,
          HtmlCompat.FROM_HTML_MODE_COMPACT));
    }
    builder.setStyle(bigTextStyle);

    if (!MoEUtils.isEmptyString(notificationPayload.sound)) {
      Uri tone = Uri.parse(
          "android.resource://" + context.getPackageName() + "/raw/" + notificationPayload.sound);
      if (tone != null) {
        builder.setSound(tone);
      }
    }
    addActionButtonToNotification(builder);
    return builder;
  }

  public NotificationCompat.Builder buildImageNotification(NotificationCompat.Builder builder) {
    Bitmap bitmap = MoEngageNotificationUtils.scaleLandscapeBitmap(context,
        MoEHelperUtils.downloadImageBitmap(notificationPayload.imageUrl));
    if (bitmap == null) return builder;
    NotificationCompat.BigPictureStyle pictureStyle = new BigPictureStyle().bigPicture(bitmap);
    pictureStyle.setBigContentTitle(
        HtmlCompat.fromHtml(notificationPayload.text.title, HtmlCompat.FROM_HTML_MODE_COMPACT));
    if (VERSION.SDK_INT >= VERSION_CODES.N) {
      pictureStyle.setSummaryText(
          HtmlCompat.fromHtml(notificationPayload.text.message, HtmlCompat.FROM_HTML_MODE_COMPACT));
    } else {
      if (!MoEUtils.isEmptyString(notificationPayload.text.summary)) {
        pictureStyle.setSummaryText(HtmlCompat
            .fromHtml(notificationPayload.text.summary, HtmlCompat.FROM_HTML_MODE_COMPACT));
      } else {
        pictureStyle.setSummaryText(HtmlCompat.fromHtml(notificationPayload.text.message,
            HtmlCompat.FROM_HTML_MODE_COMPACT));
      }
    }
    builder.setStyle(pictureStyle)
        .setChannelId(PushConstants.NOTIFICATION_RICH_CHANNEL_ID);
    return builder;
  }

  private void setNotificationSmallIcon(NotificationCompat.Builder builder) {
    int resourceId = -1;
    if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) {
      resourceId = SdkConfig.getConfig().pushConfig.largeIcon;
    } else {
      resourceId = SdkConfig.getConfig().pushConfig.smallIcon;
    }
    if (resourceId != -1) {
      builder.setSmallIcon(resourceId);
    }
  }

  private void setNotificationLargeIcon(NotificationCompat.Builder builder) {
    if (VERSION.SDK_INT < VERSION_CODES.LOLLIPOP) return;
    if (SdkConfig.getConfig().pushConfig.isLargeIconOptedOut) return;
    Bitmap bitmap = null;
    if (!MoEUtils.isEmptyString(notificationPayload.largeIconUrl)) {
      bitmap = MoEHelperUtils.downloadImageBitmap(notificationPayload.largeIconUrl);
    } else {
      bitmap = BitmapFactory.decodeResource(context.getResources(),
          SdkConfig.getConfig().pushConfig.largeIcon, null);
    }
    if (bitmap != null) {
      builder.setLargeIcon(bitmap);
    }
  }

  private void setUpNotificationChannel() {
    if (MoEngageNotificationUtils.isReNotification(notificationPayload.payload)){
      notificationPayload.channelId = PushConstants.NOTIFICATION_RICH_CHANNEL_ID;
      return;
    }
    if (MoEngageNotificationUtils.isNotificationChannelExists(context,
        notificationPayload.channelId)) {
      return;
    }
    notificationPayload.channelId = PushConstants.NOTIFICATION_FALLBACK_CHANNEL_ID;
  }

  private void addActionButtonToNotification(NotificationCompat.Builder builder) {
    if (notificationPayload.actionButtonList == null
        || notificationPayload.actionButtonList.size() == 0) {
      return;
    }

    for (int index = 0; index < notificationPayload.actionButtonList.size(); index++) {
      ActionButton actionButton = notificationPayload.actionButtonList.get(index);
      if (actionButton.action == null) continue;
      Intent clickIntent;
      if (PushConstants.ACTION_REMIND_ME_LATER.equals(actionButton.action.actionType)) {
        clickIntent =
            MoEngageNotificationUtils.getIntentForSnooze(context, notificationPayload.payload,
                notificationId);
      } else {
        clickIntent = MoEngageNotificationUtils.getRedirectIntent(context,
            notificationPayload.payload, notificationId);
      }
      clickIntent.putExtra(PushActionMapperConstants.KEY_ACTION_ID, actionButton.actionId);
      Action[] actions = new Action[] { actionButton.action };
      clickIntent.putExtra(PushConstants.ACTION, actions);
      PendingIntent intent =
          PendingIntent.getActivity(context, notificationId + (1000 + index), clickIntent,
              PendingIntent.FLAG_UPDATE_CURRENT);
      NotificationCompat.Action notificationAction =
          new NotificationCompat.Action(getImageResourceId(actionButton.icon), actionButton.title,
              intent);
      builder.addAction(notificationAction);
    }
  }

  private int getImageResourceId(String iconName) {
    if (MoEUtils.isEmptyString(iconName)) return 0;
    try {
      int imageResource = context.getResources()
          .getIdentifier(iconName, "drawable", context.getPackageName());
      if (imageResource > 0) return imageResource;
      imageResource = android.R.drawable.class.getField(iconName).getInt(null);
      if (imageResource > 0) return imageResource;
    } catch (Exception e) {
      Logger.e(TAG + " getImageResourceId() : ", e);
    }
    return 0;
  }

  void addClickAndClearCallbacks(NotificationCompat.Builder builder) {
    // clear callback
    Intent finalIntent = new Intent(context, MoEPushWorker.class);
    finalIntent.putExtras(notificationPayload.payload);
    finalIntent.setAction(MoEPushWorker.NOTIFICATION_CLEARED);
    PendingIntent intent =
        PendingIntent.getService(context,
            notificationId | PushConstants.NOTIFICATION_CLEARED_REQUEST_ID,
            finalIntent,
            PendingIntent.FLAG_UPDATE_CURRENT);
    builder.setDeleteIntent(intent);
    // click callback
    PendingIntent contentIntent =
        PendingIntent.getActivity(context, notificationId, actionIntent,
            PendingIntent.FLAG_UPDATE_CURRENT);
    builder.setContentIntent(contentIntent);
  }

  void addAutoDismissIfAny() {
    if (notificationPayload.autoDismissTime == -1) return;
    Logger.v(TAG + " addAutoDismissIfAny() : Dismiss time: " + notificationPayload.autoDismissTime);
    Intent finalIntent = new Intent(context, MoEPushReceiver.class);
    finalIntent.putExtra(PushConstants.ACTION_NOTIFICATION_DISMISS, notificationId);
    finalIntent.setAction(PushConstants.ACTION_NOTIFICATION_DISMISS);
    PendingIntent intent = PendingIntent.getBroadcast(context, notificationId, finalIntent,
        PendingIntent.FLAG_UPDATE_CURRENT);
    AlarmManager alarmManager = (AlarmManager) context.getSystemService(Context.ALARM_SERVICE);
    alarmManager.set(AlarmManager.RTC_WAKEUP, notificationPayload.autoDismissTime * 1000, intent);
  }
}
