/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.pushbase.push;

import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.net.Uri;
import android.os.Build.VERSION;
import android.os.Build.VERSION_CODES;
import android.os.Bundle;
import android.util.DisplayMetrics;
import android.util.TypedValue;
import androidx.annotation.RestrictTo;
import androidx.annotation.RestrictTo.Scope;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEFileManager;
import com.moengage.core.MoEUtils;
import com.moengage.core.Properties;
import com.moengage.core.executor.Task;
import com.moengage.core.executor.TaskManager;
import com.moengage.pushbase.MoEPushHelper;
import com.moengage.pushbase.PushActionMapperConstants;
import com.moengage.pushbase.PushConstants;
import com.moengage.pushbase.activities.PushClickDialogTracker;
import com.moengage.pushbase.activities.PushTracker;
import com.moengage.pushbase.model.NotificationPayload;
import com.moengage.pushbase.model.TemplateTrackingMeta;
import com.moengage.pushbase.repository.PayloadParser;
import com.moengage.pushbase.repository.PushDAO;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import org.json.JSONObject;

import static com.moe.pushlibrary.utils.MoEHelperConstants.NOTIFICATION_RECEIVED_MOE;

/**
 * Utility class for Notifications sent from MoEngage platform.
 *
 * @author MoEngage (abhishek@moengage.com)
 * @version 1.0
 * @since 5.3
 */
public final class MoEngageNotificationUtils {

  private MoEngageNotificationUtils() {
    //constructor intentionally hidden
  }

  private static final String TAG = PushConstants.MODULE_TAG + "MoEngageNotificationUtils";

  /**
   * Use {@link MoEPushHelper#isFromMoEngagePlatform(Bundle)}
   */
  @Deprecated public static boolean isFromMoEngagePlatform(Bundle extras) {
    return MoEPushHelper.getInstance().isFromMoEngagePlatform(extras);
  }

  public static String convertBundleToJsonString(Bundle newBundle) {
    Set<String> keys = newBundle.keySet();
    JSONObject jsonObject = new JSONObject();
    for (String key : keys) {
      try {
        jsonObject.put(key, newBundle.get(key));
      } catch (Exception e) {
        Logger.e(TAG + " convertBundleToJsonString() : ", e);
      }
    }
    return jsonObject.toString();
  }

  public static void addPayloadToUri(Bundle extras, Uri.Builder uriBuilder) {
    try {
      if (null == extras || extras.isEmpty()) return;
      Set<String> keySet = extras.keySet();
      for (String key : keySet) {
        Object value = extras.get(key);
        if (value != null) {
          uriBuilder.appendQueryParameter(key, String.valueOf(value));
        }
      }
    } catch (Exception e) {
      Logger.e(TAG + " getMoEngageExtrasAsUriParam :", e);
    }
  }

  /**
   * Helper method to check if Notification has to be shown again
   *
   * @param extras Application context
   */
  public static boolean isReNotification(Bundle extras) {
    return extras.getBoolean(PushConstants.KEY_RE_NOTIFY, false);
  }

  /**
   * Use {@link MoEPushHelper#isFromMoEngagePlatform(Map)}
   */
  @Deprecated public static boolean isFromMoEngagePlatform(Map<String, String> extras) {
    return MoEPushHelper.getInstance().isFromMoEngagePlatform(extras);
  }

  public static void logNotificationImpression(final Context context, final Bundle extras) {
    try {
      if (!MoEPushHelper.getInstance().isFromMoEngagePlatform(extras)) return;
      String campaignId = extras.getString(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID, "");
      if (MoEUtils.isEmptyString(campaignId)) {
        Logger.v(TAG + " logNotificationImpression() : Either campaign id is not present or "
            + "empty string.");
        return;
      }
      Properties properties = new Properties();
      properties.setNonInteractive();
      if (campaignId.contains(PushConstants.REAL_TIME_TRIGGER_IDENTIFIER)) {
        campaignId = campaignId.substring(0, campaignId.indexOf(
            PushConstants.REAL_TIME_TRIGGER_IDENTIFIER));
        extras.putString(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID, campaignId);
      }
      properties.addAttribute(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID,
          extras.getString(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID));

      addAttributesToProperties(extras, properties);
      // track impression
      MoEHelper.getInstance(context).trackEvent(NOTIFICATION_RECEIVED_MOE, properties);
    } catch (Exception e) {
      Logger.e("PushMessageListener:trackNotification", e);
    }
  }

  public static void logNotificationClick(final Context context, final Intent intent) {
    try {
      Logger.v(TAG + " logNotificationClick() : Will log notification click.");
      if (null == intent) return;
      Bundle extras = intent.getExtras();
      if (extras == null) return;
      if (!MoEPushHelper.getInstance().isFromMoEngagePlatform(extras)) return;
      String campaignId = extras.getString(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID, "");
      if (MoEUtils.isEmptyString(campaignId)) {
        Logger.e(TAG + " logNotificationClick() : Either campaign id is not present or "
            + "empty string.");
        return;
      }
      Properties attributes = new Properties();
      if (campaignId.contains(PushConstants.REAL_TIME_TRIGGER_IDENTIFIER)) {
        campaignId = campaignId.substring(0, campaignId.indexOf(
            PushConstants.REAL_TIME_TRIGGER_IDENTIFIER));
        extras.putString(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID, campaignId);
      }
      attributes.addAttribute(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID, campaignId);
      if (extras.containsKey(PushActionMapperConstants.KEY_ACTION_ID)) {
        attributes.addAttribute("gcm_action_id", extras.getString(
            PushActionMapperConstants.KEY_ACTION_ID));
      }
      addAttributesToProperties(extras, attributes);
      // remove MoEngage specific keys.
      intent.removeExtra(PushConstants.TEMPLATE_META);
      intent.removeExtra(PushConstants.REAL_TIME_TRIGGER_OFFLINE_IDENTIFIER);
      intent.removeExtra(PushConstants.ATTRIBUTE_MOE_PUSH_SOURCE);
      intent.removeExtra(PushConstants.ATTRIBUTE_FROM_APP_OPEN);
      intent.removeExtra(MoEConstants.PUSH_CAMPAIGN_MOE_ATTRIBUTES);
      // track notification clicked
      MoEHelper.getInstance(context)
          .trackEvent(MoEHelperConstants.EVENT_NOTIFICATION_CLICKED, attributes);

      // update record for notification clicked for the same
      // to reflect in the inbox
      updateClickToInbox(context, extras);
    } catch (Exception e) {
      Logger.e(TAG + " logNotificationClicked", e);
    }
  }

  public static void addAttributesToProperties(Bundle payload, Properties properties) {
    try {
      if (payload == null) return;
      if (payload.containsKey(PushConstants.REAL_TIME_TRIGGER_OFFLINE_IDENTIFIER)) {
        properties.addAttribute(PushConstants.REAL_TIME_TRIGGER_OFFLINE_IDENTIFIER, true);
      }
      if (payload.containsKey(PushConstants.ATTRIBUTE_MOE_PUSH_SOURCE)) {
        properties.addAttribute(PushConstants.ATTRIBUTE_PUSH_SOURCE, payload.getString(
            PushConstants.ATTRIBUTE_MOE_PUSH_SOURCE));
      }
      if (payload.containsKey(PushConstants.ATTRIBUTE_FROM_APP_OPEN)) {
        properties.addAttribute(PushConstants.ATTRIBUTE_FROM_APP_OPEN,
            Boolean.parseBoolean(payload.getString(PushConstants.ATTRIBUTE_FROM_APP_OPEN)));
      }
      if (payload.containsKey(PushConstants.TEMPLATE_META)) {
        TemplateTrackingMeta meta = payload.getParcelable(PushConstants.TEMPLATE_META);
        if (meta != null) {
          if (!MoEUtils.isEmptyString(meta.templateName)) {
            properties.addAttribute(PushConstants.TEMPLATE_NAME, meta.templateName);
          }
          if (meta.cardId != -1) {
            properties.addAttribute(PushConstants.CARD_ID, meta.cardId);
          }
          if (meta.widgetId != -1) {
            properties.addAttribute(PushConstants.WIDGET_ID, meta.widgetId);
          }
        }
      }
      if (!payload.containsKey(MoEConstants.PUSH_CAMPAIGN_MOE_ATTRIBUTES)) return;
      JSONObject campaignAttributes =
          new JSONObject(payload.getString(MoEConstants.PUSH_CAMPAIGN_MOE_ATTRIBUTES));
      if (campaignAttributes.length() == 0) return;
      Iterator<String> keys = campaignAttributes.keys();
      while (keys.hasNext()) {
        String key = keys.next();
        String value = campaignAttributes.getString(key);
        properties.addAttribute(key, value);
      }
    } catch (Exception e) {
      Logger.e(TAG + " addAttributesToBuilder() : ", e);
    }
  }

  /**
   * Marks the notification as clicked.
   *
   * @param context Application Context
   * @param pushPayload Push payload
   */
  @RestrictTo(Scope.LIBRARY_GROUP)
  public static void updateClickToInbox(Context context, Bundle pushPayload) {
    PushDAO.getInstance().updateNotificationClick(context, pushPayload);
  }

  @RestrictTo(Scope.LIBRARY_GROUP)
  public static void addNotificationToInboxIfRequired(final Context context,
      final Bundle extras) {
    try {
      Logger.v(
          TAG + " addNotificationToInboxIfRequired() : Will try to add notification to inbox.");
      NotificationPayload payload = new PayloadParser().parsePayload(extras);
      if (payload.shouldIgnoreInbox) {
        Logger.v(TAG + " addNotificationToInboxIfRequired() : Will not add notification to the "
            + "inbox. Skip key added.");
        return;
      }
      PushDAO.getInstance().saveCampaign(context, payload);
    } catch (Exception e) {
      Logger.e(TAG + " addNotificationToInboxIfRequired() : Exception ", e);
    }
  }

  public static Intent getRedirectIntent(final Context context, Bundle payloadBundle,
      int notificationId) {
    Intent pushTrackerIntent = new Intent(context, PushTracker.class);
    pushTrackerIntent.setAction("" + System.currentTimeMillis());
    pushTrackerIntent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
    pushTrackerIntent.putExtras(payloadBundle)
        .putExtra(PushConstants.MOE_NOTIFICATION_ID, notificationId);
    return pushTrackerIntent;
  }

  public static Intent getIntentForSnooze(final Context context, Bundle payloadBundle,
      int notificationId) {
    Intent snoozeTrackerIntent = new Intent(context, PushClickDialogTracker.class);
    snoozeTrackerIntent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_CLEAR_TASK);
    snoozeTrackerIntent.putExtras(payloadBundle)
        .putExtra(PushConstants.MOE_NOTIFICATION_ID, notificationId);
    return snoozeTrackerIntent;
  }

  public static void deleteImagesIfRequired(final Context context, final Bundle payload) {
    try {
      Task task = new Task() {
        @Override public void execute() {
          try {
            PayloadParser payloadParser = new PayloadParser();
            NotificationPayload notificationPayload = payloadParser.parsePayload(payload);
            if (notificationPayload == null || MoEUtils.isEmptyString(
                notificationPayload.campaignId)) {
              return;
            }
            if (notificationPayload.isPersistent) return;
            MoEFileManager fileManager = new MoEFileManager(context);
            if (fileManager.doesDirectoryExists(notificationPayload.campaignId)) {
              fileManager.deleteFolder(notificationPayload.campaignId);
            }
          } catch (Exception e) {
            Logger.e(TAG + " execute() : ", e);
          }
        }
      };
      TaskManager.getInstance().execute(task);
    } catch (Exception e) {
      Logger.e(TAG + " deleteImagesIfRequired() : ", e);
    }
  }

  public static int transformToPx(Context context, int dp) {
    return (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, (float) dp,
        context.getResources().getDisplayMetrics());
  }

  public static void createMoEngageChannels(Context context) {
    createChannelIfRequired(context, PushConstants.NOTIFICATION_FALLBACK_CHANNEL_ID,
        PushConstants.NOTIFICATION_FALLBACK_CHANNEL_NAME, true, false);
    createChannelIfRequired(context, PushConstants.NOTIFICATION_RICH_CHANNEL_ID,
        PushConstants.NOTIFICATION_RICH_CHANNEL_NAME, false, true);
  }

  public static void createChannelIfRequired(Context context, String channelId, String channelName,
      boolean isVibrationEnabled, boolean shouldDisableSound) {
    if (VERSION.SDK_INT < VERSION_CODES.O) return;
    if (isNotificationChannelExists(context, channelId)) return;
    NotificationManager manager =
        (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
    NotificationChannel channel =
        new NotificationChannel(channelId, channelName, NotificationManager.IMPORTANCE_DEFAULT);
    channel.enableVibration(isVibrationEnabled);
    if (shouldDisableSound) {
      channel.setSound(null, null);
    }
    manager.createNotificationChannel(channel);
  }

  public static boolean isNotificationChannelExists(Context context, String channelId) {
    if (VERSION.SDK_INT < VERSION_CODES.O) return true;
    NotificationManager manager =
        (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
    return manager.getNotificationChannel(channelId) != null;
  }

  public static Bitmap scaleLandscapeBitmap(Context context, Bitmap imageBitmap) {
    if (imageBitmap == null) return null;

    if (imageBitmap.getWidth() > imageBitmap.getHeight()) {
      DisplayMetrics displayMetrics = context.getResources().getDisplayMetrics();
      int height = (imageBitmap.getHeight() * displayMetrics.widthPixels) / imageBitmap.getWidth();
      try {
        imageBitmap =
            Bitmap.createScaledBitmap(imageBitmap, displayMetrics.widthPixels, height, true);
      } catch (Exception e) {
        Logger.e(TAG + " scaleBitmapToDeviceSpecs", e);
      }
    }
    return imageBitmap;
  }
}