/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.pushbase.push;

import android.app.IntentService;
import android.app.NotificationManager;
import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.os.Parcelable;
import androidx.annotation.Keep;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEUtils;
import com.moengage.core.Properties;
import com.moengage.pushbase.MoEPushHelper;
import com.moengage.pushbase.PushConstants;
import com.moengage.pushbase.model.action.Action;
import com.moengage.pushbase.model.action.DismissAction;

/**
 * This class works as a Push Worker, it does task related to GCM token update
 * and notification display
 *
 * @author MoEngage (abhishek@moengage.com)
 * @version 1.0
 * @since 5.3
 */
@Keep
public class MoEPushWorker extends IntentService {

  public static final String NOTIFICATION_CLEARED = "ACTION_NOTIFICATION_CLEARED";
  public static final String NOTIFICATION_CLOSE_CLICKED = "ACTION_NOTIFICATION_CLOSE_CLICK";

  public MoEPushWorker() {
    super("RegistrationIntentService");
  }

  private static final String TAG = PushConstants.MODULE_TAG + "MoEPushWorker";

  /**
   * This method is invoked on the worker thread with a request to process.
   * Only one Intent is processed at a time, but the processing happens on a
   * worker thread that runs independently from other application logic.
   * So, if this code takes a long time, it will hold up other requests to
   * the same IntentService, but it will not hold up anything else.
   * When all requests have been handled, the IntentService stops itself,
   * so you should not call {@link #stopSelf}.
   *
   * @param intent The value passed to {@link
   * Context#startService(Intent)}.
   */
  @Override protected void onHandleIntent(Intent intent) {
    try {
      if (intent == null) return;
      Logger.v(TAG + " onHandleIntent() : Will process intent.");
      MoEUtils.dumpIntentExtras(TAG, intent.getExtras());
      String intentAction = intent.getAction();
      if (MoEUtils.isEmptyString(intentAction)) return;
      Logger.v(TAG + " onHandleIntent() : Intent Action: " + intentAction);
      switch (intentAction) {
        case NOTIFICATION_CLEARED:
          handleNotificationCleared(intent);
          break;
        case NOTIFICATION_CLOSE_CLICKED:
          dismissNotification(intent);
          break;
      }
    } catch (Exception e) {
      Logger.e( TAG + " onHandleIntent() : ", e);
    }
  }

  private void dismissNotification(Intent intent) {
    Bundle extras = intent.getExtras();
    if (extras == null) return;
    // delete images if required
    MoEngageNotificationUtils.deleteImagesIfRequired(getApplicationContext(), extras);

    if (!extras.containsKey(PushConstants.ACTION)) return;
    Parcelable[] parcelables = extras.getParcelableArray(PushConstants.ACTION);
    if (parcelables == null || parcelables.length == 0) return;
    Parcelable parcelable = parcelables[0];
    if (parcelable == null) return;
    Action action = (Action) parcelable;
    if (!(action instanceof DismissAction)) return;
    DismissAction dismissAction = (DismissAction) action;
    if (dismissAction.notificationId == -1) return;
    NotificationManager manager =
        (NotificationManager) getSystemService(Context.NOTIFICATION_SERVICE);
    manager.cancel(dismissAction.notificationId);
    if (!extras.containsKey(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID) && MoEUtils.isEmptyString(
        extras.getString(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID))) {
      Logger.e(TAG + " dismissNotification() : Campaign Id not present.");
      return;
    }
    Properties properties = new Properties();
    properties.addAttribute(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID,
        extras.getString(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID));
    MoEngageNotificationUtils.addAttributesToProperties(extras, properties);
    MoEHelper.getInstance(getApplicationContext())
        .trackEvent(MoEConstants.NOTIFICATION_DISMISSED, properties);
  }

  private void handleNotificationCleared(Intent intent) {
    Logger.v(TAG + " handleNotificationCleared() : ");
    Bundle extras = intent.getExtras();
    if (extras == null) return;
    //clear carousel images from internal storage
    MoEngageNotificationUtils.deleteImagesIfRequired(getApplicationContext(), extras);
    MoEPushHelper.getInstance()
        .getMessageListener()
        .onNotificationCleared(getApplicationContext(), extras);
  }

}
