/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.pushbase.push;

import android.app.Activity;
import android.app.AlarmManager;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import androidx.fragment.app.FragmentActivity;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.ActionMapperConstants;
import com.moengage.core.ActionManagerBase;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.core.Properties;
import com.moengage.pushbase.MoEPushHelper;
import com.moengage.pushbase.MoEPushReceiver;
import com.moengage.pushbase.PushConstants;
import com.moengage.pushbase.activities.PushClickDialogTracker;
import com.moengage.pushbase.fragments.LaterDialogFragment;
import com.moengage.pushbase.model.action.Action;
import com.moengage.pushbase.model.action.CallAction;
import com.moengage.pushbase.model.action.CopyAction;
import com.moengage.pushbase.model.action.CustomAction;
import com.moengage.pushbase.model.action.DismissAction;
import com.moengage.pushbase.model.action.NavigationAction;
import com.moengage.pushbase.model.action.RemindLaterAction;
import com.moengage.pushbase.model.action.ShareAction;
import com.moengage.pushbase.model.action.SnoozeAction;
import com.moengage.pushbase.model.action.TrackAction;
import java.util.Calendar;

/**
 * @author Umang Chamaria
 * Date: 04/03/20
 */
public class ActionManager extends ActionManagerBase {

  private static final String TAG = PushConstants.MODULE_TAG + "ActionManager";

  public void onActionPerformed(Activity activity, Action action) {
    try {
      if (action == null || activity == null || MoEUtils.isEmptyString(action.actionType)) return;
      Logger.v(TAG + " onActionPerformed() : Action: " + action);
      switch (action.actionType) {
        case PushConstants.ACTION_NAVIGATE:
          navigationAction(activity, action);
          break;
        case PushConstants.ACTION_CALL:
          callAction(activity, action);
          break;
        case PushConstants.ACTION_SHARE:
          shareAction(activity, action);
          break;
        case PushConstants.ACTION_TRACK_ATTR:
          trackAction(activity.getApplicationContext(), action);
          break;
        case PushConstants.ACTION_COPY:
          copyAction(activity.getApplicationContext(), action);
          break;
        case PushConstants.ACTION_CUSTOM:
          customAction(activity.getApplicationContext(), action);
          break;
        case PushConstants.ACTION_SNOOZE:
          snoozeAction(activity, action);
          break;
        case PushConstants.ACTION_REMIND_ME_LATER:
          remindLaterAction(activity, action);
          break;
        case PushConstants.ACTION_DISMISS:
          dismissAction(activity.getApplicationContext(), action);
          break;
        default:
          Logger.e(TAG + " onActionPerformed() : Did not find a suitable action.");
      }
    } catch (Exception e) {
      Logger.e(TAG + " onActionPerformed() : ", e);
    }
  }

  private void customAction(Context context, Action action) {
    if (!(action instanceof CustomAction)) {
      Logger.e(TAG + " customAction() : Not a valid custom action");
      return;
    }
    CustomAction customAction = (CustomAction) action;
    Logger.v(TAG + " customAction() : Action: " + customAction);
    MoEPushHelper.getInstance()
        .getMessageListener()
        .handleCustomAction(context, customAction.payload);
  }

  private void copyAction(Context context, Action action) {
    if (!(action instanceof CopyAction)) {
      Logger.v(TAG + " copyAction() : Not a valid copy action");
      return;
    }
    CopyAction copyAction = (CopyAction) action;
    Logger.v(TAG + " copyAction() : Action: " + copyAction);
    MoEHelperUtils.copyTextToClipboardAndShowToast(context, copyAction.textToCopy,
        "");
  }

  private void shareAction(Activity context, Action action) {
    if (!(action instanceof ShareAction)) {
      Logger.v(TAG + " shareAction() : Not a valid share action");
      return;
    }
    ShareAction shareAction = (ShareAction) action;
    Logger.v(TAG + " shareAction() : Action: " + shareAction);
    triggerShareIntent(context, shareAction.content);
  }

  private void callAction(Activity context, Action action) {
    if (!(action instanceof CallAction)) {
      Logger.v(TAG + " callAction() : Not a valid call action");
      return;
    }
    CallAction callAction = (CallAction) action;
    Logger.v(TAG + " callAction() : Action: " + callAction);
    if (MoEUtils.isEmptyString(callAction.number)) return;
    if (!isPhoneNumberValid(callAction.number)) {
      Logger.v(TAG + " callAction() : Not a valid phone number");
      return;
    }
    triggerCallIntent(context, callAction.number);
  }

  private void snoozeAction(Activity activity, Action action) {
    if (!(action instanceof SnoozeAction)) {
      Logger.v(TAG + " snoozeAction() : Not a valid snooze action");
      return;
    }
    Intent intent = activity.getIntent();
    if (intent == null) return;
    Bundle bundle = intent.getExtras();
    if (bundle == null) return;
    bundle.putBoolean(PushConstants.KEY_RE_NOTIFY, true);
    SnoozeAction snoozeAction = (SnoozeAction) action;
    Logger.v(TAG + " snoozeAction() : Action: " + snoozeAction);
    Context context = activity.getApplicationContext();
    if (snoozeAction.hoursAfterClick < 0 || snoozeAction.hoursAfterClick > 25) return;
    Intent laterIntent = new Intent(context, MoEPushReceiver.class);
    laterIntent.putExtras(MoEUtils.deepCopy(bundle));
    laterIntent.setAction(PushConstants.ACTION_SHOW_NOTIFICATION);
    PendingIntent alarmPendingIntent =
        PendingIntent.getBroadcast(activity.getApplicationContext(),
            (int) System.currentTimeMillis(),
            laterIntent,
            PendingIntent.FLAG_UPDATE_CURRENT);
    Calendar calendar = Calendar.getInstance();
    calendar.add(Calendar.HOUR_OF_DAY, snoozeAction.hoursAfterClick);
    AlarmManager alarmManager = (AlarmManager) context.getSystemService(Context.ALARM_SERVICE);
    if (alarmManager != null) {
      alarmManager.set(AlarmManager.RTC_WAKEUP, calendar.getTimeInMillis(), alarmPendingIntent);
    }
  }

  private void trackAction(Context context, Action action) {
    if (!(action instanceof TrackAction)) {
      Logger.v(TAG + " trackAction() : Not a valid track action");
      return;
    }
    TrackAction trackAction = (TrackAction) action;
    Logger.v(TAG + " trackAction() : Action: " + trackAction);
    if (MoEUtils.isEmptyString(trackAction.trackType) || MoEUtils.isEmptyString(trackAction.name)) {
      return;
    }
    switch (trackAction.trackType) {
      case PushConstants.TRACK_TYPE_EVENT:
        Properties properties = new Properties();
        if (!MoEUtils.isEmptyString(trackAction.value)) {
          properties.addAttribute(ActionMapperConstants.KEY_VALUE_OF, trackAction.value);
        }
        MoEHelper.getInstance(context).trackEvent(trackAction.name, properties);
        break;
      case PushConstants.TRACK_TYPE_USER_ATTRIBUTE:
        if (MoEUtils.isEmptyString(trackAction.value)) return;
        MoEHelper.getInstance(context).setUserAttribute(trackAction.name, trackAction.value);
        break;
      default:
        Logger.v(TAG + " trackAction() : Not a track type.");
    }
  }

  private void navigationAction(Activity activity, Action action) {
    if (!(action instanceof NavigationAction)) {
      Logger.v(TAG + " navigationAction() : Not a valid navigation action");
      return;
    }
    NavigationAction navigationAction = (NavigationAction) action;
    Logger.v(TAG + " navigationAction() : Navigation action:" + navigationAction);
    Bundle payload = new Bundle();
    payload.putParcelable(PushConstants.NAV_ACTION, navigationAction);
    payload.putBoolean(PushConstants.IS_DEFAULT_ACTION, false);
    MoEPushHelper.getInstance().getMessageListener().onHandleRedirection(activity, payload);
  }

  private void dismissAction(Context context, Action action) {
    if (!(action instanceof DismissAction)) {
      Logger.v(TAG + " dismissAction() : Not a valid dismiss action");
      return;
    }
    DismissAction dismissAction = (DismissAction) action;
    if (dismissAction.notificationId < -1) return;
    NotificationManager notificationManager =
        (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
    if (notificationManager != null) {
      notificationManager.cancel(dismissAction.notificationId);
    }
  }

  private void remindLaterAction(Activity activity, Action action) {
    if (!(action instanceof RemindLaterAction)) {
      Logger.v(TAG + " remindLaterAction() : Not a valid remind later action");
      return;
    }
    RemindLaterAction remindLaterAction = (RemindLaterAction) action;
    Logger.v(TAG + " remindLaterAction() : Remind later action: " + remindLaterAction);
    Intent intent = activity.getIntent();
    if (intent == null) return;
    Bundle bundle = intent.getExtras();
    if (bundle == null) return;
    bundle.putParcelable(PushConstants.ACTION_REMIND_ME_LATER, remindLaterAction);
    LaterDialogFragment dialogFragment = new LaterDialogFragment();
    dialogFragment.setItemSelected((PushClickDialogTracker) activity);
    dialogFragment.setArguments(bundle);
    dialogFragment.show(((FragmentActivity) activity).getSupportFragmentManager(), "laterDialog");
  }
}
