/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.pushbase.fragments;

import android.app.AlertDialog;
import android.app.Dialog;
import android.content.DialogInterface;
import android.os.Bundle;
import androidx.annotation.Keep;
import androidx.fragment.app.DialogFragment;
import android.view.ContextThemeWrapper;
import com.moengage.core.Logger;
import com.moengage.pushbase.PushConstants;
import com.moengage.pushbase.listener.OptionSelectionListener;
import com.moengage.pushbase.model.action.RemindLaterAction;
import java.util.Calendar;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * DialogFragment to show dialog with options for Remind Later
 *
 * @author MoEngage (prashant@moengage.com)
 * @version 1.0
 */
@Keep
public class LaterDialogFragment extends DialogFragment {

  private static final String TAG = PushConstants.MODULE_TAG + "LaterDialogFragment";
  private OptionSelectionListener itemSelected;

  public void setItemSelected(OptionSelectionListener itemSelected) {
    this.itemSelected = itemSelected;
  }

  @Override public Dialog onCreateDialog(Bundle savedInstanceState) {
    Logger.v(TAG + " onCreateDialog() : onCreateDialog called.");
    final Map<CharSequence, Long> optionsMap = new LinkedHashMap<>();
    Bundle extras = getArguments();
    if (extras != null && extras.containsKey(PushConstants.ACTION_REMIND_ME_LATER)) {
      try {
        RemindLaterAction remindLaterAction =
            extras.getParcelable(PushConstants.ACTION_REMIND_ME_LATER);
        if (remindLaterAction != null) {
          if (remindLaterAction.remindAfterHours != -1) {
            int triggerTime = remindLaterAction.remindAfterHours + Calendar.getInstance()
                .get(Calendar.HOUR_OF_DAY);
            int displayTime = triggerTime > 12 ? triggerTime - 12 : triggerTime;
            String todayOption = "Today (" + displayTime + ":" + Calendar.getInstance()
                .get(Calendar.MINUTE) + (triggerTime > 11 ? "PM" : "AM") + ")";
            final Calendar c = Calendar.getInstance();
            c.add(Calendar.MINUTE, (remindLaterAction.remindAfterHours * 60));
            optionsMap.put(todayOption, c.getTimeInMillis());
          }

          if (remindLaterAction.remindTomorrowAt != -1) {
            final Calendar c = Calendar.getInstance();
            c.add(Calendar.DAY_OF_MONTH, 1);
            c.set(Calendar.HOUR_OF_DAY, remindLaterAction.remindTomorrowAt);
            c.set(Calendar.MINUTE, 0);
            int displayTime = remindLaterAction.remindTomorrowAt;
            displayTime = displayTime > 12 ? displayTime - 12 : displayTime;
            String tomorrowOption =
                "Tomorrow ("
                    + displayTime
                    + (remindLaterAction.remindTomorrowAt > 11 ? "PM" : "AM")
                    + ")";
            optionsMap.put(tomorrowOption, c.getTimeInMillis());
          }
        }
      } catch (Exception e) {
        Logger.e(TAG + " onCreateDialog() : ", e);
      }
    } else {
      extras = new Bundle();
    }

    extras.putBoolean(PushConstants.KEY_RE_NOTIFY, true);

    String dateTimeOption = "Pick a date and time";
    optionsMap.put(dateTimeOption, -1L);

    CharSequence[] options = new CharSequence[optionsMap.size()];
    options = optionsMap.keySet().toArray(options);

    AlertDialog.Builder builder = new AlertDialog.Builder(
        new ContextThemeWrapper(getActivity(), android.R.style.Theme_Holo_Light_Dialog));

    final CharSequence[] finalOptions = options;
    builder.setTitle("Later").setItems(finalOptions, new DialogInterface.OnClickListener() {
      public void onClick(DialogInterface dialog, int which) {

        long triggerAtMillis = optionsMap.get(finalOptions[which]);
        itemSelected.onItemSelected(triggerAtMillis);
      }
    });

    return builder.create();
  }

  @Override public void onCancel(DialogInterface dialog) {
    super.onCancel(dialog);
    itemSelected.onDialogCancelled();
  }
}
