/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.pushbase.activities;

import android.content.Intent;
import android.os.Bundle;
import android.os.Parcelable;
import androidx.annotation.Keep;
import androidx.fragment.app.FragmentActivity;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEDispatcher;
import com.moengage.pushbase.MoEPushHelper;
import com.moengage.pushbase.PushConstants;
import com.moengage.pushbase.model.action.Action;
import com.moengage.pushbase.push.ActionManager;
import com.moengage.pushbase.push.MoEngageNotificationUtils;
import com.moengage.pushbase.push.PushMessageListener;

/**
 * @author MoEngage (abhishek@moengage.com)
 * @version 1.0
 * @since 5.0.7
 */
@Keep
public class PushTracker extends FragmentActivity {
  private static final String TAG = PushConstants.MODULE_TAG + "PushTracker";

  @Override protected void onCreate(Bundle savedInstanceState) {
    try {
      super.onCreate(savedInstanceState);
      Logger.v(TAG + " onCreate() : inside Push Tracker.");
      //tracking notification clicked
      Intent intent = getIntent();
      if (null == intent) return;

      Bundle extras = intent.getExtras();
      if (null == extras) {
        return;
      }
      boolean syncNow = false;
      if (intent.hasExtra(MoEHelperConstants.GCM_EXTRA_WEB_URL)) {
        syncNow = true;
      }

      PushMessageListener pushMessageListener = MoEPushHelper.getInstance().getMessageListener();
      //IN case the notification has actions and the action button is clicked it is not
      // dismissed, dismiss notification forcefully
      pushMessageListener.dismissNotificationAfterClick(getApplicationContext(), extras);
      //log click impressions
      pushMessageListener.logNotificationClicked(getApplicationContext(), getIntent());
      // delete images if required
      MoEngageNotificationUtils.deleteImagesIfRequired(getApplicationContext(), extras);
      //test in-app campaign
      if (extras.containsKey(MoEConstants.PUSH_EXTRA_INAPP_META) || extras.containsKey(
          MoEConstants.PUSH_EXTRA_INAPP_LEGACY_META)) {
        MoEDispatcher.getInstance(getApplicationContext()).showInAppFromPush(extras);
      }
      if (extras.containsKey(PushConstants.ACTION)) {
        processActionClick(extras);
      } else {
        extras.putBoolean(PushConstants.IS_DEFAULT_ACTION, true);
        //notification click callback required for handling redirection
        pushMessageListener.onHandleRedirection(this, extras);
      }
      if (syncNow) {
        MoEDispatcher.getInstance(getApplicationContext()).sendInteractionData();
      }
      finish();
      Logger.v(TAG + " onCreate() : Completed.");
    } catch (Exception e) {
      Logger.e(TAG + " onCreate() : ", e);
    }
  }

  private void processActionClick(Bundle extras) {
    try {
      Logger.v(TAG + " processActionClick() : Processing click for moe_action.");
      Parcelable[] parcelables = extras.getParcelableArray(PushConstants.ACTION);
      if (parcelables == null) return;
      ActionManager actionManager = new ActionManager();
      for (Parcelable parcelable : parcelables) {
        Action action = (Action) parcelable;
        actionManager.onActionPerformed(this, action);
      }
    } catch (Exception e) {
      Logger.e(TAG + " processActionClick() : ", e);
    }
  }
}
