/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.pushbase.activities;

import android.app.AlarmManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.os.Parcelable;
import androidx.annotation.Keep;
import androidx.fragment.app.FragmentActivity;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.pushbase.MoEPushHelper;
import com.moengage.pushbase.MoEPushReceiver;
import com.moengage.pushbase.PushConstants;
import com.moengage.pushbase.fragments.DatePickerFragment;
import com.moengage.pushbase.fragments.TimePickerFragment;
import com.moengage.pushbase.listener.DateSelectionListener;
import com.moengage.pushbase.listener.OptionSelectionListener;
import com.moengage.pushbase.listener.TimeSelectionListener;
import com.moengage.pushbase.model.action.Action;
import com.moengage.pushbase.push.ActionManager;
import com.moengage.pushbase.push.MoEngageNotificationUtils;
import com.moengage.pushbase.push.PushMessageListener;
import java.util.Calendar;

/**
 * @author MoEngage (prashant@moengage.com)
 * @version 1.0
 * @since 6.0.00
 * Activity to handle Later/Snooze action in Notification
 */
@Keep
public class PushClickDialogTracker extends FragmentActivity implements DateSelectionListener,
    TimeSelectionListener, OptionSelectionListener {

  private static final String TAG = PushConstants.MODULE_TAG + "PushClickDialogTracker";

  private int year, month, day;
  private Bundle extras;

  @Override
  protected void onCreate(Bundle savedInstanceState) {
    try {
      super.onCreate(savedInstanceState);
      Logger.v(TAG + " onCreate() :");
      Intent intent = getIntent();
      if (null == intent) return;

      extras = intent.getExtras();
      if (null == extras) {
        return;
      }
      //In case the notification has actions and the action button is clicked it is not dismissed
      PushMessageListener pushMessageListener = MoEPushHelper.getInstance().getMessageListener();
      //dismiss notification forcefully
      pushMessageListener.dismissNotificationAfterClick(getApplicationContext(), extras);
      //log click impressions
      pushMessageListener.logNotificationClicked(getApplicationContext(), getIntent());
      // delete images if required
      MoEngageNotificationUtils.deleteImagesIfRequired(getApplicationContext(), extras);
      extras.putBoolean(PushConstants.KEY_RE_NOTIFY, true);
      if (extras.containsKey(PushConstants.ACTION)) {
        processActionClick(extras);
      } else {
        finish();
      }
    } catch (Exception e) {
      Logger.e(TAG + " onCreate() : ", e);
    }
  }

  @Override public void onDateSelected(int year, int month, int day) {
    try {
      Logger.v(TAG + " onDateSelected() : Selected date: " + year + "-" + month + "-" + day);
      this.year = year;
      this.day = day;
      this.month = month;
      TimePickerFragment newFragment = new TimePickerFragment();
      newFragment.setTimeSelectedListener(this);
      newFragment.show(getSupportFragmentManager(), "timePicker");
    } catch (Exception e) {
      Logger.e(TAG + " onDateSelected() : ", e);
    }
  }

  @Override public void onTimeSelected(int hourOfDay, int minute) {
    try {
      Logger.v(TAG + " onTimeSelected() : Selected time: " + hourOfDay + ":" + minute);
      final Calendar c = Calendar.getInstance();
      c.set(year, month, day, hourOfDay, minute, 0);
      long triggerAtMillis = c.getTimeInMillis();
      Logger.v(TAG + " onTimeSelected() : Alarm trigger time: " + triggerAtMillis);
      Intent laterIntent = new Intent(this, MoEPushReceiver.class);
      laterIntent.putExtras(MoEUtils.deepCopy(extras));
      laterIntent.setAction(PushConstants.ACTION_SHOW_NOTIFICATION);
      final PendingIntent alarmPendingIntent =
          PendingIntent.getBroadcast(getApplicationContext(), (int) MoEUtils.currentMillis(),
              laterIntent,
              PendingIntent.FLAG_UPDATE_CURRENT);

      AlarmManager alarmManager = (AlarmManager) getSystemService(Context.ALARM_SERVICE);
      alarmManager.set(AlarmManager.RTC_WAKEUP, triggerAtMillis, alarmPendingIntent);
      finish();
    } catch (Exception e) {
      Logger.e(TAG + " onTimeSelected() : ", e);
    }
  }

  @Override public void onTimeDialogCancelled() {
    Logger.v(TAG + " onTimeDialogCancelled() : Dialog cancelled finishing activity.");
    finish();
  }

  @Override public void onDateDialogCancelled() {
    Logger.v(TAG + " onDateDialogCancelled() : Dialog cancelled finishing activity.");
    finish();
  }

  private void processActionClick(Bundle extras) {
    try {
      Logger.v(TAG + " processActionClick() : Processing click on action button.");
      Parcelable[] parcelables = extras.getParcelableArray(PushConstants.ACTION);
      if (parcelables == null) {
        finish();
        return;
      }
      ActionManager actionManager = new ActionManager();
      for (Parcelable parcelable : parcelables) {
        Action action = (Action) parcelable;
        actionManager.onActionPerformed(this, action);
      }
    } catch (Exception e) {
      Logger.e(TAG + " processActionClick() : ", e);
    }
  }

  @Override public void onItemSelected(long time) {
    try {
      Logger.v(TAG + " onItemSelected() : Item selected. Time: " + time);
      if (time != -1) {
        Intent finalIntent = new Intent(getApplicationContext(), MoEPushReceiver.class);
        finalIntent.putExtras(MoEUtils.deepCopy(extras));
        finalIntent.setAction(PushConstants.ACTION_SHOW_NOTIFICATION);
        PendingIntent intent = PendingIntent.getBroadcast(getApplicationContext(), 123, finalIntent,
            PendingIntent.FLAG_UPDATE_CURRENT);
        AlarmManager alarmManager =
            (AlarmManager) getApplication().getSystemService(Context.ALARM_SERVICE);
        alarmManager.set(AlarmManager.RTC_WAKEUP, time,
            intent);
        finish();
      } else {
        DatePickerFragment newFragment = new DatePickerFragment();
        newFragment.setArguments(extras);
        newFragment.setDateSelectedListener(this);
        newFragment.show(getSupportFragmentManager(), "datePicker");
      }
    } catch (Exception e) {
      Logger.e(TAG + " onItemSelected() : ", e);
    }
  }

  @Override public void onDialogCancelled() {
    finish();
  }
}
