/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.pushbase;

import android.net.Uri;
import android.os.Bundle;
import android.os.Parcelable;
import androidx.annotation.Nullable;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.core.RemoteConfig;
import com.moengage.core.internal.analytics.SourceProcessor;
import com.moengage.core.model.TrafficSource;
import com.moengage.pushbase.model.action.Action;
import com.moengage.pushbase.model.action.NavigationAction;

/**
 * @author Umang Chamaria
 * Date: 2020/08/04
 */
public class PushSourceProcessor {

  private static final String TAG = PushConstants.MODULE_TAG + "PushSourceProcessor";

  private Bundle payload;

  public PushSourceProcessor(Bundle pushPayload) {
    this.payload = pushPayload;
  }

  @Nullable public TrafficSource getTrafficSourceForCampaign() {
    try {
      Logger.v(TAG + " getTrafficSourceForCampaign() : Will to process traffic source for "
          + "campaign.");
      if (hasAction()) {
        Logger.v(TAG + " getTrafficSourceForCampaign() : Will processes source from moe_action.");
        return getTrafficSourceFromAction();
      }
      Logger.v(TAG + " getTrafficSourceForCampaign() : Will process source from default action.");
      SourceProcessor processor = new SourceProcessor();
      TrafficSource source = null;
      String deepLink = getDeepLinkFromPayload(payload);
      if (!MoEUtils.isEmptyString(deepLink)) {
        source = processor.getTrafficSourceFromUrl(Uri.parse(deepLink),
            RemoteConfig.getConfig().additionalSourceIdentifiers);
        if (source != null) return source;
      }
      return processor.getTrafficSourceFromExtras(payload,
          RemoteConfig.getConfig().additionalSourceIdentifiers);
    } catch (Exception e) {
      Logger.e(TAG + " getTrafficSourceForCampaign() : ", e);
    }
    return null;
  }

  private TrafficSource getTrafficSourceFromAction() {
    Parcelable[] parcelables = payload.getParcelableArray(PushConstants.ACTION);
    if (parcelables == null) return null;
    for (Parcelable parcelable : parcelables) {
      Action action = (Action) parcelable;
      if (action instanceof NavigationAction) {
        return getTrafficFromNavigation((NavigationAction) action);
      }
    }
    return null;
  }

  private TrafficSource getTrafficFromNavigation(NavigationAction action) {
    if (action.navigationType == null) return null;
    SourceProcessor sourceProcessor = new SourceProcessor();
    switch (action.navigationType) {
      case PushConstants.NAVIGATION_TYPE_DEEP_LINK:
      case PushConstants.NAVIGATION_TYPE_RICH_LANDING:
        return sourceProcessor.getTrafficSourceFromUrl(getUriFromAction(action),
            RemoteConfig.getConfig().additionalSourceIdentifiers);
      case PushConstants.NAVIGATION_TYPE_SCREEN_NAME:
        if (action.keyValuePair == null) return null;
        return sourceProcessor.getTrafficSourceFromExtras(action.keyValuePair,
            RemoteConfig.getConfig().additionalSourceIdentifiers);
    }
    return null;
  }

  private Uri getUriFromAction(NavigationAction action) {
    Uri uri = Uri.parse(action.navigationUrl);
    if (action.keyValuePair == null) return uri;
    Uri.Builder builder = uri.buildUpon();
    for (String key : action.keyValuePair.keySet()) {
      builder.appendQueryParameter(key, action.keyValuePair.getString(key));
    }
    return builder.build();
  }

  private boolean hasAction() {
    return payload.containsKey(PushConstants.ACTION);
  }

  @Nullable private String getDeepLinkFromPayload(Bundle payload) {
    if (payload.containsKey(MoEHelperConstants.MOE_WEB_URL)) {
      return payload.getString(MoEHelperConstants.MOE_WEB_URL);
    } else if (payload.containsKey(MoEHelperConstants.GCM_EXTRA_WEB_URL)) {
      return payload.getString(MoEHelperConstants.GCM_EXTRA_WEB_URL);
    } else {
      return null;
    }
  }
}
