/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.pushbase;

import android.content.Context;
import android.content.Intent;
import android.os.Bundle;
import android.os.Looper;
import androidx.annotation.NonNull;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEUtils;
import com.moengage.core.executor.TaskManager;
import com.moengage.pushbase.push.PushMessageListener;
import com.moengage.pushbase.push.task.LogNotificationImpressionTask;
import com.moengage.pushbase.push.task.MoEPushWorkerTask;
import java.util.Map;

/**
 * Helper class for push related hooks.
 * @author Umang Chamaria
 * Date: 07/01/20
 */
public class MoEPushHelper {

  private static final String TAG = PushConstants.MODULE_TAG + "MoEPushHelper";

  private static MoEPushHelper instance;
  private PushMessageListener messageListener = new PushMessageListener();

  private MoEPushHelper() {

  }

  public static MoEPushHelper getInstance() {
    if (instance == null) {
      synchronized (MoEPushHelper.class) {
        if (instance == null) instance = new MoEPushHelper();
      }
    }
    return instance;
  }

  /**
   * @deprecated Use MoEFireBaseHelper.getInstance().passPushPayload(context: Context, payload: Bundle)
   */
  @Deprecated
  @SuppressWarnings("ConstantConditions") public void handlePushPayload(@NonNull Context context,
      @NonNull Bundle pushPayload) {
    try {
      if (context == null || pushPayload == null) {
        Logger.w(TAG + " handlePushPayload() : Context or Push Payload is null");
        return;
      }

      if (Looper.myLooper() == Looper.getMainLooper()) {
        TaskManager.getInstance().addTaskToQueueBeginning(new MoEPushWorkerTask(context,
            MoEConstants.SHOW_NOTIFICATION, pushPayload));
      } else {
        getMessageListener().onMessageReceived(context, pushPayload);
      }
    } catch (Exception e) {
      Logger.e(TAG + " handlePushPayload() : ", e);
    }
  }

  /**
   * @deprecated Use MoEFireBaseHelper.getInstance().passPushPayload(context: Context, payload: Map<String, String>)
   */
  @SuppressWarnings("ConstantConditions") @Deprecated
  public void handlePushPayload(@NonNull Context context,
      @NonNull Map<String, String> pushPayload) {
    if (context == null || pushPayload == null) {
      Logger.w(TAG + " handlePushPayload() : Context or Push payload is null");
      return;
    }
    Bundle payloadBundle = MoEUtils.convertMapToBundle(pushPayload);
    if (payloadBundle == null) return;
    handlePushPayload(context, payloadBundle);
  }

  public PushMessageListener getMessageListener() {
    return messageListener;
  }

  /**
   * Set a Custom {@link PushMessageListener} to get callbacks for push notification or customise
   * the push notification.
   *
   * @param messageListener instance of a class extending {@link PushMessageListener}
   * @since 9.8.02
   */
  @SuppressWarnings("ConstantConditions") public void setMessageListener(
      @NonNull PushMessageListener messageListener) {
    if (messageListener != null) {
      this.messageListener = messageListener;
    }
  }

  /**
   * Checks whether the FCM message was being delivered from the MoEngage
   * Platform or not
   *
   * @param pushPayload Payload received from FCM
   * @return Returns true if the message was being delivered from MoEngage
   * Platform else false
   * @since 9.8.02
   */
  @SuppressWarnings("ConstantConditions") public boolean isFromMoEngagePlatform(
      @NonNull Map<String, String> pushPayload) {
    try {
      if (pushPayload == null) return false;
      if (!pushPayload.containsKey(MoEHelperConstants.GCM_EXTRA_CONFIRMATION_KEY)) return false;
      return MoEHelperConstants.GCM_EXTRA_CONFIRMATION_VALUE.equals(
          pushPayload.get(MoEHelperConstants.GCM_EXTRA_CONFIRMATION_KEY));
    } catch (Exception e) {
      Logger.e(TAG + " isFromMoEngagePlatform() : ", e);
    }
    return false;
  }

  /**
   * Checks whether the FCM message was being delivered from the MoEngage
   * Platform or not
   *
   * @param pushPayload Payload received from FCM
   * @return Returns true if the message was being delivered from MoEngage
   * Platform else false
   * @since 9.8.02
   */
  @SuppressWarnings("ConstantConditions") public boolean isFromMoEngagePlatform(
      @NonNull Bundle pushPayload) {
    try {
      if (pushPayload == null) return false;
      if (!pushPayload.containsKey(MoEHelperConstants.GCM_EXTRA_CONFIRMATION_KEY)) return false;
      return MoEHelperConstants.GCM_EXTRA_CONFIRMATION_VALUE.equals(
          pushPayload.getString(MoEHelperConstants.GCM_EXTRA_CONFIRMATION_KEY));
    } catch (Exception e) {
      Logger.e(TAG + " isFromMoEngagePlatform() : ", e);
    }
    return false;
  }

  void setMessageListenerInternal(Object messageListener) {
    if (messageListener != null && messageListener instanceof PushMessageListener) {
      this.messageListener = (PushMessageListener) messageListener;
    }
  }

  /**
   * Tracks notification impression.<br/>
   * <b>Note:</b> Generally this method is not required. Only call this API if you are not
   * passing the payload to MoEngage SDK. Call this method from onMessageReceived() of your
   * Firebase Receiver.
   *
   * @param context instance of {@link Context}
   * @param notificationPayload Push Campaign Payload
   * @since 9.8.02
   */
  @SuppressWarnings("ConstantConditions") public void logNotificationReceived(
      @NonNull Context context, @NonNull Map<String,
      String> notificationPayload) {
    Logger.v(TAG
        + " logNotificationReceived() : Will attempt to log notification received."
        + notificationPayload);
    if (context == null || notificationPayload == null) {
      Logger.w(TAG + " logNotificationReceived() : Either context is null or payload is null. "
          + "Cannot track notification impression.");
      return;
    }
    if (!isFromMoEngagePlatform(notificationPayload)) {
      Logger.w(TAG + " logNotificationReceived() : Notification payload not from MoEngage.");
      return;
    }
    TaskManager.getInstance()
        .addTaskToQueue(new LogNotificationImpressionTask(context, notificationPayload));
  }

  /**
   * Track notification click. Make sure the intent has the push notification payload received
   * Firebase as intent extras.<br/>
   * <b>Note:</b> Generally this method is not required. Only call this API if you are not
   * passing the payload to MoEngage SDK. Call this method from the onCreate() of your intent
   * which is fired on notification click.
   *
   * @param context instance of {@link Context}
   * @param intent instance of {@link Intent}
   * @since 9.8.02
   */
  @SuppressWarnings("ConstantConditions") public void logNotificationClick(@NonNull Context context,
      @NonNull Intent intent) {
    Logger.v(TAG + " logNotificationClick() : Will attempt to log notification clicked." + intent);
    if (context == null || intent == null) {
      Logger.w(TAG
          + " logNotificationClick() : Either instance Context or Intent is null. Cannot track "
          + "notification click.");
      return;
    }
    Bundle bundle = intent.getExtras();
    if (bundle == null) {
      Logger.w(TAG + " logNotificationClick() : Intent does not have any extras. Cannot log "
          + "notification click.");
      return;
    }
    if (!isFromMoEngagePlatform(bundle)) {
      Logger.w(TAG + " logNotificationClick() : Intent does not have push payload from MoEngage "
          + "Platform.");
      return;
    }
    getMessageListener().logNotificationClicked(context, intent);
  }

  /**
   * Checks if the given payload is a Silent Notification or not.
   *
   * @param pushPayload Push Payload received from FCM
   * @return true if it is a Silent Push from MoEngage platform, else false
   */
  @SuppressWarnings("ConstantConditions") public boolean isSilentPush(
      @NonNull Map<String, String> pushPayload) {
    if (pushPayload == null) return false;
    return pushPayload.containsKey(MoEHelperConstants.NOTIFICATION_TYPE)
        && PushConstants.NOTIFICATION_TYPE_SILENT.equals(pushPayload.get(
        MoEHelperConstants.NOTIFICATION_TYPE));
  }
}
