/* ************************************************************************
 *
 * MOENGAGE CONFIDENTIAL
 * __________________
 *
 *  [2014] - [2015] MoEngage Inc.
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of MoEngage Inc. The intellectual and technical concepts
 * contained herein are proprietary to MoEngage Incorporated
 * and its suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from MoEngage Incorporated.
 */
package com.moengage.pushbase.push;

import android.Manifest;
import android.app.Activity;
import android.app.Notification;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.ContentValues;
import android.content.Context;
import android.content.Intent;
import android.database.Cursor;
import android.graphics.Bitmap;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.support.v4.app.NotificationCompat;
import android.support.v4.app.NotificationManagerCompat;
import android.support.v4.app.TaskStackBuilder;
import android.text.TextUtils;
import android.util.TypedValue;
import android.view.View;
import android.widget.RemoteViews;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.providers.MoEDataContract;
import com.moe.pushlibrary.providers.MoEDataContract.CampaignListEntity;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.location.GeoManager;
import com.moengage.push.PushManager;
import com.moengage.pushbase.PushActionMapperConstants;
import com.moengage.pushbase.PushConstants;
import com.moengage.pushbase.R;
import com.moengage.pushbase.activities.PushTracker;
import java.util.Map;
import org.json.JSONArray;
import org.json.JSONObject;

import static com.moe.pushlibrary.utils.MoEHelperConstants.EXTRA_MSG_RECEIVED_TIME;

/**
 * This class is a callback for the push message delivery. In order to modify the behavior
 * simply extend and override methods.
 *
 * @author MoEngage (abhishek@moengage.com)
 * @version 1.0
 * @since 5.3
 */
public class PushMessageListener {

  private final Object lock = new Object();

  /**
   * Callback which is triggered when a GCM payload is received.
   *
   * @param context Instance of the Application Context
   * @param extras the {@link Bundle} which was passed with the GCM payload
   */
  public final void onMessagereceived(Context context, Bundle extras) {
    synchronized (lock) {
      try {
        if (ConfigurationProvider.getInstance(context).isPushNotificationOptedOut()){
          Logger.e(  "PushMessageListener onMessagereceived() : push notification opted out "
              + "cannot show push");
          return;
        }
        if (null == extras || null == context) return;
        MoEHelperUtils.dumpIntentExtras(extras);
        //sentViaMoEngagePlatform
        if (MoEngageNotificationUtils.isFromMoEngagePlatform(extras)) {
          logNotificationState(context);
          if (MoEngageNotificationUtils.hasNotificationExpired(extras)) {
            Logger.i("Campaign expired, will not be shown");
            logCampaignImpression(context, extras);
          } else {
            ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
            String campaignId = MoEngageNotificationUtils.getCampaignIdIfAny(extras);
            //duplicate check
            if (isDuplicateCampaign(context, campaignId, provider)
                && !MoEngageNotificationUtils.isReNotification(extras)) {
              return;
            }
            //set the last campaign id
            saveCampaignId(context, provider, campaignId);
            //set campaign received time
            extras.putLong(EXTRA_MSG_RECEIVED_TIME, System.currentTimeMillis());
            //check if it is required to show notification
            if (isNotificationRequired(context, extras)) {
              Logger.i("PushMessageListener: onMessageReceived Will try to show notification");
              //get the boilerplate notification
              MoEPushCallBacks.getInstance().onPushReceived(extras);
              enableLogsIfRequired(context, extras);
              NotificationCompat.Builder builder = onCreateNotification(context, extras, provider);
              //get the redirection intent
              Intent finalIntent = getRedirectIntent(context);
              extras.putAll(MoEngageNotificationUtils.getMoEngageExtras(extras));
              finalIntent.putExtras(extras);
              boolean updateExisting =
                  MoEngageNotificationUtils.getNotificationDisplayType(extras, context) == 1;
              int notificationId;
              if (MoEngageNotificationUtils.isCarouselNotification(extras)
                  && MoEngageNotificationUtils.getNotificationIdIfAny(extras) != -1) {
                notificationId = MoEngageNotificationUtils.getNotificationIdIfAny(extras);
              } else {
                notificationId = getNotificationId(context, provider, updateExisting);
              }
              //set the notification id
              MoEngageNotificationUtils.setNotificationId(finalIntent, notificationId);
              //sets the auto dismiss if it is required by the notification
              MoEngageNotificationUtils.setNotificationAutoDismissIfAny(context, notificationId,
                  extras);
              //required for logging the notification cleared event
              MoEngageNotificationUtils.setNotificationClearedCallback(context, builder,
                  notificationId, extras);
              //get the pending content intent
              PendingIntent contentIntent =
                  MoEngageNotificationUtils.getContentIntent(context, finalIntent, updateExisting,
                      notificationId);
              //add action buttons here
              MoEngageNotificationUtils.setActionButtonIfPresentAndSupported(context, extras,
                  builder, finalIntent, notificationId);
              //set the content intent
              builder.setContentIntent(contentIntent);
              //build the notification
              Notification notification = builder.build();
              //customize it
              customizeNotification(notification, context, extras);
              //create Custom Notification if required
              if (MoEngageNotificationUtils.isCarouselNotification(extras)) {
                createCarouselNotification(notification, context, finalIntent, extras,
                    notificationId);
              }
              //app has to call isNotificationRequired by all means if this class is being overridden
              if (!isNotificationRequiredCalled) {
                throw new IllegalStateException(
                    "super.isNotificationRequired(context, extras) not called.");
              }
              //post the created notification
              NotificationManager notificationManager =
                  (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
              notificationManager.notify(notificationId, notification);
            } else if (!MoEngageNotificationUtils.isSilentPush(extras)) {
              onNotificationNotRequired(context, extras);
            }
            //resetting the state so that this can be reused
            isNotificationRequiredCalled = false;
            if (!MoEngageNotificationUtils.isSilentPush(extras)
                && !TextUtils.isEmpty(MoEngageNotificationUtils.getCampaignIdIfAny(extras))
                && !MoEngageNotificationUtils.isReNotification(extras)) {
              if (!MoEngageNotificationUtils.isSkipNotificationCenter(extras)) {
                addToMoEngageInbox(context, extras);
              }
              logCampaignImpression(context, extras);
              onPostNotificationReceived(context, extras);
            }
          }
        } else {
          onNonMoEngageMessageReceived(context, extras);
        }
      } catch (Exception e) {
        Logger.f("PushMessageListener:onMessageReceived", e);
      }
    }
  }

  private void enableLogsIfRequired(Context context, Bundle extras) {
    if (extras != null && extras.containsKey(PushConstants.ENABLE_DEBUG_LOGS)){
      String logState = extras.getString(PushConstants.ENABLE_DEBUG_LOGS);
      if (!TextUtils.isEmpty(logState)){
        boolean logStateToSet = false;
        switch (logState){
          case "true":
            logStateToSet = true;
            break;
          case "false":
            logStateToSet = false;
            break;
        }
        ConfigurationProvider.getInstance(context).setDebugLogStatus(logStateToSet);
        Logger.setLogLevel(Logger.VERBOSE);
        Logger.enableDebugLog(context);
      }
    }
  }

  private void saveCampaignId(Context context, ConfigurationProvider provider, String campaignId) {
    try {
      provider.setLastPushCampaignId(campaignId);
      ContentValues values = new ContentValues();
      values.put(CampaignListEntity.CAMPAIGN_ID, campaignId);
      values.put(CampaignListEntity.CAMPAIGN_ID_TTL,
          System.currentTimeMillis() + provider.getCampaignIdTTL());
      context.getContentResolver().insert(CampaignListEntity.getContentUri(context), values);
    } catch (Exception e) {
      Logger.f("PushMessageListener saveCampaignId() ", e);
    }
  }

  protected void onPostNotificationReceived(final Context context, final Bundle extras) {
  }

  /**
   * This method should be used to handle non monegage gcm payloads.
   *
   * @param context Instance of the Application Context
   * @param extras the {@link Bundle} which was passed with the GCM payload
   */
  public void onNonMoEngageMessageReceived(final Context context, final Bundle extras) {
  }

  /**
   * @param context An instance of the application {@link Context}
   * @param extras The {@link Bundle} extras representing the GCM payload
   */
  public void onNotificationNotRequired(final Context context, final Bundle extras) {
  }

  /**
   * You should always call super if method is overridden or it will throw an
   * IllegalStateException.
   * Default Checks -
   *
   * @param context Instance of the Application Context
   * @param extras the {@link Bundle} which was passed with the GCM payload
   */
  public boolean isNotificationRequired(final Context context, final Bundle extras) {
    isNotificationRequiredCalled = true;
    //uninstall check - using silent push mechanism
    if (MoEngageNotificationUtils.isSilentPush(extras)) {
      //Silent push so Just call acknowledgement
      if ((MoEHelperUtils.hasPermission(context, Manifest.permission.ACCESS_FINE_LOCATION)
          || MoEHelperUtils.hasPermission(context, Manifest.permission.ACCESS_COARSE_LOCATION))
          && shouldMakeGeoFenceCall(extras)) {
        GeoManager.LocationHandler locationHandler = GeoManager.getInstance().getHandler(context);
        if (locationHandler != null) {
          locationHandler.updateFenceAndLocation(context);
        }
      }
      return false;
    } else if (!TextUtils.isEmpty(MoEngageNotificationUtils.getCampaignIdIfAny(extras))) {
      //show notification if it is from moengage platform and has campaign ID
      return !MoEngageNotificationUtils.isPushToInbox(extras);
    }
    return false;
  }

  /**
   * Creates the notification from GCM payload that was received.
   *
   * @param context Instance of the Application Context
   * @param extras the {@link Bundle} which was passed with the GCM payload
   * @param provider The {@link ConfigurationProvider} of MoEngage SDK
   */
  public NotificationCompat.Builder onCreateNotification(final Context context, final Bundle extras,
      final ConfigurationProvider provider) {

    String channelId =
        MoENotificationChannel.getInstance().getNotificationChannelId(context, extras);
    if (TextUtils.isEmpty(channelId) || !MoENotificationChannel.getInstance().isChannelExists
        (context, channelId)){
      Logger.e(
          "PushMessageListener: onCreateNotification() Did not find channel id setting using "
              + "Fallback channel");
      MoENotificationChannel.getInstance().createFallbackNotificationChanelIfRequired(context);
      channelId = PushConstants.NOTIFICATION_FALLBACK_CHANNEL_ID;
    }
    NotificationCompat.Builder builder = new NotificationCompat.Builder(context, channelId);

    builder.setAutoCancel(MoEngageNotificationUtils.isAutoCancelEnabled(extras));
    MoEngageNotificationUtils.setSubTextIfAny(extras, builder);
    MoEngageNotificationUtils.setContentIfPresent(extras, builder);
    MoEngageNotificationUtils.setTitleIfPresent(extras, builder);
    MoEngageNotificationUtils.setCategoryIfPresentAndSupported(extras, builder);
    MoEngageNotificationUtils.setSmallIcon(context, builder, provider);
    MoEngageNotificationUtils.setColorOrLargeIconIfPresentAndSupported(context, extras, builder,
        provider);
    MoEngageNotificationUtils.setNotificationPriorityIfPresentAndSupported(extras, builder);
    MoEngageNotificationUtils.setTickerTextIfPresent(extras, builder);
    //flexibility to customize sound
    setNotificationSound(context, extras, builder, provider);
    MoEngageNotificationUtils.setVisibilityIfPresentAndSupported(extras, builder);
    MoEngageNotificationUtils.setNotificationStyle(context, extras, builder);

    return builder;
  }

  /**
   * A boolean which is set when {@link #isNotificationRequired(Context, Bundle)} is called
   */
  private boolean isNotificationRequiredCalled = false;

  /**
   * Returns the redirectIntent which will be started from the pending intent of the notification
   *
   * @param context Instance of the Application Context
   */
  public Intent getRedirectIntent(final Context context) {
    Intent pushTrackerIntent = new Intent(context, PushTracker.class);
    pushTrackerIntent.setAction("" + System.currentTimeMillis());
    pushTrackerIntent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
    return pushTrackerIntent;
  }

  /**
   * Logs impression for the notification if it was shown
   *
   * @param context Instance of the Application Context
   * @param extras the {@link Bundle} which was passed with the GCM payload
   */
  public final void logCampaignImpression(final Context context, final Bundle extras) {
    //intentionally done this way. Do not change
    MoEngageNotificationUtils.logNotificationImpression(context, extras);
  }

  public final int getNotificationId(final Context context,
      final ConfigurationProvider configProvider, final boolean update) {
    return MoEngageNotificationUtils.getNotificationId(context, configProvider, update);
  }

  /**
   * @param context Instance of the Application Context
   * @param extras the {@link Bundle} which was passed with the GCM payload
   */
  public final void addToMoEngageInbox(final Context context, final Bundle extras) {
    Logger.v("PushMessagingListener: addToMoEngageInbox: ");
    String msgDetails = MoEngageNotificationUtils.convertBundletoJSONString(extras);
    ContentValues values = new ContentValues();
    if (null != msgDetails) {
      values.put(MoEDataContract.MessageEntity.MSG_DETAILS, msgDetails);
    }
    long gTime = extras.getLong(EXTRA_MSG_RECEIVED_TIME);
    values.put(MoEDataContract.MessageEntity.GTIME, gTime);
    values.put(MoEDataContract.MessageEntity.MSG_CLICKED, 0);
    values.put(MoEDataContract.MessageEntity.MSG_TTL,
        MoEngageNotificationUtils.getNotificationTTL(extras, gTime));
    String msgTags = MoEngageNotificationUtils.getMessageTagsIfAny(extras);
    if (TextUtils.isEmpty(msgTags)) {
      values.put(MoEDataContract.MessageEntity.MSG_TAG, "general");
    } else {
      values.put(MoEDataContract.MessageEntity.MSG_TAG, msgTags);
    }
    //adding to inbox
    Uri newRecord = context.getContentResolver()
        .insert(MoEDataContract.MessageEntity.getContentUri(context), values);
    if (null != newRecord) {
      Logger.v("PushMessagingListener: added new record with entry: " + newRecord);
    } else {
      Logger.f("PushMessagingListener: FAILED to add new record with entry: ");
    }
  }

  /***/
  public void setNotificationSound(final Context context, final Bundle extras,
      final NotificationCompat.Builder builder, final ConfigurationProvider provider) {
    MoEngageNotificationUtils.setSoundIfPresentAndSupported(context, extras, builder, provider);
  }

  /**
   * Customize the notification like, vibration, lights, etc.
   */
  public void customizeNotification(final Notification notification, final Context context,
      final Bundle extras) {

    boolean vibrate = true;
    if (Build.VERSION.SDK_INT <= Build.VERSION_CODES.JELLY_BEAN_MR2
        && !MoEHelperUtils.hasPermission(context, Manifest.permission.VIBRATE)) {
      vibrate = false;
    }

    if (vibrate && !MoEngageNotificationUtils.isVibrationDisabled(extras)) {
      notification.defaults |= Notification.DEFAULT_VIBRATE;
    }

    int ledColor = MoEngageNotificationUtils.getNotificationLedLightColor(extras);
    if (-1 != ledColor) {
      notification.flags |= Notification.FLAG_SHOW_LIGHTS;
      notification.ledARGB = ledColor;
    } else {
      notification.defaults = Notification.DEFAULT_LIGHTS;
    }
  }

  private boolean isDuplicateCampaign(final Context context, final String currentCampaignId,
      final ConfigurationProvider provider) {
    if (TextUtils.isEmpty(currentCampaignId)) return false;
    String lastCampaignId = provider.getLastPushCampaignId();
    if (currentCampaignId.equals(lastCampaignId)) {
      Logger.e("PushMessagingListener:isDuplicateCampaign-->Last campaign ID and current "
          + "campaign ID is same : " + currentCampaignId);
      return true;
    }
    return isMessageShown(context, currentCampaignId);
  }

  private boolean isNotTheIntendedRecipient(Context context, Bundle extras,
      ConfigurationProvider provider) {
    String recipientId = MoEngageNotificationUtils.getRecipientUserId(extras);
    if (TextUtils.isEmpty(recipientId)) {
      return false;
    } else {
      String thisUser = provider.getCurrentUserId();
      return !recipientId.equals(thisUser);
    }
  }

  public void onHandleRedirection(Activity activity, Bundle extras) {
    Logger.v("PushMessageListener: onHandleRedirection()");
    Intent launcherIntent = MoEHelperUtils.getLauncherActivityIntent(activity);
    try {
      String notificationType = extras.getString(MoEHelperConstants.GCM_EXTRA_NOTIFICATION_TYPE);
      Intent redirectIntent;
      if (!TextUtils.isEmpty(notificationType)
          && MoEHelperConstants.GCM_EXTRA_WEB_NOTIFICATION.equals(notificationType)) {
        Uri finalLink = null;
        if (PushManager.getInstance().isMoEngageExtrasOptedOut() || extras.containsKey
            (PushConstants.OPT_OUT_OF_MOE_EXTRAS)){
          if (extras.containsKey(PushConstants.MOE_WEB_URL)){
            String url = extras.getString(PushConstants.MOE_WEB_URL);
            if (!TextUtils.isEmpty(url)){
              finalLink = Uri.parse(url);
            }
          }
        }else {
          Uri link = Uri.parse(extras.getString(MoEHelperConstants.GCM_EXTRA_WEB_URL));
          //append extras to URI
          Uri.Builder builder = link.buildUpon();
          builder.appendQueryParameter(MoEHelperConstants.NAVIGATION_PROVIDER_KEY,
              MoEHelperConstants.NAVIGATION_PROVIDER_VALUE);
          builder.appendQueryParameter(MoEHelperConstants.NAVIGATION_SOURCE_KEY,
              MoEHelperConstants.NAVIGATION_SOURCE_NOTIFICATION);
          builder.appendQueryParameter(MoEHelperConstants.EXTRA_IS_FROM_BACKGROUND,
              "" + !MoEHelper.isAppInForeground());
          //remove web notification tags, campaign id, type
          extras.remove(MoEHelperConstants.GCM_EXTRA_WEB_NOTIFICATION);
          extras.remove(MoEHelperConstants.GCM_EXTRA_NOTIFICATION_TYPE);
          //add extras to URI
          MoEngageNotificationUtils.setMoEngageExtrastoUri(extras, builder);
          finalLink = builder.build();
        }
        //if( MoEngageNotificationUtils.hasCouponCode( ))
        //use the new URI
        if (!MoEPushCallBacks.getInstance().onPushNavigationAction(null, extras, finalLink)) {
          Logger.v("PushMessagingListener:onHandleRedirection-->Web notification");
          if (finalLink == null) return;
          Logger.v("PushMessagingListener:onHandleRedirection : Final URI : " + finalLink.toString());
          redirectIntent = new Intent(Intent.ACTION_VIEW, finalLink);
          redirectIntent.addFlags(getIntentFlags(extras));
          activity.startActivity(redirectIntent);
        }
        return;
      } else {
        String activityName = extras.getString(MoEHelperConstants.GCM_EXTRA_ACTIVITY_NAME);
        if (!TextUtils.isEmpty(activityName)) {
          redirectIntent = new Intent(activity, Class.forName(activityName));
        }else {
          redirectIntent = launcherIntent;
        }
        extras.putBoolean(MoEHelperConstants.EXTRA_IS_FROM_BACKGROUND,
            !MoEHelper.isAppInForeground());
        //denoting provider
        extras.putString(MoEHelperConstants.NAVIGATION_PROVIDER_KEY,
            MoEHelperConstants.NAVIGATION_PROVIDER_VALUE);
        extras.putString(MoEHelperConstants.NAVIGATION_SOURCE_KEY,
            MoEHelperConstants.NAVIGATION_SOURCE_NOTIFICATION);
        redirectIntent.putExtras(extras);
        redirectIntent.addFlags(getIntentFlags(extras));
        //Intent upIntent = NavUtils.getParentActivityIntent(activity, Class.forName(activityName));
        if (!MoEPushCallBacks.getInstance().onPushNavigationAction(activityName, extras, null)) {
          if (!PushManager.getInstance().isBackStackBuilderOptedOut(activity.getApplicationContext())) {
            TaskStackBuilder builder = TaskStackBuilder.create(activity);
            // This activity is NOT part of this app's task, so create a new task
            // when navigating up, with a synthesized back stack.
            builder.addNextIntentWithParentStack(redirectIntent).startActivities();
          } else {
            activity.startActivity(redirectIntent);
          }
        }
        return;
      }
    } catch (ClassNotFoundException e) {
      Logger.f("PushMessagingListener:onHandleRedirection--> Activity not found ", e);
    } catch (Exception e) {
      Logger.f("PushMessagingListener:onHandleRedirection--> generic exception ", e);
    }
    activity.startActivity(launcherIntent);
  }

  public final void logNotificationClicked(final Context context, final Intent intent) {
    //intentionally done this way. Do not change
    MoEngageNotificationUtils.logNotificationClick(context, intent);
  }

  /**
   * Forcefully dismisses a {@link Notification} posted to the status bar.
   * This is required when action buttons are clicked since autoCancel
   * does not work with action button
   *
   * @param context An instance of the Application {@link Context}
   * @param extras The {@link Bundle} which is passed with the intent
   */
  public final void dismissNotificationAfterClick(final Context context, final Bundle extras) {
    int notificationId = MoEngageNotificationUtils.getNotificationIdIfAny(extras);
    if (MoEngageNotificationUtils.isAutoCancelEnabled(extras) && -1 != notificationId) {
      NotificationManager manager =
          (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
      manager.cancel(notificationId);
    }
  }

  /**
   * Get the intent flags for the redirection intent
   *
   * @param extras The {@link Bundle} which is passed with the intent
   * @return {@link Intent} flags to be set to the redirection activity
   */
  public int getIntentFlags(final Bundle extras) {
    return Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_SINGLE_TOP;
  }

  /*
   * Create Carousel style notification
   * @param notification {@link Notification}
   * @param context An instance of the Application {@link Context}
   * @param finalIntent
   * @param extras The {@link Bundle} which is passed with the intent
   */
  private void createCarouselNotification(Notification notification, Context context,
      Intent finalIntent, Bundle extras, int notificationId) {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
      Logger.v("PushMessageListener : createCarouselNotification");
      try {
        JSONObject carouselObject = MoEngageNotificationUtils.getCarouselObject(extras);
        if (carouselObject == null) return;

        JSONArray imagesArray = MoEngageNotificationUtils.getImagesArray(carouselObject);
        if (imagesArray == null) return;

        RemoteViews rv = new RemoteViews(context.getPackageName(), R.layout.carousel_custom);
        String title = MoEngageNotificationUtils.getCarouselTitle(carouselObject, extras);
        String text = MoEngageNotificationUtils.getCarouselText(carouselObject, extras);
        String subText = MoEngageNotificationUtils.getCarouselSubText(carouselObject, extras);

        int smallNotificationIconResourceId =
            MoEngageNotificationUtils.getCarouselSmallNotificationIcon(context);
        int largeNotificationIconResourceId =
            MoEngageNotificationUtils.getCarouselLargeNotificationIcon(context);

        rv.setTextViewText(R.id.title, title);
        rv.setTextViewText(R.id.time, MoEngageNotificationUtils.getTime());
        rv.setTextViewText(R.id.text2, text);

        boolean hasThreeLines = false;
        if (subText != null) {
          hasThreeLines = true;
          rv.setViewVisibility(R.id.text, View.VISIBLE);
          rv.setTextViewText(R.id.text, subText);
          rv.setImageViewResource(R.id.profile_badge_line3, smallNotificationIconResourceId);
          rv.setViewVisibility(R.id.profile_badge_line3, View.VISIBLE);
        } else {
          rv.setImageViewResource(R.id.profile_badge_line2, smallNotificationIconResourceId);
          rv.setViewVisibility(R.id.profile_badge_line2, View.VISIBLE);
          rv.setViewVisibility(R.id.line3, View.GONE);
          float regularTextSize =
              context.getResources().getDimensionPixelSize(R.dimen.notification_text_size);
          rv.setTextViewTextSize(R.id.text2, TypedValue.COMPLEX_UNIT_PX, regularTextSize);
        }
        rv.setImageViewResource(R.id.icon, largeNotificationIconResourceId);
        rv.setViewPadding(R.id.line1, 0,
            MoEngageNotificationUtils.calculateTopPadding(context, hasThreeLines), 0, 0);

        if (carouselObject.has(PushActionMapperConstants.CAROUSEL_AUTOSTART)
            && carouselObject.getBoolean(PushActionMapperConstants.CAROUSEL_AUTOSTART)) {
          if (!createAnimatedCarouselNotification(context, extras, rv, finalIntent, imagesArray, carouselObject
              )) {
            return;
          }
        } else {
          int idx = extras.getInt(PushActionMapperConstants.IMG_INDEX, 0);
          Logger.v("PushMessageListener : createCarouselNotification idx" + idx);
          String campaignId = MoEngageNotificationUtils.getCampaignIdIfAny(extras);
          String fileName = campaignId + imagesArray.getJSONObject(idx)
              .getString(PushActionMapperConstants.IMG_ID);
          Bitmap bmp = MoEngageNotificationUtils.loadImageFromStorage(context, fileName);
          if (bmp == null) {
            MoEngageNotificationUtils.fetchAndSaveImages(context, imagesArray, campaignId);
            bmp = MoEngageNotificationUtils.loadImageFromStorage(context, fileName);
            if (bmp == null) return;
          }
          rv.setImageViewBitmap(R.id.big_picture, bmp);

          rv.setOnClickPendingIntent(R.id.big_picture,
              MoEngageNotificationUtils.getImagePendingIntent(context, finalIntent, idx,
                  imagesArray));

          //set navigation intent
          Intent navIntent = new Intent(context, MoEPushWorker.class);
          navIntent.setAction(MoEPushWorker.EXTRA_SERVICE_CAROUSEL);
          navIntent.putExtras(extras);
          navIntent.putExtra("MOE_NOTIFICATION_ID", notificationId);

          rv.setOnClickPendingIntent(R.id.next_btn,
              MoEngageNotificationUtils.getNavPendingIntent(context, navIntent,
                  PushActionMapperConstants.IMG_ACTION_NEXT, notificationId, idx));
          rv.setOnClickPendingIntent(R.id.prev_btn,
              MoEngageNotificationUtils.getNavPendingIntent(context, navIntent,
                  PushActionMapperConstants.IMG_ACTION_PREV, 2 * notificationId, idx));
        }

        MoEngageNotificationUtils.addCarouselActionButton(context, rv, extras, finalIntent,
            notificationId);
        if (MoEngageNotificationUtils.isReNotification(extras)) {
          MoEngageNotificationUtils.disableSoundAndVibration(notification);
        }

        notification.bigContentView = rv;
      } catch (Exception e) {
        Logger.f("PushMessageListener : createCarouselNotification : Exception occurred " + e);
      }
    }
  }

  /*
   * helper method to create Animated Carousel Notification
   * @param context Application context
   * @param extras received in push pay load
   * @param rv RemoteView inflated for Carousel Notification
   * @param imagesArray JSONArray of Carousel images received in push payload
   * @param finalIntent Intent for Images
   */
  private boolean createAnimatedCarouselNotification(Context context, Bundle extras, RemoteViews rv,
      Intent finalIntent, JSONArray imagesArray, JSONObject carouselObject) {
    try {
      String direction = PushConstants.CAROUSEL_ANIMATION_RIGHT_TO_LEFT;
      int imagesCount = imagesArray.length();
      if (imagesCount < 3) {
        Logger.v("PushMessageListener : createAnimatedCarouselNotification : Can't show "
            + "animated carousel. Images count is less than 3");
        return false;
      }
      String campaignId = MoEngageNotificationUtils.getCampaignIdIfAny(extras);
      MoEngageNotificationUtils.fetchAndSaveImages(context, imagesArray, campaignId);
      if (carouselObject.has("anim_direction")){
        direction = carouselObject.getString("anim_direction");
        switch (direction){
          case PushConstants.CAROUSEL_ANIMATION_LEFT_TO_RIGHT:
            rv.setViewVisibility(R.id.flipper_layout_left_to_right, View.VISIBLE);
            break;
          case PushConstants.CAROUSEL_ANIMATION_RIGHT_TO_LEFT:
            rv.setViewVisibility(R.id.flipper_layout_right_to_left, View.VISIBLE);

        }
      }else {
        rv.setViewVisibility(R.id.flipper_layout_right_to_left, View.VISIBLE);
      }
      for (int idx = 0; idx < imagesCount; idx++) {
        String fileName =
            campaignId + imagesArray.getJSONObject(idx).getString(PushActionMapperConstants.IMG_ID);
        Bitmap bmp = MoEngageNotificationUtils.loadImageFromStorage(context, fileName);
        if (bmp != null) {
          int viewFlipperImageId = MoEngageNotificationUtils.getViewFlipperImageId(idx, direction);
          rv.setImageViewBitmap(viewFlipperImageId, bmp);
          rv.setViewVisibility(viewFlipperImageId, View.VISIBLE);
          rv.setOnClickPendingIntent(viewFlipperImageId,
              MoEngageNotificationUtils.getImagePendingIntent(context, finalIntent, idx,
                  imagesArray));
        } else {
          Logger.v("PushMessageListener : createAnimatedCarouselNotification : One of the images "
              + "is null rolling back to narrow style");
          MoEngageNotificationUtils.deleteImagesFromInternal(context, campaignId);
          return false;
        }
      }
      rv.setViewVisibility(R.id.next_btn, View.GONE);
      rv.setViewVisibility(R.id.prev_btn, View.GONE);
      return true;
    } catch (Exception e) {
      Logger.f("PushMessageListener : createAnimatedCarouselNotification : Exception "
          + "occurred "
          + e);
      return false;
    }
  }

  public final void onMessagereceived(Context context, Map<String, String> extras) {
    Bundle bundle = MoEUtils.convertMapToBundle(extras);
    if (bundle != null) onMessagereceived(context, bundle);
  }

  private boolean shouldMakeGeoFenceCall(Bundle bundle){
    if (bundle.containsKey(PushConstants.PUSH_PAYLOAD_EXTRA)){
      return bundle.getString(PushConstants.PUSH_PAYLOAD_EXTRA).equals("true");
    }
    return true;
  }

  private void logNotificationState(Context context){
    try{
      MoEHelper.getInstance(context).setUserAttribute("PUSH_PREFERENCE_ANDROID",
          NotificationManagerCompat.from(context).areNotificationsEnabled());
    } catch(Exception e){
      Logger.f("PushMessageListener: logNotificationState: ",e);
    }
  }

  private boolean isMessageShown(Context context, String campaignId) {
    Cursor cursor = null;
    try {
      cursor = context.getContentResolver()
          .query(MoEDataContract.CampaignListEntity.getContentUri(context), null, null, null,
              null);
      if (cursor != null && cursor.moveToFirst()) {
        do {
          String savedCampaignId =
              cursor.getString(cursor.getColumnIndex(CampaignListEntity.CAMPAIGN_ID));
          if (!TextUtils.isEmpty(savedCampaignId) && savedCampaignId.equals(campaignId)) {
            Logger.e("PushMessageListener isDuplicateMessage() : Campaign already shown : "
                + campaignId);
            return true;
          }
        } while (cursor.moveToNext());
      }
    } catch (Exception e) {
      Logger.e("PushMessageListener isDuplicateMessage() ", e);
    } finally {
      if (cursor != null) {
        cursor.close();
      }
    }
    return false;
  }
}
