/* ************************************************************************
 *
 * MOENGAGE CONFIDENTIAL
 * __________________
 *
 *  [2014] - [2015] MoEngage Inc.
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of MoEngage Inc. The intellectual and technical concepts
 * contained herein are proprietary to MoEngage Incorporated
 * and its suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from MoEngage Incorporated.
 */
package com.moengage.pushbase.push;

import android.app.AlarmManager;
import android.app.Notification;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.net.Uri;
import android.os.Build;
import android.os.Build.VERSION;
import android.os.Bundle;
import android.provider.Settings;
import android.support.annotation.Nullable;
import android.support.v4.app.NotificationCompat;
import android.text.TextUtils;
import android.util.DisplayMetrics;
import android.view.View;
import android.view.WindowManager;
import android.widget.RemoteViews;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import com.moengage.core.MoEEventManager;
import com.moengage.core.MoEUtils;
import com.moengage.pushbase.PushActionMapperConstants;
import com.moengage.pushbase.PushConstants;
import com.moengage.pushbase.R;
import com.moengage.pushbase.activities.SnoozeTracker;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import static com.moe.pushlibrary.utils.MoEHelperConstants.EXTRA_MSG_RECEIVED_TIME;
import static com.moe.pushlibrary.utils.MoEHelperConstants.GCM_EXTRA_ACTIVITY_NAME;
import static com.moe.pushlibrary.utils.MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID;
import static com.moe.pushlibrary.utils.MoEHelperConstants.GCM_EXTRA_CONFIRMATION_KEY;
import static com.moe.pushlibrary.utils.MoEHelperConstants.GCM_EXTRA_CONFIRMATION_VALUE;
import static com.moe.pushlibrary.utils.MoEHelperConstants.GCM_EXTRA_GEO_ID;
import static com.moe.pushlibrary.utils.MoEHelperConstants.GCM_EXTRA_UNIQUE_ID;
import static com.moe.pushlibrary.utils.MoEHelperConstants.NOTIFICATION_RECEIVED_MOE;

/**
 * Utility class for Notifications sent from MoEngage platform.
 *
 * @author MoEngage (abhishek@moengage.com)
 * @version 1.0
 * @since 5.3
 */
public final class MoEngageNotificationUtils {

  private MoEngageNotificationUtils() {
    //constructor intentionally hidden
  }

  /**
   * Checks whether the GCM broadcast was being delivered from the MoEngage
   * Platform or not
   *
   * @param intent The Intent associated with the Broadcast
   * @return Returns true if the GCM intent was being delivered from MoEngage
   * Platform
   */
  public static boolean isFromMoEngagePlatform(Intent intent) {
    Bundle extras = intent.getExtras();
    return isFromMoEngagePlatform(extras);
  }

  /**
   * Checks whether the GCM broadcast was being delivered from the MoEngage
   * Platform or not
   *
   * @param extras Retrieves a map of extended data from the GCM intent;
   * <code>intent.getExtras()</code>
   * @return Returns true if the GCM intent was being delivered from MoEngage
   * Platform
   */
  public static boolean isFromMoEngagePlatform(Bundle extras) {
    try {
      if (null == extras) {
        Logger.e("MoEngageNotificationUtils:No Intent extra available");
      } else if (extras.containsKey(MoEHelperConstants.GCM_EXTRA_CONFIRMATION_KEY)
          && extras.getString(MoEHelperConstants.GCM_EXTRA_CONFIRMATION_KEY)
          .equals(MoEHelperConstants.GCM_EXTRA_CONFIRMATION_VALUE)) {
        return true;
      }
    } catch (Exception e) {
      Logger.f("MoEngageNotificationUtils: isFromMoEngagePlatform ", e);
    }
    return false;
  }

  public static boolean isSilentPush(Bundle extras) {
    String notificationType = extras.getString(NOTIFICATION_CONTENT_TYPE);
    return !TextUtils.isEmpty(notificationType) && notificationType.equals(
        NOTIFICATION_SILENT_NOTIFICATION);
  }

  public static boolean isPushToInbox(Bundle extras) {
    return extras.containsKey(DIRECT_PUSH_INBOX);
  }

  public static void stripMoEngageExtras(Bundle extras) {
    extras.remove(EXTRA_MSG_RECEIVED_TIME);
    extras.remove(NOTIFICATION_RECEIVED_MOE);
    extras.remove(GCM_EXTRA_CAMPAIGN_ID);
    extras.remove(GCM_EXTRA_CONFIRMATION_KEY);
    extras.remove(GCM_EXTRA_ACTIVITY_NAME);
    extras.remove(GCM_EXTRA_GEO_ID);
    extras.remove(GCM_EXTRA_UNIQUE_ID);
  }

  public static boolean isMoEngageCampaign(Bundle extras) {
    if (extras == null || !extras.containsKey(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID)) {
      Logger.e("MoEngageNotificationUtils:isAMoEngageCampaign--> no campaign ID so skipping "
          + "to show notification");
      return false;
    }
    return true;
  }

  public static void setNotificationId(Intent finalIntent, int notificationId) {
    finalIntent.putExtra(EXTRA_KEY_NOTIFICATIONID, notificationId);
  }

  private static final String EXTRA_KEY_NOTIFICATIONID = "MOE_NOTIFICATION_ID";

  public static int getNotificationIdIfAny(Bundle extras) {
    if (extras.containsKey(EXTRA_KEY_NOTIFICATIONID)) {
      return extras.getInt(EXTRA_KEY_NOTIFICATIONID);
    }
    return -1;
  }

  public static String getCampaignIdIfAny(Bundle extras) {
    if (extras.containsKey(NOTIFICATION_CAMPAIGN_ID)) {
      return extras.getString(NOTIFICATION_CAMPAIGN_ID);
    } else {
      return null;
    }
  }

  public static String getNotificationTypeIfAny(Bundle extras) {
    return extras.getString(NOTIFICATION_CONTENT_TYPE);
  }

  public static String getNotificationTitleIfAny(Bundle extras) {
    return extras.getString(NOTIFICATION_TITLE);
  }

  public static void setTitleIfPresent(Bundle extras, NotificationCompat.Builder builder) {
    builder.setContentTitle(getNotificationTitleIfAny(extras));
  }

  public static String getNotificationContentTextIfAny(Bundle extras) {
    return extras.getString(NOTIFICATION_CONTENT);
  }

  public static void setContentIfPresent(Bundle extras, NotificationCompat.Builder builder) {
    builder.setContentText(getNotificationContentTextIfAny(extras));
  }

  public static String getNotificationSubTextIfAny(Bundle extras) {
    return extras.getString(NOTIFICATION_SUB_TEXT);
  }

  public static void setSubTextIfAny(Bundle extras, NotificationCompat.Builder builder) {
    builder.setSubText(getNotificationSubTextIfAny(extras));
  }

  public static void setNotificationPriorityIfPresentAndSupported(Bundle extras,
      NotificationCompat.Builder builder) {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
      try {
        if (extras.containsKey(NOTIFICATION_PRIORITY)) {
          int notificationPriority = Integer.parseInt(extras.getString(NOTIFICATION_PRIORITY));
          if (notificationPriority >= Notification.PRIORITY_MIN
              && notificationPriority <= Notification.PRIORITY_MAX) {
            builder.setPriority(notificationPriority);
          } else {
            builder.setPriority(Notification.PRIORITY_DEFAULT);
          }
        }
      } catch (Exception e) {
        Logger.f("MoEngageNotificationUtils: setNotificationPriorityIfPresentAndSupported", e);
      }
    }
  }

  public static void setSmallIcon(Context context, NotificationCompat.Builder notificationBuilder,
      ConfigurationProvider configProvider) {
    try {
      int smallNotificationIconResourceId = 0;
      if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) {
        smallNotificationIconResourceId = configProvider.getNotificationLargeIconIfAny();
      } else {
        smallNotificationIconResourceId = configProvider.getNotificationSmallIcon();
      }
      if (smallNotificationIconResourceId != -1){
        notificationBuilder.setSmallIcon(smallNotificationIconResourceId);
      }
    } catch (Exception e) {
      Logger.f("MoEngageNotificationUtils: setSmallIcon", e);
    }
  }

  public static void setColorOrLargeIconIfPresentAndSupported(Context context, Bundle extras,
      NotificationCompat.Builder builder, ConfigurationProvider provider) {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
      try {
        int largeIcon = provider.getNotificationLargeIconIfAny();
        Bitmap bmpLargeIcon = null;
        int color = provider.getNotificationColor();
        // setting notification color
        if (color != -1) {
          builder.setColor(context.getResources().getColor(color));
        }
        if (provider.isNotificationLargeIconOptedOut()) return;
        if (extras.containsKey(NOTIFICATION_LARGE_ICON)) {
          String nLrgIcon = extras.getString(NOTIFICATION_LARGE_ICON);
          if (TextUtils.isEmpty(nLrgIcon) && largeIcon != -1) {
            //if notification does not specify large icon
            bmpLargeIcon = BitmapFactory.decodeResource(context.getResources(), largeIcon, null);
          } else {
            bmpLargeIcon = MoEHelperUtils.downloadImageBitmap(nLrgIcon);
            //if large icon download fails
            if (bmpLargeIcon == null && largeIcon != -1) {
              bmpLargeIcon = BitmapFactory.decodeResource(context.getResources(), largeIcon, null);
            }
          }
        } else if (largeIcon != -1) {
          bmpLargeIcon = BitmapFactory.decodeResource(context.getResources(), largeIcon, null);
        }
        if (bmpLargeIcon != null) {
          builder.setLargeIcon(bmpLargeIcon);
        }
      } catch (Exception e) {
        Logger.f("MoEngageNotificationUtils: setColorOrLargeIconIfPresentAndSupported", e);
      }
    }
  }

  public static void setSoundIfPresentAndSupported(Context context, Bundle extras,
      NotificationCompat.Builder builder, ConfigurationProvider configProvider) {
    //check if notification sound is enabled or not
    if (!configProvider.isNotificationSoundEnabled()) return;

    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.HONEYCOMB) {
      try {
        boolean playSound = true;
        Uri toneUri = null;
        String notificationTone = configProvider.getNotificationToneIfAny();
        if (null != notificationTone) {
          toneUri = Uri.parse(
              "android.resource://" + context.getPackageName() + "/raw/" + notificationTone);
        } else {
          toneUri = Settings.System.DEFAULT_NOTIFICATION_URI;
        }
        if (extras.containsKey(NOTIFICATION_TONE_DISABLED)) {
          playSound = false;
        } else {
          //Notification sound can be overridden from dashboard
          if (extras.containsKey(NOTIFICATION_TONE_SYSTEM)) {
            toneUri = Settings.System.DEFAULT_NOTIFICATION_URI;
          } else if (extras.containsKey(NOTIFICATION_TONE)) {
            notificationTone = extras.getString(NOTIFICATION_TONE);
            if (!TextUtils.isEmpty(notificationTone)) {
              toneUri = Uri.parse(
                  "android.resource://" + context.getPackageName() + "/raw/" + notificationTone);
            } else {
              Logger.v("Notification tone is not required");
            }
          }
        }
        if (playSound && null != toneUri) {
          builder.setSound(toneUri);
        }
      } catch (Exception e) {
        Logger.f("MoEngageNotificationUtils: setSoundIfPresentAndSupported", e);
      }
    }
  }

  public static void setVisibilityIfPresentAndSupported(Bundle extras,
      NotificationCompat.Builder builder) {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
      try {
        if (extras.containsKey(NOTIFICATION_PRIVACY)) {
          builder.setVisibility(Integer.parseInt(extras.getString(NOTIFICATION_PRIVACY)));
        }
      } catch (Exception e) {
        Logger.f("MoEngageNotificationUtils: setVisibilityIfPresentAndSupported", e);
      }
    }
  }

  public static void setTickerTextIfPresent(Bundle extras, NotificationCompat.Builder builder) {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.HONEYCOMB) {
      try {
        if (extras.containsKey(NOTIFICATION_TICKER)) {
          builder.setTicker(extras.getCharSequence(NOTIFICATION_TICKER));
        }
      } catch (Exception e) {
        Logger.f("MoEngageNotificationUtils: setTickerTextIfPresent", e);
      }
    }
  }

  public static void setCategoryIfPresentAndSupported(Bundle extras,
      NotificationCompat.Builder builder) {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
      try {
        if (extras.containsKey(NOTIFICATION_CATEGORY)) {
          builder.setCategory(extras.getString(NOTIFICATION_CATEGORY));
        }
      } catch (Exception e) {
        Logger.f("MoEngageNotificationUtils: setCategoryIfPresentAndSupported", e);
      }
    }
  }

  public static int getNotificationDisplayType(Bundle extras, Context context) {
    int type = ConfigurationProvider.getInstance(context).getNotificationDisplayType();
    if (isShowOnlyOneNotification(extras)) {
      type = 1;
    } else if (isShowMultipleNotification(extras)) {
      type = 2;
    }
    return type;
  }

  public static boolean isImageNotification(Bundle extras) {
    return extras.containsKey(NOTIFICATION_IMAGE_URL);
  }

  public static Bitmap scaleBitmapToDeviceSpecs(Bitmap imageBitmap, Context context) {
    if (imageBitmap == null) {
      Logger.e("MoEngageNotificationUtils: scaleBitmapToDeviceSpecs");
      return null;
    }
    if (imageBitmap.getWidth() > imageBitmap.getHeight()) {
      WindowManager windowManager =
          (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
      DisplayMetrics displayMetrics = new DisplayMetrics();
      windowManager.getDefaultDisplay().getMetrics(displayMetrics);
      int heightPixels = MoEHelperUtils.getPxFromDp(displayMetrics.densityDpi, FIXED_HEIGHT_DP);
      int widthPixels = 2 * heightPixels;
      if (widthPixels > displayMetrics.widthPixels) {
        widthPixels = displayMetrics.widthPixels;
      }
      if (Build.MANUFACTURER.equals("OPPO")){
        widthPixels -= MoEHelperUtils.getPxFromDp(displayMetrics.densityDpi, 16);
      }
      try {
        imageBitmap = Bitmap.createScaledBitmap(imageBitmap, widthPixels, heightPixels, true);
      } catch (OutOfMemoryError e1) {
        Logger.f("MoEngageNotificationUtils: scaleBitmapToDeviceSpecs", e1);
      } catch (Exception e) {
        Logger.f("MoEngageNotificationUtils: scaleBitmapToDeviceSpecs", e);
      }
    }
    return imageBitmap;
  }

  public static boolean isAutoCancelEnabled(Bundle extras) {
    return !extras.containsKey(NOTIFICATION_DISABLE_AUTO_CANCEL);
  }

  public static String convertBundletoJSONString(Bundle newBundle) {
    Set<String> keys = newBundle.keySet();
    JSONObject jsonObject = new JSONObject();
    for (String key : keys) {
      try {
        jsonObject.put(key, newBundle.get(key));
      } catch (Exception e) {
        Logger.f("MoEngageNotificationUtils: convertBundletoJSONString", e);
      }
    }

    return jsonObject.toString();
  }

  public static String getNotificationCouponCode(Bundle extras) {
    return extras.getString(NOTIFICATION_COUPON_CODE);
  }

  public static boolean hasCouponCode(Bundle extras) {
    return extras.containsKey(NOTIFICATION_COUPON_CODE);
  }

  public static boolean isVibrationDisabled(Bundle extras) {
    return extras.containsKey(NOTIFICATION_DISABLE_VIBRATION);
  }

  public static int getNotificationLedLightColor(Bundle extras) {
    try {
      return Integer.parseInt(extras.getString(NOTIFICATION_LED_LIGHT_COLOR));
    } catch (Exception e) {
      return -1;
    }
  }

  public static boolean isShowOnlyOneNotification(Bundle extras) {
    return extras.containsKey(NOTIFICATION_DISP_TYPE_SINGLE);
  }

  public static boolean isShowMultipleNotification(Bundle extras) {
    return extras.containsKey(NOTIFICATION_DISP_TYPE_MULTIPLE);
  }

  public static boolean isPublicVersionAvailable(Bundle extras) {
    return extras.containsKey(NOTIFICATION_PUBLIC_VERSION);
  }

  //public static void setPublicVersionIfAvailableAndSupported(Bundle extras, Context context,
  // NotificationCompat.Builder builder){
  //  if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
  //    try{
  //      if(isPublicVersionAvailable(extras)){
  //        String publicText = extras.getString(NOTIFICATION_PUBLIC_VERSION);
  //        //TODO IMPLEMENT THE public version notification creation
  //        Logger.e("Public Version is not yet implemented");
  //
  //      }
  //    }catch(Exception e){
  //
  //    }
  //  }
  //}

  public static void setActionButtonIfPresentAndSupported(Context context, Bundle extras,
      NotificationCompat.Builder builder, Intent redirectIntent, int notificationId) {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
      try {
        PendingIntent finalIntent = null;
        boolean updateExisting = getNotificationDisplayType(extras, context) == 1;
        JSONArray actions = getActions(extras);
        if (null == actions) {
          //OLD SDK deprecated KEY
          if (extras.containsKey(NOTIFICATION_ACTION)) {
            finalIntent = getContentIntent(context, redirectIntent, updateExisting, notificationId);
            builder.addAction(0, extras.getString(NOTIFICATION_ACTION), finalIntent);
          }
        } else {
          int btnCount = actions.length();
          for (int i = 0; i < btnCount; i++) {
            Intent actionIntent = null;
            JSONObject actionObject = actions.getJSONObject(i);
            Logger.v("MoEngageNotificationUtils : action button: " + actionObject.toString());
            String actionTag = actionObject.getString(PushActionMapperConstants.KEY_ACTION_TAG);
            String actionMessage =
                actionObject.getString(PushActionMapperConstants.KEY_ACTION_TITLE);
            String actionIcon = actionObject.has(PushActionMapperConstants.KEY_ACTION_ICON)
                ? actionObject.getString(PushActionMapperConstants.KEY_ACTION_ICON) : null;
            if (actionTag.equals(PushActionMapperConstants.ACTION_REMIND_EXACT) || actionTag.equals(
                PushActionMapperConstants.ACTION_REMIND_INEXACT)) {
              actionIntent = getIntentForSnooze(context);
              actionIntent.putExtras(redirectIntent.getExtras());
            } else {
              actionIntent = redirectIntent;
            }
            actionIntent.putExtra(PushActionMapperConstants.KEY_ACTION_TAG, actionTag);
            actionIntent.putExtra(PushActionMapperConstants.KEY_ACTION_PAYLOAD,
                actionObject.toString());
            actionIntent.putExtra(PushActionMapperConstants.KEY_ACTION_ID,
                actionObject.getString(PushActionMapperConstants.KEY_ACTION_ID));
            int actionIntentId = notificationId + ((i + 1) * 1000);
            finalIntent = PendingIntent.getActivity(context, actionIntentId, actionIntent,
                PendingIntent.FLAG_UPDATE_CURRENT);
            int imageResource = 0;
            if (!TextUtils.isEmpty(actionIcon)) {
              try {
                imageResource = context.getResources()
                    .getIdentifier(actionIcon, "drawable", context.getPackageName());
                if (imageResource == 0) {
                  imageResource = android.R.drawable.class.getField(actionIcon).getInt(null);
                }
              } catch (Exception e) {
                Logger.f("MoEngageNotificationUtils: setActionButtonIfPresentAndSupported", e);
              }
            }
            if (!(extras.containsKey(PushActionMapperConstants.KEY_RENOTIFY) && actionTag.equals(
                PushActionMapperConstants.ACTION_REMIND_INEXACT))) {
              builder.addAction(imageResource, actionMessage, finalIntent);
            }
          }
        }
      } catch (Exception e) {
        Logger.f("MoEngageNotificationUtils: setActionButtonIfPresentAndSupported", e);
      }
    }
  }

  public static final PendingIntent getContentIntent(final Context context,
      final Intent finalIntent, final boolean update, final int notificationId) {
    return PendingIntent.getActivity(context, notificationId, finalIntent,
        PendingIntent.FLAG_UPDATE_CURRENT);
  }

  public static JSONArray getActions(Bundle extras) {
    String actions = extras.getString(NOTIFICATION_ACTION_BUTTONS);
    if (TextUtils.isEmpty(actions)) {
      return null;
    }
    try {
      return new JSONArray(actions);
    } catch (Exception e) {
      Logger.f("MoEngageNotificationUtils: getActions", e);
    }
    return null;
  }

  //ADDED IN v5.3
  private static final String NOTIFICATION_TONE_DISABLED = "gcm_sound_disabled";
  private static final String NOTIFICATION_TONE = "gcm_tone";
  private static final String NOTIFICATION_TONE_SYSTEM = "gcm_tone_system";
  private static final String NOTIFICATION_DISP_TYPE_SINGLE = "gcm_show_single";
  private static final String NOTIFICATION_DISP_TYPE_MULTIPLE = "gcm_show_multi";
  private static final String NOTIFICATION_PUBLIC_VERSION = "gcm_pub_v";
  private static final String NOTIFICATION_CATEGORY = "gcm_category";
  private static final String NOTIFICATION_TICKER = "gcm_ticker";
  private static final String NOTIFICATION_LED_LIGHT_COLOR = "gcm_led";
  private static final String NOTIFICATION_SUB_TEXT = "gcm_subtext";
  private static final String NOTIFICATION_DISABLE_AUTO_CANCEL = "gcm_dnc";
  private static final String NOTIFICATION_DISABLE_VIBRATION = "gcm_no_vib";

  //ADDED IN v5.x
  private static final String NOTIFICATION_PRIVACY = "gcm_privacy";
  private static final String NOTIFICATION_PRIORITY = "gcm_priority";
  private static final String NOTIFICATION_LARGE_ICON = "gcm_l_ic";
  private static final String NOTIFICATION_ACTION_BUTTONS = "gcm_actions";

  //since v3.7
  private static final String NOTIFICATION_TITLE = "gcm_title";
  private static final String NOTIFICATION_CONTENT = "gcm_alert";
  private static final String NOTIFICATION_ACTION = "gcm_action_title";

  private static final String NOTIFICATION_ACTIVITY_NAME = "gcm_activityName";
  //private static final String NOTIFICATION_WEB_NOTIFICATION = "gcm_webNotification";
  private static final String NOTIFICATION_SILENT_NOTIFICATION = "gcm_silentNotification";
  private static final String NOTIFICATION_CONTENT_TYPE = "gcm_notificationType";
  private static final String NOTIFICATION_WEB_URL = "gcm_webUrl";
  private static final String NOTIFICATION_CAMPAIGN_ID = "gcm_campaign_id";
  private static final String NOTIFICATION_GEO_ID = "gcm_geo_id";
  private static final String NOTIFICATION_COUPON_CODE = "gcm_coupon_code";
  private static final String NOTIFICATION_IMAGE_URL = "gcm_image_url";
  private static final String NOTIFICATION_MSG_TAG = "gcm_msg_tag";

  private static final String GCM_ACTION_NAME = "name";
  private static final String GCM_ACTION_ICON = "ic";
  private static final String GCM_ACTION_DEEPLINK = "dl";
  private static final String GCM_ACTION_CLASSNAME = "cls";
  private static final String GCM_ACTION_EXTRAS = "ex";
  private static final String GCM_ACTION_LINK = "lk";

  //since v5.3.02
  private static String MSG_TTL = "gcm_msgttl";
  private static String MSG_AUTODISMISS = "gcm_dismiss";
  private static final String DIRECT_PUSH_INBOX = "gcm_push2inbox";

  private static final long DEFAULT_TTL = 7776000000L; //90days validity
  private static final String RECIPIENT_USER_ID = "unique_id";
  private static final int FIXED_HEIGHT_DP = 192;
  private static final int REQ_CLEARED = 501;

  //Added in v5.3.29
  private static String GCM_MESSAGE_IGNORE = "gcm_message_ignore";

  /**
   * Key to be passed in payload for carousel notification
   */
  private static final String GCM_CAROUSEL_NOTIFICATION = "gcm_carousel";

  /**
   * Key to be passed in payload for carousel images info like id,url,deep linking
   */
  private static final String NOTIFICATION_CUSTOM_IMAGES = "gcm_images";

  public static String getRedirectActivityNameIfAny(Bundle extras) {
    if (extras.containsKey(NOTIFICATION_ACTIVITY_NAME)) {
      return extras.getString(NOTIFICATION_ACTIVITY_NAME);
    } else {
      return null;
    }
  }

  public static String getDeeplinkURIStringIfAny(Bundle extras) {
    if (extras.containsKey(NOTIFICATION_WEB_URL)) {
      return extras.getString(NOTIFICATION_WEB_URL);
    } else {
      return null;
    }
  }

  public static String getCampaignGeoIDIfAny(Bundle extras) {
    if (extras.containsKey(NOTIFICATION_GEO_ID)) {
      return extras.getString(NOTIFICATION_GEO_ID);
    } else {
      return null;
    }
  }


  public static long getNotificationTTL(Bundle extras, long receivedTime) {
    if (extras.containsKey(PushConstants.MOE_NOTIFICATION_EXPIRY)){
      //expiry key from dashboard, value received in seconds. convert to millis
      return (Long.parseLong(extras.getString(PushConstants.MOE_NOTIFICATION_EXPIRY)) * 1000L);
    }else if (extras.containsKey(MSG_TTL)) {
      //legacy key, this was used when feature was not present on the dashboard.
      return convertTimeToLong(extras.getString(MSG_TTL));
    } else {
      return receivedTime + DEFAULT_TTL;
    }
  }

  public static String getMessageTagsIfAny(Bundle extras) {
    if (extras.containsKey(NOTIFICATION_MSG_TAG)) {
      return extras.getString(NOTIFICATION_MSG_TAG);
    }
    return null;
  }

  public static int getNotificationId(Context context, ConfigurationProvider provider,
      boolean update) {
    if (update) {
      return provider.getNotificationId();
    } else {
      int id = provider.getNotificationId();
      provider.updateNotificationId(++id);
      return id;
    }
  }

  public static void setNotificationStyle(Context context, Bundle extras,
      NotificationCompat.Builder builder) {
    if (isImageNotification(extras)) {
      String summary = getNotificationSubTextIfAny(extras);
      if (TextUtils.isEmpty(summary)) {
        summary = getNotificationContentTextIfAny(extras);
      }
      Bitmap new_bitmap = MoEHelperUtils.downloadImageBitmap(
          extras.getString(MoEHelperConstants.GCM_EXTRA_IMAGE_URL));
      if (null == new_bitmap) {
        builder.setStyle(new NotificationCompat.BigTextStyle().bigText(
            getNotificationContentTextIfAny(extras) + " "));
      } else {
        new_bitmap = scaleBitmapToDeviceSpecs(new_bitmap, context);
        if (VERSION.SDK_INT < 24) {
          builder.setStyle(new NotificationCompat.BigPictureStyle().bigPicture(new_bitmap)
              .setSummaryText(summary));
        } else {
          builder.setStyle(new NotificationCompat.BigPictureStyle().bigPicture(new_bitmap)
              .setSummaryText(getNotificationContentTextIfAny(extras)));
        }
      }
    } else {
      builder.setStyle(new NotificationCompat.BigTextStyle().bigText(
          getNotificationContentTextIfAny(extras) + " "));
    }
  }

  public static String getRecipientUserId(Bundle extras) {
    return extras.getString(RECIPIENT_USER_ID);
  }

  public static void setNotificationAutoDismissIfAny(Context context, int notificationId,
      Bundle extras) {
    long notificationExpiryTime = getNotificationExpiryTime(extras);
    if (notificationExpiryTime != -1) {
      Logger.v(
          "MoEngageNotificationUtils: setNotificationAutoDismissIfAny: setting an auto dismiss "
              + "after: "
              + notificationExpiryTime);
      Intent finalIntent = new Intent(context, MoEPushWorker.class);
      finalIntent.putExtra(MoEPushWorker.NOTIFICATION_DISMISS, notificationId);
      finalIntent.setAction(MoEPushWorker.NOTIFICATION_DISMISS);
      PendingIntent intent = PendingIntent.getService(context, notificationId, finalIntent,
          PendingIntent.FLAG_UPDATE_CURRENT);
      AlarmManager alarmManager = (AlarmManager) context.getSystemService(Context.ALARM_SERVICE);
      alarmManager.set(AlarmManager.RTC_WAKEUP, notificationExpiryTime, intent);
    }
  }

  public static void setNotificationClearedCallback(Context context,
      NotificationCompat.Builder builder, int notificationId, Bundle extras) {
    Intent finalIntent = new Intent(context, MoEPushWorker.class);
    finalIntent.putExtras(extras);
    finalIntent.setAction(MoEPushWorker.NOTIFICATION_CLEARED);
    PendingIntent intent =
        PendingIntent.getService(context, notificationId | REQ_CLEARED, finalIntent,
            PendingIntent.FLAG_UPDATE_CURRENT);
    builder.setDeleteIntent(intent);
  }

  /**
   * Returns the MoEngage Extras which need to be set to the final deeplinked intent
   *
   * @param gcmIntentExtra the {@link Bundle} which was passed with the GCM payload
   */
  public static Bundle getMoEngageExtras(final Bundle gcmIntentExtra) {
    Bundle moengageExtras = new Bundle();
    //moengageExtras.putLong(EXTRA_MSG_RECEIVED_TIME, System.currentTimeMillis());
    moengageExtras.putString(NOTIFICATION_RECEIVED_MOE, "true");
    moengageExtras.putString(GCM_EXTRA_CAMPAIGN_ID,
        MoEngageNotificationUtils.getCampaignIdIfAny(gcmIntentExtra));
    moengageExtras.putString(GCM_EXTRA_CONFIRMATION_KEY, GCM_EXTRA_CONFIRMATION_VALUE);
    moengageExtras.putString(GCM_EXTRA_ACTIVITY_NAME,
        gcmIntentExtra.getString(GCM_EXTRA_ACTIVITY_NAME));
    moengageExtras.putString(GCM_EXTRA_GEO_ID, gcmIntentExtra.getString(GCM_EXTRA_GEO_ID));
    moengageExtras.putString(GCM_EXTRA_UNIQUE_ID, gcmIntentExtra.getString(GCM_EXTRA_UNIQUE_ID));
    return moengageExtras;
  }

  private static final String SEPARATOR = "%26";
  private static final String ASSIGN = "%3D";
  private static final String KEY_MOENGAGE_EXTRAS = "moextras";

  public static void setMoEngageExtrastoUri(Bundle extras, Uri.Builder uriBuilder) {
    try {
      if (null == extras || extras.isEmpty()) return;
      StringBuilder builder = new StringBuilder();
      Set<String> keySet = extras.keySet();
      int length = keySet.size();
      for (String key : keySet) {
        length--;
        if (TextUtils.isEmpty(key)) continue;
        if (MoEHelperConstants.GCM_EXTRA_WEB_URL.equals(key)) continue;
        builder.append(key);
        builder.append(ASSIGN);
        builder.append(extras.get(key));
        if (length > 0) {
          builder.append(SEPARATOR);
        }
      }
      uriBuilder.appendQueryParameter(KEY_MOENGAGE_EXTRAS, builder.toString());
    } catch (Exception e) {
      Logger.f("MoEHelperUtils: getMoEngageExtrasAsUriParam :", e);
    }
  }

  public static void setMoEngageExtrasToBundleIfAny(Intent intent) {
    try {
      if (null == intent) return;
      Uri data = intent.getData();
      if (null == data) return;
      Bundle bundle = intent.getExtras();
      if (null == bundle) bundle = new Bundle();
      //get key
      String extras = data.getQueryParameter(KEY_MOENGAGE_EXTRAS);
      if (TextUtils.isEmpty(extras)) return;
      //split extras here
      if (extras.contains(SEPARATOR)) {
        String[] keySet = extras.split(SEPARATOR);
        for (String kv : keySet) {
          String[] kvSplit = splitKVPairs(kv);
          if (kvSplit.length == 2) {
            bundle.putString(kvSplit[0], kvSplit[1]);
          }
        }
      } else {
        String[] kvSplit = splitKVPairs(extras);
        bundle.putString(kvSplit[0], kvSplit[1]);
      }
      intent.putExtras(bundle);
    } catch (Exception e) {
      Logger.f("MoEHelperUtils: getMoEngageExtrasAsUriParam :", e);
    }
  }

  private static String[] splitKVPairs(String kv) {
    return kv.split(ASSIGN);
  }

  public static boolean hasNotificationExpired(Bundle extras) {
    long notificationEndTime = getNotificationExpiryTime(extras);
    if (notificationEndTime == -1) {
      Logger.v("MoEngageNotificationUtils#hasNotificationExpired : Notification does not have "
          + "an expiry time");
      return false;
    }
    long currentSystemTime = System.currentTimeMillis();
    boolean result = currentSystemTime > notificationEndTime;
    if (result) {
      Logger.v("MoEngageNotificationUtils#hasNotificationExpired : "
          + "received notification has expired");
      extras.putBoolean(MoEHelperConstants.EXTRA_CAMPAIGN_EXPIRED, true);
    } else {
      Logger.v("MoEngageNotificationUtils#hasNotificationExpired : "
          + "received notification has not expired");
    }
    return result;
  }

  public static long getNotificationExpiryTime(Bundle extras) {
    String notificationDismissTime =
        extras.containsKey(MSG_AUTODISMISS) ? extras.getString(MSG_AUTODISMISS) : null;
    return convertTimeToLong(notificationDismissTime);
  }

  private static long convertTimeToLong(String time) {
    try {
      if (!TextUtils.isEmpty(time)) {
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");
        dateFormat.setTimeZone(TimeZone.getDefault());
        Date date = dateFormat.parse(time);
        return date.getTime();
      }
    } catch (ParseException e) {
      Logger.f("MoEngageNotificationUtils$hasNotificationExpired : exception while"
          + " parsing date "
          + e.getMessage());
    } catch (Exception e) {
      Logger.f("MoEngageNotificationUtils$hasNotificationExpired : exception while"
          + " parsing date "
          + e.getMessage());
    }
    return -1;
  }

  public static boolean isSkipNotificationCenter(Bundle extras) {
    String trueValue = "true";
    if (extras.containsKey(GCM_MESSAGE_IGNORE)) {
      return trueValue.equalsIgnoreCase(extras.getString(GCM_MESSAGE_IGNORE).trim());
    }
    return false;
  }

  /**
   * Check if the notification is Carousel Notification
   *
   * @param extras push payload which is received
   */
  public static boolean isCarouselNotification(Bundle extras) {
    return extras.containsKey(GCM_CAROUSEL_NOTIFICATION);
  }

  /**
   * Get Title for Notification
   *
   * @param carouselObject JSONObject received in carousel push payload
   * @param extras Bundle objcet received in carousel push payload
   * @return Title for Carousel Notification Style if present, if not then Title for Normal
   * Notification
   */
  public static String getCarouselTitle(JSONObject carouselObject, Bundle extras) {
    return getVal(carouselObject, extras, NOTIFICATION_TITLE);
  }

  /**
   * Get Text for Notification
   *
   * @param carouselObject JSONObject received in carousel push payload
   * @param extras Bundle objcet received in carousel push payload
   * @return Text for Carousel Notification Style if present, if not then Text for Normal
   * Notification
   */
  public static String getCarouselText(JSONObject carouselObject, Bundle extras) {
    return getVal(carouselObject, extras, NOTIFICATION_CONTENT);
  }

  /**
   * Get SubText for Notification
   *
   * @param carouselObject JSONObject received in carousel push payload
   * @param extras Bundle objcet received in carousel push payload
   * @return SubText for Carousel Notification Style if present, if not then SubText for Normal
   * Notification
   */
  public static String getCarouselSubText(JSONObject carouselObject, Bundle extras) {
    return getVal(carouselObject, extras, NOTIFICATION_SUB_TEXT);
  }

  /* (non-Javadoc)
   * Helper method to fetch value from Carousel JSONObject if present else from Extras push payload
   * @param jsonObject JSONObject received in push payload
   * @param extras Bundle object received in push payload
   * @param key Key of the attribute to be fetched
   */
  private static String getVal(JSONObject jsonObject, Bundle extras, String key) {
    try {
      return jsonObject.has(key) ? jsonObject.getString(key) : extras.getString(key);
    } catch (Exception e) {
      Logger.f("MoEngageNotificationUtils : getVal : Exception Occurred" + e);
    }
    return extras.getString(key);
  }

  /**
   * Helper method to get small notification icon resource for Carousel style notification
   *
   * @param context Application context
   * @return small notification resource
   */
  public static int getCarouselSmallNotificationIcon(Context context) {
    try {
      return ConfigurationProvider.getInstance(context).getNotificationSmallIcon();
    } catch (Exception e) {
      Logger.f("MoEngageNotificationUtils$getCarouselSmallNotificationIcon", e);
    }
    return 0;
  }

  /**
   * Helper method to get large notification icon resource for Carousel style notification
   *
   * @param context Application context
   * @return large notification resource
   */
  public static int getCarouselLargeNotificationIcon(Context context) {
    try {
      return ConfigurationProvider.getInstance(context).getNotificationLargeIconIfAny();
    } catch (Exception e) {
      Logger.f("MoEngageNotificationUtils$getCarouselLargeNotificationIcon", e);
    }
    return 0;
  }

  /* (non-Javadoc)
   * Helper method to save Bitmap to internal storage
   * @param context Application context
   * @param fileName name for the file to be saved
   * @param bitmapImage Bitmap to be saved as file in internal storage
   */
  private static void saveToInternalStorage(Context context, String fileName, Bitmap bitmapImage) {
    if (context == null || fileName == null || bitmapImage == null) {
      Logger.v("MoEngageNotificationUtils$saveToInternalStorage : "
          + "context/fileName/bitmapImage is null");
    }
    FileOutputStream fos = null;
    try {
      fos = context.openFileOutput(fileName, Context.MODE_PRIVATE);
      // Use the compress method on the BitMap object to write image to the OutputStream
      bitmapImage.compress(Bitmap.CompressFormat.PNG, 100, fos);
    } catch (Exception e) {
      Logger.f("MoEngageNotificationUtils$saveToInternalStorage: Exception occurred ", e);
    } finally {
      try {
        fos.close();
      } catch (IOException e) {
        Logger.f("MoEngageNotificationUtils$saveToInternalStorage: Exception occurred ", e);
      } catch (Exception e) {
        Logger.f("MoEngageNotificationUtils$saveToInternalStorage: Exception occurred ", e);
      }
    }
  }

  /**
   * Helper method to load Bitmap from internal storage
   *
   * @param context Application context
   * @param fileName of the Bitmap file to be load
   */
  public static Bitmap loadImageFromStorage(Context context, String fileName) {
    if (context == null || fileName == null) {
      Logger.v("MoEngageNotificationUtils$loadImageFromStorage : context/fileName is null");
      return null;
    }
    try {
      File file = context.getFileStreamPath(fileName);
      if (file.exists()) {
        FileInputStream inputStream = context.openFileInput(fileName);
        Bitmap b = BitmapFactory.decodeStream(inputStream);
        return b;
      }
    } catch (FileNotFoundException e) {
      Logger.f("MoEngageNotificationUtils$loadImageFromStorage: Exception occurred ", e);
    } catch (Exception e) {
      Logger.f("MoEngageNotificationUtils$loadImageFromStorage: Exception occurred ", e);
    }
    return null;
  }

  /**
   * Helper method to get Images Array from JSONObject passed in push payload
   *
   * @param carouselObject JSONObject received in carousel push payload
   */
  public static JSONArray getImagesArray(JSONObject carouselObject) {
    try {
      return carouselObject.getJSONArray(NOTIFICATION_CUSTOM_IMAGES);
    } catch (Exception e) {
      Logger.f("MoEngageNotificationUtils$getImagesArray : Exception Occurred", e);
    }
    return null;
  }

  /**
   * Helper method to download the images
   *
   * @param context Application context
   * @param images JSONArray of images
   * @param campaignID Campaign ID of push received
   */
  public static void fetchAndSaveImages(Context context, JSONArray images, String campaignID) {
    try {
      if (null != images && context != null) {
        int imgCount = images.length();
        for (int i = 0; i < imgCount; i++) {
          JSONObject imageObject = images.getJSONObject(i);
          Logger.v("MoEngageNotificationUtils$fetchAndSaveImages : carousel images: "
              + imageObject.toString());
          String fileName = campaignID + imageObject.getString(PushActionMapperConstants.IMG_ID);
          String url = imageObject.getString(PushActionMapperConstants.IMG_URL);
          Bitmap bmp = MoEHelperUtils.downloadImageBitmap(url);
          if (bmp != null) {
            bmp = scaleBitmapToDeviceSpecs(bmp, context);
            Logger.v("MoEngageNotificationUtils$fetchAndSaveImages : save bitmap for "
                + fileName
                + " and url: "
                + url);
            saveToInternalStorage(context, fileName, bmp);
          } else {
            Logger.v("MoEngageNotificationUtils$fetchAndSaveImages : Failed to download image for "
                + url);
          }
        }
      }
    } catch (Exception e) {
      Logger.f("MoEngageNotificationUtils$fetchAndSaveImages : Exception Occurred ", e);
    }
  }

  /**
   * Helper method to get Pending Intent for Image in Carousel Style Notification
   *
   * @param context Application context
   * @param actionIntent action to be performed when image is clicked
   * @param idx request code
   * @param imagesArray JSONArray of images
   * @return pendingIntent
   */
  public static PendingIntent getImagePendingIntent(Context context, Intent actionIntent, int idx,
      JSONArray imagesArray) {
    try {
      JSONObject imageObject = imagesArray.getJSONObject(idx);
      actionIntent.putExtra(PushActionMapperConstants.KEY_ACTION_TAG,
          PushActionMapperConstants.ACTION_NAVIGATE);
      actionIntent.putExtra(PushActionMapperConstants.IMG_ID,
          imageObject.getString(PushActionMapperConstants.IMG_ID));
      actionIntent.putExtra(PushActionMapperConstants.KEY_ACTION_PAYLOAD, imageObject.toString());
      return PendingIntent.getActivity(context, idx, actionIntent,
          PendingIntent.FLAG_UPDATE_CURRENT);
    } catch (Exception e) {
      Logger.f("MoEngageNotificationUtils$getImagePendingIntent : Exception Occurred ", e);
    }
    return null;
  }

  /**
   * Helper method to get next Image from Images Array based on Navigation Direction
   *
   * @param extras Bundle received in push payload
   * @return index of next Image in Images Array
   */
  public static int getNextImageIndex(Bundle extras) {
    try {
      JSONArray imagesArray = getImagesArray(getCarouselObject(extras));
      int currImgIndex = extras.containsKey(PushActionMapperConstants.IMG_INDEX) ? extras.getInt(
          PushActionMapperConstants.IMG_INDEX) : 0;
      if (extras.containsKey(PushActionMapperConstants.IMG_ACTION_NEXT)) {
        if (extras.getBoolean(PushActionMapperConstants.IMG_ACTION_NEXT)) {
          int nextImgIndex = ++currImgIndex;
          if (nextImgIndex == imagesArray.length()) nextImgIndex = 0;
          return nextImgIndex;
        } else {
          int prevImgIndex = --currImgIndex;
          if (prevImgIndex == -1) prevImgIndex = imagesArray.length() - 1;
          return prevImgIndex;
        }
      }
    } catch (Exception e) {
      Logger.f("MoEngageNotificationUtils$getNextImageFileName : Exception Occured" + e);
    }
    return 0;
  }

  /**
   * Helper method to get carousel JSON object from Bundle received in push payload
   *
   * @param extras Bundle received in push payload
   * @return JSON object
   */
  @Nullable public static JSONObject getCarouselObject(Bundle extras) {
    String carousel = extras.getString(GCM_CAROUSEL_NOTIFICATION);
    if (TextUtils.isEmpty(carousel)) {
      return null;
    }
    try {
      return new JSONObject(carousel);
    } catch (Exception e) {
      Logger.f("MoEngageNotificationUtils$getCarouselObject", e);
    }
    return null;
  }

  /**
   * Helper method to get Pending Intent for navigation button in Carousel Style Notification
   *
   * @param context Application context
   * @param actionIntent action to be performed when image is clicked
   * @param actionTag Tag for navigation direction
   * @param requestCode requestCode for Pending Intent
   * @param idx current Image Index in Carousel Style Notification
   * @return pendingIntent
   */
  public static PendingIntent getNavPendingIntent(Context context, Intent actionIntent,
      String actionTag, int requestCode, int idx) {
    actionIntent.putExtra(PushActionMapperConstants.KEY_ACTION_TAG, actionTag);
    actionIntent.putExtra(actionTag, true);
    actionIntent.putExtra(PushActionMapperConstants.IMG_INDEX, idx);
    return PendingIntent.getService(context, requestCode, actionIntent,
        PendingIntent.FLAG_UPDATE_CURRENT);
  }

  /**
   * Helper method to delete images from Internal storage
   *
   * @param context Application context
   * @param campaignId Campaign ID for which Images have to be deleted
   */
  public static void deleteImagesFromInternal(Context context, String campaignId) {
    if (context == null) {
      Logger.v("MoEngageNotificationUtils$deleteImagesFromInternal context is null");
      return;
    }
    try {
      String image_files[] = context.fileList();
      for (String fileName : image_files) {
        if (fileName.contains(campaignId)) {
          context.deleteFile(fileName);
        }
      }
    } catch (Exception e) {
      Logger.f("MoEngageNotificationUtils$deleteImagesFromInternal Exception ocurred" + e);
    }
  }

  /**
   * Helper method to check if Notification has to be shown again
   *
   * @param extras Application context
   */
  public static boolean isReNotification(Bundle extras) {
    return extras.containsKey(PushActionMapperConstants.KEY_RENOTIFY);
  }

  /**
   * Helper method to add Action Button to Carousel Style Notification
   *
   * @param context Application context
   * @param rv Remote View used for creating Carousel Notification
   * @param extras Bundle passed as push payload
   * @param redirectIntent Intent to be triggered when notification is clicked
   */
  public static void addCarouselActionButton(Context context, RemoteViews rv, Bundle extras,
      Intent redirectIntent, int notificationId) {
    try {
      JSONArray actions = getActions(extras);
      if (actions != null) {
        PendingIntent finalIntent = null;
        int maxBtnCount = actions.length() <= 3 ? actions.length() : 3;
        for (int i = 0; i < maxBtnCount; i++) {
          Intent actionIntent = null;
          JSONObject actionObject = actions.getJSONObject(i);
          Logger.v("MoEngageNotificationUtils$addCarouselActionButton: " + actionObject.toString());
          String actionTag = actionObject.getString(PushActionMapperConstants.KEY_ACTION_TAG);
          String actionMessage = actionObject.getString(PushActionMapperConstants.KEY_ACTION_TITLE);
          String actionIcon =
              actionObject.has(PushActionMapperConstants.KEY_ACTION_ICON) ? actionObject.getString(
                  PushActionMapperConstants.KEY_ACTION_ICON) : null;
          if (actionTag.equals(PushActionMapperConstants.ACTION_REMIND_EXACT) || actionTag.equals(
              PushActionMapperConstants.ACTION_REMIND_INEXACT)) {
            actionIntent = getIntentForSnooze(context);
            actionIntent.putExtras(redirectIntent.getExtras());
          } else {
            actionIntent = redirectIntent;
          }
          actionIntent.putExtra(PushActionMapperConstants.KEY_ACTION_TAG, actionTag);
          actionIntent.putExtra(PushActionMapperConstants.KEY_ACTION_PAYLOAD,
              actionObject.toString());
          actionIntent.putExtra(PushActionMapperConstants.KEY_ACTION_ID,
              actionObject.getString(PushActionMapperConstants.KEY_ACTION_ID));
          finalIntent =
              PendingIntent.getActivity(context, (notificationId * (i + 100)), actionIntent,
                  PendingIntent.FLAG_UPDATE_CURRENT);
          int buttonId = getCarouselActionButtonId(i);
          if (buttonId == -1) return;
          int imageResource = 0;
          if (!TextUtils.isEmpty(actionIcon)) {
            try {
              imageResource = context.getResources()
                  .getIdentifier(actionIcon, "drawable", context.getPackageName());
              if (imageResource == 0) {
                imageResource = android.R.drawable.class.getField(actionIcon).getInt(null);
              }
            } catch (Exception e) {
              Logger.d("MoEngageNotificationUtils$addCarouselActionButton", e);
            }
          }
          rv.setTextViewText(buttonId, actionMessage);
          rv.setOnClickPendingIntent(buttonId, finalIntent);
          rv.setViewVisibility(buttonId, View.VISIBLE);
          rv.setTextViewCompoundDrawablesRelative(buttonId, imageResource, 0, 0, 0);
        }
      }
    } catch (Exception e) {
      Logger.f("MoEngageNotificationUtils$addCarouselActionButton : Exception", e);
    }
  }

  /* (non-Javadoc)
   * Helper method to get ID of Button
   * @param id index of action button in push payload
   */
  private static int getCarouselActionButtonId(int id) {
    int viewId = -1;
    switch (id) {
      case 0:
        viewId = R.id.action1;
        break;
      case 1:
        viewId = R.id.action2;
        break;
      case 2:
        viewId = R.id.action3;
        break;
      default:
        Logger.e("MoEngageNotificationUtils$getCarouselActionButtionId Invalid Id " + id);
        viewId = -1;
    }
    return viewId;
  }

  /**
   * Helper method to calculate top padding based on number of lines in Carousel Style Notification
   *
   * @param context Application context
   * @param hasThreeLines boolean value denoting if Carousel Notification has three lines or not
   * @return top padding for Carousel Style Notification
   */
  public static int calculateTopPadding(Context context, boolean hasThreeLines) {
    return context.getResources()
        .getDimensionPixelSize(
            hasThreeLines ? R.dimen.notification_top_pad_narrow : R.dimen.notification_top_pad);
  }

  /**
   * Helper method to get current time for Carousel Style Notification
   *
   * @return time in kk:mm format
   */
  public static String getTime() {
    Calendar calendar = Calendar.getInstance();
    SimpleDateFormat sdf = new SimpleDateFormat("kk:mm");
    return sdf.format(calendar.getTime());
  }

  /**
   * Helper method to disable Sound and Vibration for Carousel Re-Notification
   */
  public static void disableSoundAndVibration(Notification notification) {
    notification.sound = null;
    notification.vibrate = null;
  }

  /**
   * Helper method to get view ID of ImageView
   *
   * @param id index of image view in push payload
   */
  public static int getViewFlipperImageId(int id, String direction) {
    int viewId = -1;
    switch (direction) {
      case PushConstants.CAROUSEL_ANIMATION_LEFT_TO_RIGHT:
        viewId = getViewIdForLeftToRight(id);
        break;
      case PushConstants.CAROUSEL_ANIMATION_RIGHT_TO_LEFT:
        viewId = getViewIdForRightToLeft(id);
        break;
      default:
        Logger.e("MoEngageNotificationUtils$getViewFlipperImageId Invalid Id " + id);
        viewId = -1;
    }
    return viewId;
  }

  private static int getViewIdForLeftToRight(int id) {
    int viewId = -1;
    switch (id) {
      case 0:
        viewId = R.id.flip_picture1_lr;
        break;
      case 1:
        viewId = R.id.flip_picture2_lr;
        break;
      case 2:
        viewId = R.id.flip_picture3_lr;
        break;
      default:
        Logger.e("MoEngageNotificationUtils$getViewIdForLeftToRight Invalid Id " + id);
        viewId = -1;
    }
    return viewId;
  }

  private static int getViewIdForRightToLeft(int id) {
    int viewId = -1;
    switch (id) {
      case 0:
        viewId = R.id.flip_picture1_rl;
        break;
      case 1:
        viewId = R.id.flip_picture2_rl;
        break;
      case 2:
        viewId = R.id.flip_picture3_rl;
        break;
      default:
        Logger.e("MoEngageNotificationUtils$getViewIdForRightToLeft Invalid Id " + id);
        viewId = -1;
    }
    return viewId;
  }

  /**
   * Returns the snoozeTrackerIntent which will be started from the pending intent of the
   * notification snooze action
   *
   * @param context Instance of the Application Context
   */
  public final static Intent getIntentForSnooze(final Context context) {
    Intent snoozeTrackerIntent = new Intent(context, SnoozeTracker.class);
    snoozeTrackerIntent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_CLEAR_TASK);
    return snoozeTrackerIntent;
  }

  public static boolean isFromMoEngagePlatform(Map<String, String> extras) {
    try {
      if (null == extras) {
        Logger.e("MoEngageNotificationUtils:No Intent extra available");
      } else if (extras.containsKey(MoEHelperConstants.GCM_EXTRA_CONFIRMATION_KEY)) {
        String confirmationValue =
            (String) extras.get(MoEHelperConstants.GCM_EXTRA_CONFIRMATION_KEY);
        return confirmationValue.equals(MoEHelperConstants.GCM_EXTRA_CONFIRMATION_VALUE);
      }
    } catch (Exception e) {
      Logger.f("MoEngageNotificationUtils: isFromMoEngagePlatform ", e);
    }
    return false;
  }

  public static boolean isFromMoEngagePlatform(String payload) {
    try {
      JSONObject jsonPayload = new JSONObject(payload);
      Bundle bundlePayload = MoEUtils.jsonToBundle(jsonPayload);
      if (bundlePayload != null) {
        return isFromMoEngagePlatform(bundlePayload);
      }
    } catch (Exception e) {
      Logger.f("MoEngageNotificationUtils: isFromMoEngagePlatform ", e);
    }
    return false;
  }

  public static final void logNotificationImpression(final Context context, final Bundle extras) {
    try {
      if (!isFromMoEngagePlatform(extras)) return;
      if (!extras.containsKey(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID)) {
        return;
      }
      JSONObject newJson = new JSONObject();
      String campaignId = extras.getString(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID);
      if (campaignId != null && campaignId.contains("DTSDK")){
        campaignId = campaignId.substring(0, campaignId.indexOf("DTSDK"));
        extras.putString(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID, campaignId);
      }
      newJson.put(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID,
          extras.getString(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID));
      if (extras.containsKey(MoEHelperConstants.GCM_EXTRA_GEO_ID)) {
        String geoId = extras.getString(MoEHelperConstants.GCM_EXTRA_GEO_ID);
        if (!TextUtils.isEmpty(geoId)) newJson.put(MoEHelperConstants.GCM_EXTRA_GEO_ID, geoId);
      }
      if (extras.containsKey(MoEHelperConstants.GCM_EXTRA_UNIQUE_ID)) {
        String uniqueId = extras.getString(MoEHelperConstants.GCM_EXTRA_UNIQUE_ID);
        if (!TextUtils.isEmpty(uniqueId)) {
          newJson.put(MoEHelperConstants.GCM_EXTRA_UNIQUE_ID, uniqueId);
        }
      }
      if (extras.containsKey(MoEHelperConstants.EXTRA_CAMPAIGN_EXPIRED)) {
        newJson.put(MoEHelperConstants.EXTRA_CAMPAIGN_EXPIRED, true);
      }
      if (extras.containsKey("received_from")) {
        newJson.put("source", extras.getString("received_from"));
      }
      if (extras.containsKey("from_appOpen")) {
        newJson.put("from_appOpen", extras.getBoolean("from_appOpen"));
      }
      if (extras.containsKey(PushConstants.ATTR_PUSH_PROVIDER)) {
        newJson.put(PushConstants.ATTR_PUSH_PROVIDER,
            extras.getString(PushConstants.ATTR_PUSH_PROVIDER));
      }
      if (extras.containsKey(PushConstants.ATTR_CAMPAIGN_ATTRIBUTES)) {
        JSONObject campaignAttributes =
            new JSONObject(extras.getString(PushConstants.ATTR_CAMPAIGN_ATTRIBUTES));
        if (campaignAttributes != null) {
          Iterator keys = campaignAttributes.keys();
          while (keys.hasNext()) {
            String key = (String) keys.next();
            String value = campaignAttributes.getString(key);
            newJson.put(key, value);
          }
        }
      }
      if (extras.containsKey("shownOffline")){
        newJson.put("shownOffline", true);
        extras.remove("shownOffline");
      }
      // track impression
      MoEEventManager.getInstance(context).trackEvent(NOTIFICATION_RECEIVED_MOE, newJson);

      //send impression data
      //MoEHelper.getInstance(context).syncInteractionDataNow();
    } catch (Exception e) {
      Logger.f("PushMessageListener:trackNotification", e);
    }
  }

  public static final void logNotificationClick(final Context context, final Intent intent) {
    try {
      if (null == intent) return;
      Bundle extras = intent.getExtras();
      if (extras == null) return;
      if (!isFromMoEngagePlatform(extras)) return;
      boolean fromInbox = MoEHelperUtils.isFromInbox(extras);
      if (extras.containsKey(NOTIFICATION_RECEIVED_MOE)) {
        //MoEUtils.setPushReceived(true, context); -- TODO see why if it is required
        if (!fromInbox) {
          String geoId = null;
          String uniqueId = null;
          if (extras.containsKey(MoEHelperConstants.GCM_EXTRA_GEO_ID)) {
            geoId = extras.getString(MoEHelperConstants.GCM_EXTRA_GEO_ID);
            uniqueId = extras.getString(MoEHelperConstants.GCM_EXTRA_UNIQUE_ID);
          }
          try {
            JSONObject newJSON = new JSONObject();
            String campaignId = extras.getString(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID);
            if (campaignId != null && campaignId.contains("DTSDK")){
              campaignId = campaignId.substring(0, campaignId.indexOf("DTSDK"));
              extras.putString(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID, campaignId);
            }
            newJSON.put(MoEHelperConstants.GCM_EXTRA_CAMPAIGN_ID, campaignId);
            if (geoId != null) newJSON.put(MoEHelperConstants.GCM_EXTRA_GEO_ID, geoId);
            if (uniqueId != null) newJSON.put(MoEHelperConstants.GCM_EXTRA_UNIQUE_ID, uniqueId);
            if (extras.containsKey("action_id")) {
              newJSON.put("gcm_action_id", extras.getString("action_id"));
            }
            if (extras.containsKey("received_from")) {
              newJSON.put("source", extras.getString("received_from"));
              extras.remove("received_from");
            }
            if (extras.containsKey("from_appOpen")) {
              newJSON.put("from_appOpen", extras.getBoolean("from_appOpen"));
              extras.remove("from_appOpen");
            }
            if (extras.containsKey("shownOffline")){
              newJSON.put("shownOffline", true);
              extras.remove("shownOffline");
            }
            if (extras.containsKey(PushConstants.ATTR_PUSH_PROVIDER)) {
              newJSON.put(PushConstants.ATTR_PUSH_PROVIDER,
                  extras.getString(PushConstants.ATTR_PUSH_PROVIDER));
            }
            if (extras.containsKey(PushConstants.ATTR_CAMPAIGN_ATTRIBUTES)) {
              JSONObject campaignAttributes =
                  new JSONObject(extras.getString(PushConstants.ATTR_CAMPAIGN_ATTRIBUTES));
              if (campaignAttributes != null) {
                Iterator keys = campaignAttributes.keys();
                while (keys.hasNext()) {
                  String key = (String) keys.next();
                  String value = campaignAttributes.getString(key);
                  newJSON.put(key, value);
                }
              }
              intent.removeExtra(PushConstants.ATTR_CAMPAIGN_ATTRIBUTES);
            }
            // track notification clicked
            MoEEventManager.getInstance(context)
                .trackEvent(MoEHelperConstants.EVENT_NOTIFICATION_CLICKED, newJSON);

            // update record for notification clicked for the same
            // to reflect in the inbox
            if (extras.containsKey(EXTRA_MSG_RECEIVED_TIME)) {
              MoEHelper.getInstance(context)
                  .trackNotificationClickedByTime(extras.getLong(EXTRA_MSG_RECEIVED_TIME));
            }
            // Removing extras for not calling this block again
            intent.removeExtra(NOTIFICATION_RECEIVED_MOE);
          } catch (JSONException e) {
            Logger.f("PushMessageListener:logNotificationClicked", e);
          }
        }
      }
    } catch (Exception e) {
      Logger.f("PushMessageListener:logNotificationClicked", e);
    }
  }
}