/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.pushamp.internal.repository.remote;

import androidx.annotation.Nullable;
import com.moengage.core.Logger;
import com.moengage.core.rest.Response;
import com.moengage.core.utils.ApiUtility;
import com.moengage.pushamp.internal.PushAmpConstants;
import com.moengage.pushamp.internal.repository.models.PushAmpSyncResponse;
import com.moengage.pushbase.PushConstants;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * API response parser for Push Amp.
 *
 * @author Umang Chamaria
 * Date: 10/04/19
 */
class ResponseParser {
  private static final String TAG = PushAmpConstants.MODULE_TAG + "ResponseParser";

  PushAmpSyncResponse parseSyncResponse(@Nullable Response response) {
    try {
      if (!ApiUtility.isSuccessfulResponse(response)) {
        Logger.v(TAG + " parseSyncResponse() : Sync Failed.");
        if (response != null) {
          Logger.e(TAG + " parseSyncResponse() : Error Message: " + response.errorMessage);
        }
        return new PushAmpSyncResponse(false);
      }
      if (response.responseBody == null) {
        Logger.v(TAG + " parseSyncResponse() :  No response campaigns to show.");
        return new PushAmpSyncResponse(true);
      }
      JSONObject responseJson = new JSONObject(response.responseBody);
      boolean isAppOpen = responseJson.optBoolean(RESPONSE_ATTR_ON_APP_OPEN, false);

      if (!responseJson.has(RESPONSE_ATTR_MESSAGE_INFO)) return new PushAmpSyncResponse(true);

      JSONArray payloadArray = responseJson.getJSONArray(RESPONSE_ATTR_MESSAGE_INFO);

      List<Map<String, String>> payloadList = new ArrayList<>();
      for (int i = 0; i < payloadArray.length(); i++) {
        JSONObject jsonPayload = payloadArray.getJSONObject(i);
        Map<String, String> payloadMap = jsonPayloadToMap(jsonPayload, isAppOpen);
        if (payloadMap != null) {
          payloadList.add(payloadMap);
        }
      }
      return new PushAmpSyncResponse(true, payloadList);
    } catch (Exception e) {
      Logger.e(TAG + " parseSyncResponse() : Exception ", e);
    }
    return new PushAmpSyncResponse(false);
  }

  @Nullable private Map<String, String> jsonPayloadToMap(JSONObject payloadJson, boolean isAppOpen) {
    try {
      if (!payloadJson.has(RESPONSE_ATTR_RESPONSE_DATA)) return null;
      payloadJson = payloadJson.getJSONObject(RESPONSE_ATTR_RESPONSE_DATA);
      Map<String, String> payloadMap = jsonToMap(payloadJson);
      if (payloadMap == null) return null;
      addPushSource(payloadMap);
      addSyncSource(payloadMap, isAppOpen);
      return payloadMap;
    } catch (JSONException e) {
      Logger.e(TAG + " jsonPayloadToMap() : Exception ", e);
    }
    return null;
  }

  private Map<String, String> jsonToMap(JSONObject payloadJson) throws JSONException {
    Map<String, String> map = new HashMap<>(payloadJson.length());
    Iterator<String> iterator = payloadJson.keys();
    while (iterator.hasNext()) {
      String key = (String) iterator.next();
      String value = payloadJson.getString(key);
      map.put(key, value);
    }
    return map;
  }

  private void addPushSource(Map<String, String> payloadMap) {
    payloadMap.put(PushConstants.ATTRIBUTE_MOE_PUSH_SOURCE,
        PushConstants.ATTRIBUTE_VALUE_RECEIVED_FROM);
  }

  private void addSyncSource(Map<String, String> payloadMap, boolean isAppOpen) {
    payloadMap.put(PushConstants.ATTRIBUTE_FROM_APP_OPEN, Boolean.toString(isAppOpen));
  }

  private static String RESPONSE_ATTR_MESSAGE_INFO = "messagesInfo";

  private static String RESPONSE_ATTR_ON_APP_OPEN = "on_app_open";

  private static String RESPONSE_ATTR_RESPONSE_DATA = "data";
}
