/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.pushamp.internal;

import android.Manifest;
import android.annotation.SuppressLint;
import android.annotation.TargetApi;
import android.app.AlarmManager;
import android.app.PendingIntent;
import android.app.job.JobInfo;
import android.app.job.JobInfo.Builder;
import android.app.job.JobScheduler;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.os.Build;
import android.os.Build.VERSION_CODES;
import androidx.annotation.Nullable;
import androidx.annotation.WorkerThread;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.core.RemoteConfig;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskManager;
import com.moengage.core.model.MoEJobParameters;
import com.moengage.push.PushHandler;
import com.moengage.push.PushManager;
import com.moengage.pushamp.internal.repository.PushAmpRepository;
import com.moengage.pushamp.internal.repository.models.PushAmpSyncRequest;
import com.moengage.pushamp.internal.repository.models.PushAmpSyncResponse;
import com.moengage.pushamp.internal.repository.remote.PushAmpServerSyncTask;
import com.moengage.pushbase.MoEPushHelper;
import java.util.List;
import java.util.Map;

/**
 * Central control point for Push-Amp.
 *
 * @author Umang Chamaria
 * Date: 10/04/19
 */
public class PushAmpController {

  private static final String TAG = PushAmpConstants.MODULE_TAG + "PushAmpController";

  public final PushAmpRepository repository;

  public boolean hasSynced = false;

  PushAmpController(PushAmpRepository repository) {
    this.repository = repository;
  }

  public void showPush(Context context, @Nullable List<Map<String, String>> payloadList) {
    Logger.v(TAG + " showPush() : Push Amp synced. Will try to show messages.");
    if (payloadList == null) {
      Logger.v(TAG + " showPush(): No push messages to be shown");
      return;
    }
    for (Map<String, String> payload : payloadList) {
      MoEPushHelper.getInstance().handlePushPayload(context, payload);
    }
  }

  /**
   * Schedule a server sync.
   *
   * @param context instance of {@link Context}
   */
  void scheduleServerSync(Context context) {
    Logger.v(TAG + " scheduleServerSync() : Will schedule server sync.");
    if (!shouldSync()) return;
    if (Build.VERSION.SDK_INT >= VERSION_CODES.LOLLIPOP) {
      scheduleSyncJob(context, repository.getMinimumSyncDelay());
    } else {
      scheduleSyncAlarm(context, repository.getMinimumSyncDelay());
    }
  }

  /**
   * Schedule an alarm for a background server sync.
   *
   * @param context instance of {@link Context}
   * @param minimumDelayDuration minimum time after which alarm should be triggered.
   */
  private void scheduleSyncAlarm(Context context, long minimumDelayDuration) {
    Logger.v(TAG + " scheduleSyncAlarm() : Scheduling sync alarm");
    Intent alarmIntent = new Intent(context, PushAmpAlarmReceiver.class);
    alarmIntent.setAction(PushAmpConstants.ACTION_SYNC_MESSAGES);
    PendingIntent pendingIntent =
        PendingIntent.getBroadcast(context, PushAmpConstants.PUSH_AMP_SYNC_ALARM_ID,
            alarmIntent, PendingIntent.FLAG_UPDATE_CURRENT);
    AlarmManager alarmManager = (AlarmManager) context.getSystemService(Context.ALARM_SERVICE);
    if (alarmManager != null) {
      alarmManager.set(AlarmManager.RTC_WAKEUP, MoEUtils.currentMillis() + minimumDelayDuration,
          pendingIntent);
    }
  }

  /**
   * Schedule a Job for a background server sync.
   *
   * @param context instance of {@link Context}
   * @param minimumDelayDuration minimum time after which alarm should be triggered.
   */
  @SuppressLint("MissingPermission") @TargetApi(VERSION_CODES.LOLLIPOP)
  private void scheduleSyncJob(Context context, long minimumDelayDuration) {
    Logger.v(TAG + " scheduleSyncJob() : scheduling sync job");
    ComponentName serviceComponent = new ComponentName(context, PushAmpSyncJob.class);
    Builder builder = new Builder(PushAmpConstants.PUSH_AMP_SYNC_JOB_ID, serviceComponent);
    builder.setOverrideDeadline(
        MoEUtils.currentMillis() + minimumDelayDuration + PushAmpConstants.MAX_OVERRIDE_DELAY);
    builder.setMinimumLatency(minimumDelayDuration);
    builder.setRequiredNetworkType(JobInfo.NETWORK_TYPE_ANY);
    if (MoEHelperUtils.hasPermission(context, Manifest.permission.RECEIVE_BOOT_COMPLETED)) {
      builder.setPersisted(true);
    }
    JobScheduler jobScheduler =
        (JobScheduler) context.getSystemService(Context.JOB_SCHEDULER_SERVICE);
    if (jobScheduler != null) {
      jobScheduler.schedule(builder.build());
    }
  }

  /**
   * Starts an {@link SDKTask} to fetch messages from the server.
   */
  private void fetchCampaignsFromServer(Context context, boolean fromAppOpen,
      MoEJobParameters jobParameters) {
    TaskManager.getInstance()
        .startTask(new PushAmpServerSyncTask(context, fromAppOpen, jobParameters));
  }

  @WorkerThread
  public void fetchAndShowCampaigns(Context context, PushAmpSyncRequest request) {
    PushAmpSyncResponse pushAmpSyncResponse = repository.fetchCampaignsFromServer(request);
    hasSynced = pushAmpSyncResponse.isSuccessful;
    if (pushAmpSyncResponse.isSuccessful && pushAmpSyncResponse.campaignList != null) {
      showPush(context, pushAmpSyncResponse.campaignList);
    }
  }

  /**
   * Initiate a server sync whenever app comes to foreground.
   *
   * @param context instance of {@link Context}
   */
  void syncOnAppForeground(Context context) {
    Logger.v(TAG + " syncOnAppForeground() : App came to foreground. Will try to fetch push-amp "
        + "messages if required.");
    if (hasSynced && (repository.getLastSyncTime() + PushAmpConstants.MINIMUM_SYNC_DELAY > MoEUtils
        .currentMillis())) {
      Logger.d(TAG + " syncOnAppForeground() : Push Amp API had synced recently, will not sync "
          + "again.");
      return;
    }
    Logger.v(TAG + " syncOnAppForeground() : Fetching campaigns from Push-Amp.");
    fetchCampaignsFromServer(context, true, null);
  }

  /**
   * Fetches messages from server and schedules the next sync.
   *
   * @param context instance of {@link Context}
   * @param jobParameters {@link JobScheduler} parameters.
   */
  void backgroundSyncAndSchedule(Context context, @Nullable MoEJobParameters jobParameters) {
    fetchCampaignsFromServer(context, false, jobParameters);
    scheduleServerSync(context);
  }

  /**
   * Checks whether push amp api should sync or not.
   *
   * @return true if push amp api should sync else false.
   */
  public boolean shouldSync() {
    if (repository.isPushNotificationOptedOut()) {
      Logger.i(TAG + " shouldSync() : Push notifications are opted out, disabling push-amp.");
      return false;
    }
    RemoteConfig remoteConfig = repository.getRemoteConfig();
    if (!remoteConfig.isAppEnabled) {
      Logger.i(TAG + " shouldSync() : App is disabled, disabling push-amp.");
      return false;
    }
    if (!remoteConfig.isPushAmpEnabled) {
      Logger.i(TAG + " shouldSync() : Push amp is disabled.");
      return false;
    }
    return true;
  }
}
