package com.moengage.core.inapp;

import android.app.Activity;
import android.content.Context;
import android.os.Bundle;
import androidx.annotation.Nullable;
import com.moe.pushlibrary.models.Event;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import com.moengage.core.model.InAppV2Meta;
import com.moengage.core.model.InAppV3Meta;
import com.moengage.core.RemoteConfig;

/**
 * @author Umang Chamaria
 */
public class InAppManager {

  private InAppHandler inAppHandler;

  private static InAppManager instance;

  private InAppManager() {
    loadInAppHandler();
  }

  private void loadInAppHandler() {
    try {
      Class handler = Class.forName("com.moengage.inapp.InAppHandlerImpl");
      inAppHandler = (InAppHandler) handler.newInstance();
      Logger.v("InAppManager:loadInAppHandler InApp Module present");
    } catch (Exception e) {
      Logger.e("InAppManager : loadInAppHandler : InApp Module not present ");
    }
  }

  public static InAppManager getInstance() {
    if (instance == null) {
      synchronized (InAppManager.class) {
        if (instance == null) instance = new InAppManager();
      }
    }
    return instance;
  }

  public void showInAppIfRequired(Context context) {
    if (isInAppModuleActive(context)) {
      inAppHandler.showInAppIfRequired(context);
    }
  }

  public void registerInAppManager(Activity activity) {
    if (isInAppModuleActive(activity.getApplicationContext())) {
      inAppHandler.registerInAppManager(activity);
    }
  }

  public void unregisterInAppManager(Activity activity) {
    if (isInAppModuleActive(activity.getApplicationContext())) {
      inAppHandler.unregisterInAppManager(activity);
    }
  }

  public void showTriggerInAppIfPossible(Context context, Event action) {
    if (isInAppModuleActive(context)) {
      inAppHandler.showTriggerInAppIfPossible(context, action);
    }
  }

  public void onLogout(Context context) {
    if (inAppHandler != null) {
      inAppHandler.onLogout(context);
    }
  }

  public void onAppClose(Context context){
    if (inAppHandler != null){
      inAppHandler.onAppClose(context);
    }
  }

  private boolean isInAppModuleActive(Context context) {
    RemoteConfig config = RemoteConfig.getConfig();
      return inAppHandler != null
          && !ConfigurationProvider.getInstance(context).isInAppOptedOut()
          && config.isInAppEnabled
          && config.isAppEnabled;
  }

  public void syncInAppsIfRequired(Context context) {
    if (isInAppModuleActive(context)) {
      inAppHandler.syncInAppIfRequired(context);
    }
  }

  public void showInAppFromPush(Context context, Bundle pushPayload){
    if (isInAppModuleActive(context)){
      inAppHandler.showInAppFromPush(context, pushPayload);
    }
  }

  @Nullable
  public InAppV3Meta generateMetaForV2Campaign(InAppV2Meta inAppV2Meta){
    if (inAppHandler != null){
      return inAppHandler.generateMetaForV2Campaign(inAppV2Meta);
    }
    return null;
  }

}