/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.events;

import android.content.Context;
import androidx.annotation.NonNull;
import androidx.annotation.RestrictTo;
import androidx.annotation.RestrictTo.Scope;
import androidx.annotation.WorkerThread;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.PayloadBuilder;
import com.moe.pushlibrary.models.Event;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEDAO;
import com.moengage.core.RemoteConfig;
import com.moengage.core.executor.TaskManager;
import java.util.Set;
import org.json.JSONObject;

/**
 * Central point for all event related processing, i.e. tracking, blacklisting, trigger-events
 *
 * @author Umang Chamaria
 */

public class MoEEventManager {

  private static final String TAG = MoEConstants.MODULE_TAG + "MoEEventManager";

  private static MoEEventManager instance = null;

  private Context context;

  private int eventCounter = 0;

  EventHandler eventHandler;

  private MoEEventManager(Context context) {
    this.context = context;
    eventHandler = new EventHandler();
  }

  public static MoEEventManager getInstance(Context context) {
    if (instance == null) {
      synchronized (MoEEventManager.class) {
        if (instance == null) instance = new MoEEventManager(context);
      }
    }
    return instance;
  }

  int getEventCounter() {
    return eventCounter;
  }

  void incrementEventCounter() {
    eventCounter++;
  }

  public void setEventCounter(int counter) {
    eventCounter = counter;
  }

  public void trackEvent(String action, JSONObject attributes) {
    trackEvent(eventHandler.transformToEventModel(action, attributes));
  }

  public void trackEvent(String action, PayloadBuilder attributes){
    trackEvent(new Event(action, attributes.build()));
  }

  public void trackEvent(Event event) {
    if (!RemoteConfig.getConfig().isAppEnabled) return;
    TaskManager.getInstance().addTaskToQueue(new TrackEventTask(context, event));
  }



  private boolean isFlushEvent(String eventName) {
    Set<String> flushEvents =
        RemoteConfig.getConfig().flushEvents;
    return flushEvents!= null && flushEvents.contains(eventName);
  }

  void flushIfRequired(@NonNull Event event) {
    if (event.eventName != null && isFlushEvent(event.eventName)) {
      Logger.v(TAG + " flushIfRequired() flush event : " + event.eventName);
      MoEHelper.getInstance(context).syncInteractionDataNow();
    }
  }

  @WorkerThread @RestrictTo(Scope.LIBRARY_GROUP)
  public void writeDataPointToStorage(Event event) {
    MoEDAO.getInstance(context).addEvent(event);
  }

}
