/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import android.content.Context;
import androidx.annotation.NonNull;
import com.moe.pushlibrary.models.Event;
import com.moe.pushlibrary.models.UserAttribute;
import com.moengage.core.events.MoEEventManager;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import org.json.JSONObject;

/**
 * Track device attributes. This intentionally uses the user attribute table since there are very
 * device attributes as of now.
 *
 * @author Umang Chamaria
 */

class SetDeviceAttributeTask extends SDKTask{
  private static final String TAG = MoEConstants.MODULE_TAG + "SetDeviceAttributeTask";
  private JSONObject deviceAttribute;

  SetDeviceAttributeTask(@NonNull Context context, @NonNull JSONObject deviceAttribute){
    super(context);
    this.deviceAttribute = deviceAttribute;
  }

  @Override public TaskResult execute(){
    try {
      Logger.v("SetDeviceAttributeTask: executing Task");
      UserAttribute currentUserAttribute = MoEUtils.getUserAttributePoJo(deviceAttribute);
      UserAttribute savedUserAttribute = null;
      if (currentUserAttribute != null){
        savedUserAttribute =
            MoEUtils.getSavedUserAttribute(context, currentUserAttribute.userAttributeName);
      }
      if (MoEUtils.shouldSendUserAttribute(currentUserAttribute, savedUserAttribute)){
        Logger.d(
            "SetDeviceAttributeTask : setUserAttribute User attribute not yet sent to server will "
                + "send : "
                + deviceAttribute.toString());
        Event event = new Event(MoEConstants.EVENT_ACTION_DEVICE_ATTRIBUTE, deviceAttribute);
        MoEEventManager.getInstance(context).writeDataPointToStorage(event);
        MoEDAO.getInstance(context).addOrUpdateUserAttribute(currentUserAttribute);
        taskResult.setIsSuccess(true);
      } else{
        Logger.d(
            "SetDeviceAttributeTask : setUserAttributes already sent once, need not send duplicate "
                + "attribute : "
                + deviceAttribute.toString());
        taskResult.setIsSuccess(false);
      }
      Logger.v("SetDeviceAttributeTask: completed Task");
    } catch (Exception e) {
      Logger.e( TAG + " execute() : Exception: ", e);
    }
    return taskResult;
  }

  @Override public String getTaskTag(){
    return TAG_SET_DEVICE_ATTRIBUTES;
  }

  @Override public boolean isSynchronous(){
    return false;
  }
}
