/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import com.moengage.core.utils.ApiUtility;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */
public class RemoteConfig {
  /**
   * If true logentry is enabled for the account.
   */
  public boolean isLogEntryEnabled;
  /**
   * Token to send data to logentry.
   */
  @Nullable public String logEntryToken;
  /**
   * Time interval after which push-amp api is synced when the app is in background.
   */
  public long pushAmpSyncDelay;
  /**
   * Events which shouldn't be sent to server even if tracked by the client.
   */
  @Nullable public Set<String> blacklistedEvents;
  /**
   * If true account is enabled.
   */
  public boolean isAppEnabled;
  /**
   * If true in-app is enabled for the account.
   */
  public boolean isInAppEnabled;
  /**
   * If true geofence is enabled for the account.
   */
  public boolean isGeofenceEnabled;
  /**
   * If true push-amp is enabled for the account.
   */
  public boolean isPushAmpEnabled;
  /**
   * Event count threshold for syncing data.
   */
  public int eventBatchCount;
  /**
   * Time interval after which sdk tries to sync data when the app is in background.
   */
  public long dataSyncRetryInterval;
  /**
   * Set of events which should be synced immediately.
   */
  public Set<String> flushEvents;
  /**
   * If true sdk will try to sync data periodically when the app is in foreground.
   */
  public boolean isPeriodicFlushEnabled;
  /**
   * Time interval after which sdk attempts to sync data.
   */
  public long periodicFlushTime;
  /**
   * Time duration after which campaign-ids cached for push-amp should be removed.
   */
  public long pushAmpCampaignExpiryTime;
  /**
   * If true real-time trigger is enabled for the account.
   */
  public boolean isRealTimeTriggerEnabled;
  /**
   * Time duration after which real time trigger is synced after app close.
   */
  public long realTimeTriggerBackgroundSyncInterval;
  /**
   * Events to be tracked even if user has opted out of data tracking.
   */
  public Set<String> gdprWhitelistEventList;
  /**
   * Time duration before which same user attribute value shouldn't be sent to server.
   */
  public long userAttributeCachingTime;
  /**
   * USER_ATTRIBUTE_UNIQUE_ID matching this regex should not be tracked.
   */
  @Nullable public List<String> blockedUniqueIdRegex;
  /**
   * Time duration after which session is marked as in-active.
   */
  public long sessionInActiveTime;
  /**
   * Additional keys to identify traffic source.
   */
  public Set<String> additionalSourceIdentifiers;
  /**
   * If true push-amp+ is enabled for the account.
   */
  public boolean isPushAmpPlusEnabled;
  /**
   * Secret key
   */
  String encryptionKey;

  public RemoteConfig() {
    isLogEntryEnabled = RemoteConfigDefault.LOG_ENTRY_STATUS;
    logEntryToken = RemoteConfigDefault.LOG_ENTRY_TOKEN;
    pushAmpSyncDelay = RemoteConfigDefault.PUSH_AMP_SYNC_INTERVAL;
    isAppEnabled = RemoteConfigDefault.ACCOUNT_STATUS;
    isInAppEnabled = RemoteConfigDefault.IN_APP_STATUS;
    isGeofenceEnabled = RemoteConfigDefault.GEO_FENCE_STATUS;
    isPushAmpEnabled = RemoteConfigDefault.PUSH_AMP_STATUS;
    eventBatchCount = RemoteConfigDefault.EVENT_BATCH_COUNT;
    dataSyncRetryInterval = RemoteConfigDefault.DATA_SYNC_RETRY_INTERVAL;
    isPeriodicFlushEnabled = RemoteConfigDefault.PERIODIC_FLUSH_STATE;
    periodicFlushTime = RemoteConfigDefault.PERIODIC_FLUSH_TIME;
    pushAmpCampaignExpiryTime = RemoteConfigDefault.PUSH_AMP_CAMPAIGN_EXPIRY_TIME;
    isRealTimeTriggerEnabled = RemoteConfigDefault.REAL_TIME_TRIGGER_STATUS;
    realTimeTriggerBackgroundSyncInterval = RemoteConfigDefault.REAL_TIME_TRIGGER_SYNC_INTERVAL;
    userAttributeCachingTime = RemoteConfigDefault.USER_ATTRIBUTE_CACHING_TIME;
    sessionInActiveTime = RemoteConfigDefault.DEFAULT_SESSION_INACTIVE_TIME;
    additionalSourceIdentifiers = new HashSet<>();
    flushEvents = new HashSet<>();
    flushEvents.addAll(RemoteConfigDefault.DEFAULT_FLUSH_EVENTS);
    gdprWhitelistEventList = new HashSet<>();
    gdprWhitelistEventList.addAll(RemoteConfigDefault.DEFAULT_GDPR_WHITELIST_EVENTS);
    isPushAmpPlusEnabled = RemoteConfigDefault.MI_PUSH_APP_STATUS;
    encryptionKey = RemoteConfigDefault.DEFAULT_DATA_ENCRYPTION_KEY;
  }

  @Override public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    RemoteConfig that = (RemoteConfig) o;

    if (isLogEntryEnabled != that.isLogEntryEnabled) return false;
    if (pushAmpSyncDelay != that.pushAmpSyncDelay) return false;
    if (isAppEnabled != that.isAppEnabled) return false;
    if (isInAppEnabled != that.isInAppEnabled) return false;
    if (isGeofenceEnabled != that.isGeofenceEnabled) return false;
    if (isPushAmpEnabled != that.isPushAmpEnabled) return false;
    if (eventBatchCount != that.eventBatchCount) return false;
    if (dataSyncRetryInterval != that.dataSyncRetryInterval) return false;
    if (isPeriodicFlushEnabled != that.isPeriodicFlushEnabled) return false;
    if (periodicFlushTime != that.periodicFlushTime) return false;
    if (pushAmpCampaignExpiryTime != that.pushAmpCampaignExpiryTime) return false;
    if (isRealTimeTriggerEnabled != that.isRealTimeTriggerEnabled) return false;
    if (realTimeTriggerBackgroundSyncInterval != that.realTimeTriggerBackgroundSyncInterval) return false;
    if (userAttributeCachingTime != that.userAttributeCachingTime) return false;
    if (sessionInActiveTime != that.sessionInActiveTime) return false;
    if (logEntryToken != null ? !logEntryToken.equals(that.logEntryToken)
        : that.logEntryToken != null) {
      return false;
    }
    if (blacklistedEvents != null ? !blacklistedEvents.equals(that.blacklistedEvents)
        : that.blacklistedEvents != null) {
      return false;
    }
    if (flushEvents != null ? !flushEvents.equals(that.flushEvents)
        : that.flushEvents != null) {
      return false;
    }
    if (gdprWhitelistEventList != null ? !gdprWhitelistEventList.equals(that.gdprWhitelistEventList)
        : that.gdprWhitelistEventList != null) {
      return false;
    }
    return blockedUniqueIdRegex != null ? blockedUniqueIdRegex.equals(that.blockedUniqueIdRegex)
        : that.blockedUniqueIdRegex == null;
  }

  private static final String TAG = "RemoteConfig";

  public static RemoteConfig fromJson(JSONObject configurationJson) {
    try {
      RemoteConfig configuration = new RemoteConfig();
      if (configurationJson.has(RESPONSE_ATTR_LOG_ENTRY_STATUS)) {
        configuration.isLogEntryEnabled =
            getStateFromResponse(configurationJson, RESPONSE_ATTR_LOG_ENTRY_STATUS,
                RemoteConfigDefault.LOG_ENTRY_STATUS);
      }
      if (configurationJson.has(RESPONSE_ATTR_LOG_ENTRY_TOKEN)) {
        configuration.logEntryToken = configurationJson.getString(RESPONSE_ATTR_LOG_ENTRY_TOKEN);
      }
      if (configurationJson.has(RESPONSE_ATTR_PUSH_AMP_SYNC_DELAY)) {
        long syncDelay = configurationJson.getLong(RESPONSE_ATTR_PUSH_AMP_SYNC_DELAY);
        if (syncDelay > 0) {
          configuration.pushAmpSyncDelay = syncDelay * 1000;
        }
      }
      if (configurationJson.has(RESPONSE_ATTR_BLACKLIST_EVENT)) {
        configuration.blacklistedEvents = ApiUtility.jsonArrayToStringSet(
            configurationJson.getJSONArray(RESPONSE_ATTR_BLACKLIST_EVENT));
      }
      if (configurationJson.has(RESPONSE_ATTR_APP_STATE)) {
        configuration.isAppEnabled = getStateFromResponse(configurationJson,
            RESPONSE_ATTR_APP_STATE,
            RemoteConfigDefault.ACCOUNT_STATUS);
      }
      if (configurationJson.has(RESPONSE_ATTR_IN_APP_STATE)) {
        configuration.isInAppEnabled =
            getStateFromResponse(configurationJson, RESPONSE_ATTR_IN_APP_STATE,
                RemoteConfigDefault.IN_APP_STATUS);
      }
      if (configurationJson.has(RESPONSE_ATTR_GEO_STATE)) {
        configuration.isGeofenceEnabled =
            getStateFromResponse(configurationJson, RESPONSE_ATTR_GEO_STATE,
                RemoteConfigDefault.GEO_FENCE_STATUS);
      }
      if (configurationJson.has(RESPONSE_ATTR_PUSH_AMP_STATE)) {
        configuration.isPushAmpEnabled =
            getStateFromResponse(configurationJson, RESPONSE_ATTR_PUSH_AMP_STATE,
                RemoteConfigDefault.PUSH_AMP_STATUS);
      }
      if (configurationJson.has(RESPONSE_ATTR_EVENT_SYNC_COUNT)) {
        configuration.eventBatchCount = configurationJson.getInt(RESPONSE_ATTR_EVENT_SYNC_COUNT);
      }
      if (configurationJson.has(RESPONSE_ATTR_DATA_SYNC_RETRY_INTERVAL)) {
        configuration.dataSyncRetryInterval =
            configurationJson.getLong(RESPONSE_ATTR_DATA_SYNC_RETRY_INTERVAL) * 1000;
      }
      if (configurationJson.has(RESPONSE_ATTR_FLUSH_EVENTS)) {
        Set<String> events = ApiUtility.jsonArrayToStringSet(
            configurationJson.getJSONArray(RESPONSE_ATTR_FLUSH_EVENTS));
        if (events != null) {
          configuration.flushEvents.addAll(events);
        }
      }
      if (configurationJson.has(RESPONSE_ATTR_PERIODIC_FLUSH_STATE)) {
        configuration.isPeriodicFlushEnabled =
            getStateFromResponse(configurationJson, RESPONSE_ATTR_PERIODIC_FLUSH_STATE,
                RemoteConfigDefault.PERIODIC_FLUSH_STATE);
      }
      if (configurationJson.has(RESPONSE_ATTR_PERIODIC_FLUSH_TIME)) {
        configuration.periodicFlushTime =
            configurationJson.getLong(RESPONSE_ATTR_PERIODIC_FLUSH_TIME);
      }
      if (configurationJson.has(RESPONSE_ATTR_CAMPAIGN_ID_EXPIRY)) {
        configuration.pushAmpCampaignExpiryTime =
            configurationJson.getLong(RESPONSE_ATTR_CAMPAIGN_ID_EXPIRY);
      }
      if (configurationJson.has(RESPONSE_ATTR_REAL_TIME_TRIGGER_STATE)) {
        configuration.isRealTimeTriggerEnabled =
            getStateFromResponse(configurationJson, RESPONSE_ATTR_REAL_TIME_TRIGGER_STATE,
                RemoteConfigDefault.REAL_TIME_TRIGGER_STATUS);
      }
      if (configurationJson.has(RESPONSE_ATTR_REAL_TIME_TRIGGER_SYNC_TIME)) {
        configuration.realTimeTriggerBackgroundSyncInterval =
            configurationJson.getLong(RESPONSE_ATTR_REAL_TIME_TRIGGER_SYNC_TIME) * 1000;
      }
      if (configurationJson.has(RESPONSE_ATTR_GDPR_WHITELIST_EVENTS)) {
        Set<String> whitelistedEvents = ApiUtility.jsonArrayToStringSet(
            configurationJson.getJSONArray(RESPONSE_ATTR_GDPR_WHITELIST_EVENTS));
        if (whitelistedEvents != null) {
          configuration.gdprWhitelistEventList.addAll(whitelistedEvents);
        }
      }

      if (configurationJson.has(RESPONSE_ATTR_USER_ATTRIBUTE_CACHING_TIME)) {
        configuration.userAttributeCachingTime =
            configurationJson.getLong(RESPONSE_ATTR_USER_ATTRIBUTE_CACHING_TIME) * 1000;
      }
      if (configurationJson.has(RESPONSE_ATTR_BLOCKED_UNIQUE_ID_REGEX)) {
        configuration.blockedUniqueIdRegex = ApiUtility.jsonArrayToStringList(
            configurationJson.getJSONArray(RESPONSE_ATTR_BLOCKED_UNIQUE_ID_REGEX));
      }
      if (configurationJson.has(RESPONSE_ATTR_SESSION_INACTIVE_TIME)) {
        configuration.sessionInActiveTime = configurationJson.getLong(
            RESPONSE_ATTR_SESSION_INACTIVE_TIME) * 1000;
      }
      if (configurationJson.has(RESPONSE_ATTR_TRAFFIC_SOURCE_EXTRA_IDENTIFIERS)) {
        Set<String> identifiers = ApiUtility.jsonArrayToStringSet(
            configurationJson.getJSONArray(RESPONSE_ATTR_TRAFFIC_SOURCE_EXTRA_IDENTIFIERS));
        if (identifiers != null) {
          configuration.additionalSourceIdentifiers = identifiers;
        }
      }
      if (configurationJson.has(RESPONSE_ATTR_MI_PUSH_STATUS)) {
        configuration.isPushAmpPlusEnabled = getStateFromResponse(configurationJson,
            RESPONSE_ATTR_MI_PUSH_STATUS, RemoteConfigDefault.MI_PUSH_APP_STATUS);
      }
      if (configurationJson.has(RESPONSE_ATTR_DATA_ENCRYPTION_KEY)) {
        String key =
            configurationJson.getString(RESPONSE_ATTR_DATA_ENCRYPTION_KEY);
        if (MoEUtils.isEmptyString(key)) {
          key = RemoteConfigDefault.DEFAULT_DATA_ENCRYPTION_KEY;
        }
        configuration.encryptionKey = key;
      }
      return configuration;
    } catch (Exception e) {
      Logger.e(TAG + " parseConfigApiResponse() : Exception ", e);
    }
    return null;
  }

  private static boolean getStateFromResponse(@NonNull JSONObject responseJSON,
      @NonNull String responseAttr, boolean defaultValue) {
    try {
      String state = responseJSON.getString(responseAttr);
      switch (state) {
        case "blocked":
          return false;
        case "allowed":
          return true;
      }
    } catch (JSONException e) {
      Logger.e(" getStateFromResponse ", e);
    }
    return defaultValue;
  }

  private static final String RESPONSE_ATTR_LOG_ENTRY_STATUS = "le_s";

  private static final String RESPONSE_ATTR_LOG_ENTRY_TOKEN = "le_tkn";

  private static final String RESPONSE_ATTR_PUSH_AMP_SYNC_DELAY = "m_s_t";

  private static final String RESPONSE_ATTR_BLACKLIST_EVENT = "b_e";

  private static final String RESPONSE_ATTR_APP_STATE = "a_s";

  private static final String RESPONSE_ATTR_IN_APP_STATE = "i_s";

  private static final String RESPONSE_ATTR_GEO_STATE = "g_s";

  private static final String RESPONSE_ATTR_PUSH_AMP_STATE = "in_s";

  private static final String RESPONSE_ATTR_EVENT_SYNC_COUNT = "e_b_c";

  private static final String RESPONSE_ATTR_DATA_SYNC_RETRY_INTERVAL = "d_s_r_i";

  private static final String RESPONSE_ATTR_FLUSH_EVENTS = "f_e";

  private static final String RESPONSE_ATTR_PERIODIC_FLUSH_TIME = "p_f_t";

  private static final String RESPONSE_ATTR_PERIODIC_FLUSH_STATE = "p_f_s";

  private static final String RESPONSE_ATTR_CAMPAIGN_ID_EXPIRY = "cid_ex";

  private static final String RESPONSE_ATTR_REAL_TIME_TRIGGER_STATE = "d_t";

  private static final String RESPONSE_ATTR_REAL_TIME_TRIGGER_SYNC_TIME = "dt_s_t";

  private static final String RESPONSE_ATTR_GDPR_WHITELIST_EVENTS = "d_t_w_e";

  private static final String RESPONSE_ATTR_USER_ATTRIBUTE_CACHING_TIME = "u_a_c_t";

  private static final String RESPONSE_ATTR_BLOCKED_UNIQUE_ID_REGEX = "b_uid_r";

  private static final String RESPONSE_ATTR_SESSION_INACTIVE_TIME = "s_i_d";

  private static final String RESPONSE_ATTR_TRAFFIC_SOURCE_EXTRA_IDENTIFIERS = "src_ext";

  private static final String RESPONSE_ATTR_MI_PUSH_STATUS = "mi_p_s";

  private static final String RESPONSE_ATTR_DATA_ENCRYPTION_KEY = "d_e_k";

  private static RemoteConfig remoteConfig;

  public static RemoteConfig getConfig() {
    if (remoteConfig == null) {
      synchronized (RemoteConfig.class) {
        if (remoteConfig == null) remoteConfig = new RemoteConfig();
      }
    }
    return remoteConfig;
  }

  public static void setRemoteConfig(RemoteConfig config) {
    remoteConfig = config;
  }

  @Override public String toString() {
    return "{\n"
        +
        "\"isLogEntryEnabled\": "
        + isLogEntryEnabled
        + ",\n"
        +
        " \"logEntryToken\": \""
        + logEntryToken
        + "\" ,\n"
        +
        " \"pushAmpSyncDelay\": "
        + pushAmpSyncDelay
        + ",\n"
        +
        " \"blacklistedEvents\": "
        + blacklistedEvents
        + ",\n"
        +
        " \"isAppEnabled\": "
        + isAppEnabled
        + ",\n"
        +
        " \"isInAppEnabled\": "
        + isInAppEnabled
        + ",\n"
        +
        " \"isGeofenceEnabled\": "
        + isGeofenceEnabled
        + ",\n"
        +
        " \"isPushAmpEnabled\": "
        + isPushAmpEnabled
        + ",\n"
        +
        " \"eventBatchCount\": "
        + eventBatchCount
        + ",\n"
        +
        " \"dataSyncRetryInterval\": "
        + dataSyncRetryInterval
        + ",\n"
        +
        " \"flushEvents\": "
        + flushEvents
        + ",\n"
        +
        " \"isPeriodicFlushEnabled\": "
        + isPeriodicFlushEnabled
        + ",\n"
        +
        " \"periodicFlushTime\": "
        + periodicFlushTime
        + ",\n"
        +
        " \"pushAmpCampaignExpiryTime\": "
        + pushAmpCampaignExpiryTime
        + ",\n"
        +
        " \"isRealTimeTriggerEnabled\": "
        + isRealTimeTriggerEnabled
        + ",\n"
        +
        " \"realTimeTriggerBackgroundSyncInterval\": "
        + realTimeTriggerBackgroundSyncInterval
        + ",\n"
        +
        " \"gdprWhitelistEventList\": "
        + gdprWhitelistEventList
        + ",\n"
        +
        " \"userAttributeCachingTime\": "
        + userAttributeCachingTime
        + ",\n"
        +
        " \"blockedUniqueIdRegex\": "
        + blockedUniqueIdRegex
        + ",\n"
        +
        " \"sessionInActiveTime\": "
        + sessionInActiveTime
        + ",\n"
        +
        " \"additionalSourceIdentifiers\": "
        + additionalSourceIdentifiers
        + ",\n"
        +
        " \"isPushAmpPlusEnabled\": "
        + isPushAmpPlusEnabled
        + ",\n"
        +
        " \"encryptionKey\": \""
        + encryptionKey
        + "\" ,\n"
        +
        '}';
  }
}
