/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import android.app.Activity;
import android.content.Context;
import android.os.Bundle;
import androidx.annotation.NonNull;
import com.moe.pushlibrary.MoEHelper;
import com.moengage.core.executor.TaskManager;
import com.moengage.core.integrations.segment.TrackSegmentEventTask;
import com.moengage.core.integrations.segment.TrackSegmentUserAttributeTask;
import com.moengage.core.model.IntegrationMeta;
import com.moengage.core.model.IntegrationPartner;
import com.moengage.push.PushManager;
import java.util.Map;
import org.json.JSONObject;

/**
 * Internal Helper class for interaction between Hybrid and Partner SDKs
 *
 * @author Umang Chamaria
 * Date: 26/06/20
 */
public class MoEIntegrationHelper {

  private static final String TAG = MoEConstants.MODULE_TAG + "MoEIntegrationHelper";

  private IntegrationPartner partner;
  private Context context;

  public MoEIntegrationHelper(Context context, IntegrationPartner partner){
    this.partner = partner;
    this.context = context;
  }

  @SuppressWarnings("ConstantConditions")
  public static void setIntegrationMeta(@NonNull IntegrationMeta meta) {
    try {
      if (meta == null) return;
      Logger.v(TAG + " setIntegrationMeta() : Meta: " + meta);
      ConfigurationCache.getInstance().setIntegrationType(meta.integrationType);
      ConfigurationCache.getInstance().setIntegrationVersion(meta.integrationVersion);
    } catch (Exception e) {
      Logger.e(TAG + " setIntegrationMeta() : ", e);
    }
  }

  /**
   * API to track event via MoEngage-Segment Bundled Integration.<br>
   * <b>Note:</b> Do not call this method. This is only for internal usage.
   *
   * @param eventName Event Name
   * @param eventAttributes Event Attributes
   * @since 10.4.00
   */
  @SuppressWarnings("ConstantConditions")
  public void trackEvent(@NonNull String eventName, JSONObject eventAttributes) {
    try {
      if (context == null) return;
      Logger.v(TAG + " trackEventFromSegment() : Event Name: " + eventName);
      Logger.v(TAG + " trackEventFromSegment() : Attributes: " + eventAttributes);
      if (MoEUtils.isEmptyString(eventName)) {
        Logger.e(TAG + " trackEventFromSegment() : Event name cannot be empty");
        return;
      }
      if (partner == IntegrationPartner.SEGMENT) {
        TaskManager.getInstance()
            .addTaskToQueue(new TrackSegmentEventTask(context, eventName, eventAttributes));
      }
    } catch (Exception e) {
      Logger.e(TAG + " trackEvent() : ", e);
    }
  }

  /**
   * API to track user attributes via MoEngage-Segment Bundled Integration.<br>
   * <b>Note:</b> Do not call this method. This is only for internal usage.
   *
   * @param userAttributesMap attribute map.
   * @since 10.4.00
   */
  @SuppressWarnings("ConstantConditions")
  public void trackUserAttribute(Map<String, Object> userAttributesMap) {
    try {
      if (context == null || userAttributesMap == null) return;
      if (partner == IntegrationPartner.SEGMENT) {
        TaskManager.getInstance()
            .addTaskToQueue(new TrackSegmentUserAttributeTask(context, userAttributesMap));
      }
    } catch (Exception e) {
      Logger.e(TAG + " trackUserAttribute() : ", e);
    }
  }

  /**
   * Initialises the MoEngage SDK.
   *
   * @param appId App-id for the MoEngage Platform.
   * @param senderId Sender-id for the FCM project, optional
   * @since 10.4.00
   */
  public void initialize(String appId, String senderId) {
    try {
      Logger.v(TAG + " initialize() : App-id: " + appId);
      Logger.v(TAG + " initialize() : Sender-Id: " + senderId);
      MoEngage.setBuildStatus(MoEUtils.isDebugBuild(context));
      if (MoEUtils.isEmptyString(appId)) return;
      SdkConfig.getConfig().appId = MoEUtils.formatAppId(appId);
      if (!MoEUtils.isEmptyString(senderId)) SdkConfig.getConfig().pushConfig.senderId = senderId;
      //force registration for push
      if (SdkConfig.getConfig().pushConfig.isFcmPushRegistrationEnabled) {
        PushManager.getInstance().registerForPush(context);
      }
    } catch (Exception e) {
      Logger.e(TAG + " initialize() : ", e);
    }
  }

  /**
   * Track anonymous id for partner integration.
   *
   * @param anonymousId instance of {@link String}
   * @since 10.4.00
   */
  @SuppressWarnings("ConstantConditions")
  public void trackAnonymousId(String anonymousId) {
    try {
      if (MoEUtils.isEmptyString(anonymousId)) return;
      if (partner == IntegrationPartner.SEGMENT) {
        ConfigurationProvider.getInstance(context).saveSegmentAnonymousId(anonymousId);
      }
    } catch (Exception e) {
      Logger.e(TAG + " trackAnonymousId() : ", e);
    }
  }

  public void onActivityStart(Activity activity){
    try {
      MoEHelper.getInstance(context).onStartInternal(activity);
    } catch (Exception e) {
      Logger.e( TAG + " onActivityStart() : ", e);
    }
  }

  public void onActivityStop(Activity activity){
    try{
      MoEHelper.getInstance(context).onStopInternal(activity);
    }catch(Exception e){
      Logger.e(TAG + " onActivityStop() : ", e);
    }
  }

  public void onActivityResumed(Activity activity){
    try{
      MoEHelper.getInstance(context).onResumeInternal(activity);
    }catch(Exception e){
      Logger.e(TAG + " onActivityResumed() : ", e);
    }
  }

  public void onActivitySavedInstance(Activity activity, Bundle outState){
    try{
      MoEHelper.getInstance(context).onSaveInstanceState(outState);
    }catch(Exception e){
      Logger.e(TAG + " onActivitySavedInstance() : ", e);
    }
  }
}
