/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Bitmap.CompressFormat;
import androidx.annotation.NonNull;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * @author Umang Chamaria
 */
public class MoEFileManager {

  private static final String TAG = MoEConstants.MODULE_TAG + "MoEFileManager";

  private final String FILE_PATH_SEPARATOR = "/";

  private final String MOENGAGE_SUBFOLDER = "moengage";

  private Context context;

  public MoEFileManager(Context context) {
    this.context = context;
    createBaseFolderIfRequired();
  }

  public void saveImageFile(@NonNull String directoryName,
      @NonNull String fileName, @NonNull Bitmap bitmap) {
    if (!doesDirectoryExists(directoryName)) createDirectory(directoryName);
    FileOutputStream fileOutputStream = null;
    try {
      fileOutputStream = new FileOutputStream(basePath() +
          FILE_PATH_SEPARATOR + directoryName + FILE_PATH_SEPARATOR + fileName);
      bitmap.compress(CompressFormat.PNG, 100, fileOutputStream);
    } catch (Exception e) {
      Logger.e(TAG + " saveImageFile() : ", e);
    } finally {
      try {
        if (fileOutputStream != null) {
          fileOutputStream.close();
        }
      } catch (IOException e) {
        Logger.e(TAG + " saveImageFile() : ", e);
      }
    }
  }

  public boolean doesDirectoryExists(String directoryName) {
    try {
      File file =
          new File(basePath() + FILE_PATH_SEPARATOR + directoryName);
      return file.exists();
    } catch (Exception e) {
      Logger.e(TAG + " doesDirectoryExists() : Directory with name " + directoryName + " does "
          + "not exist.");
    }
    return false;
  }

  public boolean createDirectory(String directoryName) {
    File file =
        new File(basePath() + FILE_PATH_SEPARATOR + directoryName);
    return file.mkdir();
  }

  public void deleteFolder(String folder) {
    File directory = new File(basePath() + FILE_PATH_SEPARATOR + folder);
    deleteDirectory(directory);
  }

  public boolean fileExistsInDirectory(String directoryName, String fileName) {
    File file = new File(basePath() + FILE_PATH_SEPARATOR +
        directoryName + FILE_PATH_SEPARATOR + fileName);
    return file.exists();
  }

  public String getPathForFile(String directoryName, String fileName) {
    File file = new File(basePath() + FILE_PATH_SEPARATOR +
        directoryName + FILE_PATH_SEPARATOR + fileName);
    return file.getAbsolutePath();
  }

  @NonNull private String basePath() {
    return context.getFilesDir().getAbsolutePath() + FILE_PATH_SEPARATOR + MOENGAGE_SUBFOLDER;
  }

  private void createBaseFolderIfRequired() {
    File file = new File(basePath());
    if (!file.exists()) {
      Logger.v(TAG + " createBaseFolderIfRequired() : MoEngage base folder does not exist. Will "
          + "create it.");
      file.mkdir();
    } else {
      Logger.v(TAG + " createBaseFolderIfRequired() : MoEngage base folder exists. Need not "
          + "create another one.");
    }
  }

  public void clearMoEngageFiles() {
    File file = new File(basePath());
    if (!file.exists()) return;
    if (file.isDirectory()) {
      deleteDirectory(file);
    }
  }

  private void deleteDirectory(File directory) {
    File[] fileList = directory.listFiles();
    if (fileList == null) {
      directory.delete();
      return;
    }
    for (File file : fileList) {
      if (file.isDirectory()) {
        deleteDirectory(file);
      } else {
        file.delete();
      }
    }
    directory.delete();
  }

  public File saveGif(String directoryName, String fileName, InputStream gifStream) {
    try {
      if (!doesDirectoryExists(directoryName)) createDirectory(directoryName);
      int bufferLength = 0;
      byte[] buffer = new byte[1024];
      File file =
          new File(basePath() + FILE_PATH_SEPARATOR + directoryName + FILE_PATH_SEPARATOR + fileName);
      FileOutputStream fileOutput = new FileOutputStream(file);
      while ((bufferLength = gifStream.read(buffer)) > 0) {
        fileOutput.write(buffer, 0, bufferLength);
      }
      fileOutput.close();
      return file;
    } catch (Exception e) {
      Logger.e( TAG + " saveGif() : ", e);
    }
    return null;
  }

  public File getFileByName(String directory, String fileName){
    return new File(basePath() + FILE_PATH_SEPARATOR + directory, fileName);
  }
}