/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import android.content.Context;
import android.os.Build;
import android.os.Build.VERSION;
import android.text.TextUtils;
import android.util.DisplayMetrics;
import android.view.WindowManager;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import com.moengage.core.model.DevicePreferences;
import com.moengage.core.model.PushTokens;
import com.moengage.core.model.TokenState;
import com.moengage.core.rest.Response;
import com.moengage.core.utils.JsonBuilder;
import java.util.TimeZone;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Makes a device add call on fresh installation
 * Service constant MSG_DEVICE_ADD
 *
 * @author Umang Chamaria
 */
public class DeviceAddTask extends SDKTask {

  private static final String TAG = MoEConstants.MODULE_TAG + "DeviceAddTask";
  private String requestTime;
  private String batchId;
  private ConfigurationProvider provider;
  private TokenState tokenState;
  private DevicePreferences devicePreference;

  DeviceAddTask(Context context, DevicePreferences devicePreference) {
    super(context);
    provider = ConfigurationProvider.getInstance(this.context);
    tokenState = new TokenState();
    this.devicePreference = devicePreference;
  }

  @Override public TaskResult execute() {
    Logger.v(TAG + " execution started");
    try {
      String appId = MoEUtils.getAppId();
      if (TextUtils.isEmpty(appId)) {
        Logger.e(TAG + " execute: Cannot make device add call, app id not present.");
        taskResult.setIsSuccess(false);
        return taskResult;
      }
      Response response = APIManager.deviceAdd(appId, getBody(), getRequestId());
      if (response != null && response.responseCode == 200){
        taskResult.setIsSuccess(true);
      }
    } catch (Exception e) {
      Logger.e(TAG + " execute() : ", e);
    }
    Logger.v(TAG + " execution completed");
    taskResult.setPayload(tokenState);
    return taskResult;
  }

  @Override public String getTaskTag() {
    return TAG_DEVICE_ADD;
  }

  @Override public boolean isSynchronous() {
    return true;
  }

  private JSONObject getSDKMeta() throws JSONException {
    JSONObject metaJSON = new JSONObject();
    batchId = MoEUtils.getRequestId();
    requestTime = MoEUtils.currentISOTime();
    metaJSON.put(MoEConstants.ATTR_BATCH_ID, batchId)
        .put(MoEConstants.REQUEST_ATTR_REQUEST_TIME, requestTime)
        .put(MoEConstants.REQUEST_ATTR_DEVICE_PREFERENCE, getPreferencesJson());
    return metaJSON;
  }

  private JSONObject getQueryParams() throws JSONException {
    JsonBuilder builder = RestUtils.getDefaultParams(context);

    builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_TIMEZONE, TimeZone.getDefault().getID());

    if (!devicePreference.isPushOptedOut) {
      PushTokens pushTokens = MoEDAO.getInstance(context).getPushTokens();
      if (!MoEUtils.isEmptyString(pushTokens.fcmToken)) {
        builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_FCM_PUSH_ID, pushTokens.fcmToken);
        tokenState.isSendingFcmToken = true;
      }
      if (!MoEUtils.isEmptyString(pushTokens.oemToken)) {
        builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_OEM_TOKEN, pushTokens.oemToken);
        tokenState.isSendingSecondaryToken = true;
      }
    }

    if (!provider.isDataTrackingOptedOut()) {
      String androidId = MoEUtils.getAndroidID(context);
      if (!TextUtils.isEmpty(androidId)) {
        builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_ANDROID_ID, androidId);
      }
      if (!SdkConfig.getConfig().isGaidTrackingOptedOut) {
        String gaid = provider.getStoredGAID();
        if (TextUtils.isEmpty(gaid)) {
          AdvertisingIdClient.AdInfo adInfo = MoEUtils.getAdvertisementInfo(context);
          if (adInfo != null) {
            gaid = adInfo.getId();
            provider.storeGAID(gaid);
          }
        }
        if (!TextUtils.isEmpty(gaid)) {
          builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_GAID, gaid);
        }
      }
      builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_OS_VERSION,
          String.valueOf(Build.VERSION.SDK_INT));
      builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_MODEL, Build.MODEL);
      builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_APP_VERSION_NAME,
          provider.getAppVersionName());

      String nwType = MoEUtils.getNetworkType(context);
      if (!TextUtils.isEmpty(nwType)) {
        builder.putString(MoEConstants.GENERIC_PARAM_KEY_NW_TYPE, nwType);
      }
    }

    return builder.build();
  }

  private JSONObject getBody() throws JSONException {
    JsonBuilder requestBody =
        getDeviceInfo().putJsonObject(MoEConstants.ATTR_SDK_META, getSDKMeta())
            .putJsonObject(MoEConstants.REQUEST_ATTR_QUERY_PARAMS, getQueryParams());
    return requestBody.build();
  }

  private JsonBuilder getDeviceInfo() throws JSONException {
    JsonBuilder deviceInfo = new JsonBuilder();
    if (SdkConfig.getConfig().isDeviceAttributeTrackingOptedOut
        || provider.isDataTrackingOptedOut()) {
      return deviceInfo;
    }
    deviceInfo.putString("OS_VERSION", VERSION.RELEASE)
        .putInt("OS_API_LEVEL", VERSION.SDK_INT)
        .putString("DEVICE", Build.DEVICE)
        .putString("MODEL", Build.MODEL)
        .putString("PRODUCT", Build.PRODUCT)
        .putString("MANUFACTURER", Build.MANUFACTURER);
    String deviceId = MoEUtils.getAndroidID(context);
    if (!TextUtils.isEmpty(deviceId)) {
      deviceInfo.putString("DEVICE_ID", deviceId);
    }
    String operator = MoEUtils.getOperatorName(context);
    if (!TextUtils.isEmpty(operator)) {
      deviceInfo.putString("CARRIER", operator);
    }
    WindowManager wm = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
    DisplayMetrics outMetrics = new DisplayMetrics();
    if (wm != null) {
      wm.getDefaultDisplay().getMetrics(outMetrics);
      deviceInfo.putInt("DENSITYDPI", outMetrics.densityDpi)
          .putInt("WIDTH", outMetrics.widthPixels)
          .putInt("HEIGHT", outMetrics.heightPixels);
    }
    if (!SdkConfig.getConfig().isGaidTrackingOptedOut) {
      AdvertisingIdClient.AdInfo adInfo = MoEUtils.getAdvertisementInfo(context);
      if (null != adInfo) {
        deviceInfo.putString(MoEConstants.ATTR_MOE_GAID, adInfo.getId())
            .putInt(MoEConstants.ATTR_IS_LAT, adInfo.isLimitAdTrackingEnabled());
      }
    }
    return deviceInfo;
  }

  private String getRequestId() {
    return MoEUtils.getSha1ForString(
        batchId + requestTime + provider.getCurrentUserId());
  }

  private JSONObject getPreferencesJson() throws JSONException {
    JsonBuilder preferences = new JsonBuilder();
    preferences.putBoolean(MoEConstants.REQUEST_ATTR_PUSH_PREFERENCE,
        !devicePreference.isPushOptedOut);
    preferences.putBoolean(MoEConstants.REQUEST_ATTR_IN_APP_PREFERENCE,
        !devicePreference.isInAppOptedOut);
    preferences.putBoolean(MoEConstants.REQUEST_ATTR_DATA_TRACKING_PREFERENCE,
        !devicePreference.isDataTrackingOptedOut);
    return preferences.build();
  }
}
