/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import android.annotation.TargetApi;
import android.app.AlarmManager;
import android.app.PendingIntent;
import android.app.job.JobInfo;
import android.app.job.JobInfo.Builder;
import android.app.job.JobScheduler;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.os.Build.VERSION;
import android.os.Build.VERSION_CODES;
import androidx.annotation.NonNull;
import androidx.annotation.RestrictTo;
import androidx.annotation.RestrictTo.Scope;
import com.moengage.core.executor.TaskManager;
import com.moengage.core.model.MoEJobParameters;

/**
 * @author Arshiya Khanum
 * Date: 2020/08/21
 *
 * @since 10.4.00
 */
@RestrictTo(Scope.LIBRARY)
public class DataManager {

  private static final String TAG = MoEConstants.MODULE_TAG + "DataManager";

  @SuppressWarnings("ConstantConditions")
  void scheduleDataSending(@NonNull Context context) {
    if (context == null) {
      Logger.e(TAG + "scheduleDataSending() : context is null");
      return;
    }
    try {
      if (VERSION.SDK_INT < VERSION_CODES.LOLLIPOP) {
        scheduleDataSendingAlarm(context, APP_CLOSE_DATA_SYNC_ALARM_ID, APP_CLOSE_SYNC_INTERVAL);
      } else {
        scheduleDataSendingJob(context, APP_CLOSE_DATA_SYNC_JOB_ID, APP_CLOSE_SYNC_INTERVAL);
      }
      scheduleBackgroundSyncIfRequired(context);
    } catch (Exception e) {
      Logger.e(TAG + "scheduleDataSending() :  Exception: ", e);
    }
  }

  public void scheduleImmediateRetry(Context context, int seconds) {
    Logger.v(
        TAG + " scheduleImmediateRetry() : Scheduling immediate retry, delay: " + seconds);
    if (VERSION.SDK_INT < VERSION_CODES.LOLLIPOP) {
      scheduleDataSendingAlarm(context, IMMEDIATE_RETRY_DATA_SYNC_ALARM_ID, seconds);
    } else {
      scheduleDataSendingJob(context, IMMEDIATE_RETRY_DATA_SYNC_JOB_ID, seconds);
    }
  }

  public void backgroundSync(Context context, MoEJobParameters jobParameters){
    Logger.v(TAG + " backgroundSync() : ");
    TaskManager.getInstance().startTask(new SendInteractionDataTask(context, jobParameters));
    scheduleBackgroundSyncIfRequired(context);
  }

  private void scheduleBackgroundSyncIfRequired(Context context){
    if (!SdkConfig.getConfig().isBackgroundSyncEnabled) return;
    long syncInterval =
        RemoteConfig.getConfig().dataSyncRetryInterval/1000;
    Logger.v(TAG + " scheduleBackgroundSyncIfRequired() : Will schedule background sync.");
    if (VERSION.SDK_INT < VERSION_CODES.LOLLIPOP) {
      scheduleDataSendingAlarm(context, BACKGROUND_DATA_SYNC_ALARM_ID, syncInterval);
    } else {
      scheduleDataSendingJob(context, BACKGROUND_DATA_SYNC_JOB_ID, syncInterval);
    }
    Logger.v(TAG + " scheduleBackgroundSyncIfRequired() : Background sync scheduled.");
  }

  private void scheduleDataSendingAlarm(@NonNull Context context, int alarmId, long syncInterval) {
    Logger.v(TAG + " scheduleDataSending() alarmId: " + alarmId + " interval: " + syncInterval);
    Intent alarmIntent = new Intent(context, MoEAlarmReceiver.class);
    PendingIntent pendingIntent =
        PendingIntent.getBroadcast(context, alarmId, alarmIntent,
            PendingIntent.FLAG_UPDATE_CURRENT);
    AlarmManager alarmManager = (AlarmManager) context.getSystemService(Context.ALARM_SERVICE);
    if (alarmManager != null) {
      alarmManager.set(AlarmManager.RTC_WAKEUP,
          MoEUtils.secondsToMillis(MoEUtils.currentSeconds() + syncInterval), pendingIntent);
    }
  }

  @TargetApi(VERSION_CODES.LOLLIPOP)
  private void scheduleDataSendingJob(@NonNull Context context, int jobId, long syncInterval) {
    Logger.v(TAG + " scheduleDataSendingJob() JobId: " + jobId + " interval: " + syncInterval);
    ComponentName serviceComponent = new ComponentName(context, DataSyncJob.class);
    JobInfo.Builder builder = new Builder(jobId, serviceComponent);
    builder.setRequiredNetworkType(JobInfo.NETWORK_TYPE_ANY);
    builder.setOverrideDeadline(MoEUtils.secondsToMillis(syncInterval * 2));
    builder.setMinimumLatency(MoEUtils.secondsToMillis(syncInterval));
    JobScheduler jobScheduler =
        (JobScheduler) context.getSystemService(Context.JOB_SCHEDULER_SERVICE);
    if (jobScheduler != null) {
      jobScheduler.schedule(builder.build());
    }
  }

  public static String ACTION_DATA_SENDING = "MOE_ACTION_DATA_SENDING";

  private static final int APP_CLOSE_DATA_SYNC_JOB_ID = 90001;
  private static final int APP_CLOSE_DATA_SYNC_ALARM_ID = 90002;
  public static final int BACKGROUND_DATA_SYNC_JOB_ID = 90003;
  public static final int BACKGROUND_DATA_SYNC_ALARM_ID = 90004;
  public static final int IMMEDIATE_RETRY_DATA_SYNC_JOB_ID = 90005;
  public static final int IMMEDIATE_RETRY_DATA_SYNC_ALARM_ID = 90006;

  public static final int BATCH_SIZE = 100;

  public static int MAX_TEST_DEVICE_TIME = 60;
  public static int IMMEDIATE_RETRY_INTERVAL_ATTEMPT_ONE = 60;
  public static int IMMEDIATE_RETRY_INTERVAL_ATTEMPT_TWO = 180;
  public static int APP_CLOSE_SYNC_INTERVAL = 3;
}
