/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.core;

import android.content.Context;
import android.net.Uri;
import androidx.annotation.Nullable;
import com.moe.pushlibrary.exceptions.SDKNotInitializedException;
import com.moengage.core.rest.RequestBuilder;
import com.moengage.core.rest.RequestBuilder.RequestType;
import com.moengage.core.rest.Response;
import com.moengage.core.rest.RestClient;
import com.moengage.core.rest.exceptions.UTF8EncodingException;
import com.moengage.core.security.SecretKeyType;
import com.moengage.core.utils.JsonBuilder;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URL;
import java.security.InvalidKeyException;
import java.util.HashMap;
import java.util.Map.Entry;
import java.util.TimeZone;
import javax.net.ssl.HttpsURLConnection;
import org.json.JSONObject;

/**
 * This class is responsible for all interactions with the MoE servers.
 *
 * @author MoEngage (abhishek@moengage.com)
 * @version 5.0
 * @since 5.0
 */
public final class APIManager {

  private APIManager() {
    // Constructor intentionally made private. Utility class cannot be
    // instantiated
  }

  static void uploadLogsToLogEntries(String api, JSONObject jsonObject) {
    try {
      URL url = new URL(api);
      HttpsURLConnection httpsURLConnection = (HttpsURLConnection) url.openConnection();
      httpsURLConnection.setDoOutput(true);
      httpsURLConnection.setRequestProperty("Accept-Charset", "UTF-8");
      httpsURLConnection.setRequestProperty("Content-type", "application/json");
      OutputStream output = httpsURLConnection.getOutputStream();
      output.write(jsonObject.toString().getBytes("UTF-8"));
      output.close();
      Logger.v(
          "API Manager : uploadLogsToLogEntries : response" + httpsURLConnection.getResponseCode());
    } catch (Exception e) {

    }
  }

  @Nullable
  public static Response deviceTriggerSyncRequest(Context context, String path, HashMap<String,
      String> paramsMap, JSONObject requestBody) {
    try {
      if (!RemoteConfig.getConfig().isRealTimeTriggerEnabled
          || !RemoteConfig.getConfig().isAppEnabled || MoEUtils.isEmptyString(
          SdkConfig.getConfig().appId)) {
        return null;
      }
      Uri.Builder uriBuilder = RestUtils.getBaseUriBuilder()
          .appendEncodedPath(path);
      RequestBuilder requestBuilder =
          RestUtils.getBaseRequestBuilder(uriBuilder.build(), RequestType.POST,
              MoEUtils.getAppId());

      JsonBuilder queryParams = RestUtils.getDefaultParams(context);
      queryParams.putString(MoEConstants.GENERIC_PARAM_V2_KEY_TIMEZONE,
          TimeZone.getDefault().getID());

      if (paramsMap != null) {
        for (Entry<String, String> entry : paramsMap.entrySet()) {
          queryParams.putString(entry.getKey(), entry.getValue());
        }
      }
      if (requestBody == null) {
        requestBody = new JSONObject();
      }
      requestBody.put(MoEConstants.REQUEST_ATTR_QUERY_PARAMS, queryParams.build());
      requestBuilder.addBody(requestBody);
      RestClient restClient = new RestClient(requestBuilder.build());
      return restClient.executeRequest();
      //return new APIResponse(client.getResponse(), client.getResponseCode());
    } catch (Exception e) {
      Logger.e("MoERestClient: deviceTriggerSyncRequest() : Exception ", e);
    }
    return null;
  }


  @Nullable static Response reportAdd(String appId, String path, String requestId, JSONObject body)
      throws IOException, SDKNotInitializedException, InvalidKeyException {
    Uri.Builder uriBuilder = RestUtils.getBaseUriBuilder()
        .appendEncodedPath(path);

    RequestBuilder requestBuilder =
        RestUtils.getBaseRequestBuilder(uriBuilder.build(), RequestType.POST, appId)
            .addHeader(MoEConstants.REQUEST_HEADER_REQUEST_ID, requestId)
            .addBody(body);

    RestClient restClient = new RestClient(requestBuilder.build());
    return restClient.executeRequest();
  }

  @Nullable public static Response configApi(String appId, JSONObject requestBody)
      throws IOException, SDKNotInitializedException, InvalidKeyException {
    Uri.Builder uriBuilder = RestUtils.getBaseUriBuilder()
        .appendEncodedPath(MoEConstants.API_ENDPOINT_CONFIG_API)
        .appendEncodedPath(appId);

    RequestBuilder
        requestBuilder = RestUtils.getBaseRequestBuilder(uriBuilder.build(), RequestType.POST, appId)
        .addBody(requestBody);

    if (SdkConfig.getConfig().isEncryptionEnabled) {
      requestBuilder
          .addHeader(MoEConstants.HEADER_MOE_PAYLOAD_ENC_KEY_TYPE, SecretKeyType.DEFAULT.name().toLowerCase())
          .enableEncryption(RemoteConfigDefault.DEFAULT_DATA_ENCRYPTION_KEY);
    }

    RestClient restClient = new RestClient(requestBuilder.build());
    return restClient.executeRequest();
  }

  public static Response deviceAdd(String appId, JSONObject requestBody, String requestId)
      throws SDKNotInitializedException, UTF8EncodingException, InvalidKeyException {
    Uri.Builder uriBuilder = RestUtils.getBaseUriBuilder()
        .appendEncodedPath(MoEConstants.API_ENDPOINT_DEVICE_ADD)
        .appendPath(appId);
    RequestBuilder requestBuilder = RestUtils.getBaseRequestBuilder(uriBuilder.build(),
        RequestType.POST, appId)
        .addBody(requestBody)
        .addHeader(MoEConstants.REQUEST_HEADER_REQUEST_ID, requestId);
    RestClient restClient = new RestClient(requestBuilder.build());
    return restClient.executeRequest();
  }
}
