/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.reports;

import android.content.Context;
import android.support.annotation.Nullable;
import com.moe.pushlibrary.models.BatchData;
import com.moe.pushlibrary.models.Event;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEDAO;
import com.moengage.core.MoEUtils;
import com.moengage.core.model.DevicePreferences;
import com.moengage.core.model.ReportBatch;
import com.moengage.core.model.ReportBatchMeta;
import com.moengage.core.model.SDKIdentifiers;
import com.moengage.core.model.TrafficSource;
import com.moengage.core.model.UserSession;
import java.util.ArrayList;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 * Date: 2019-06-06
 */
public class ReportsBatchHelper {

  private static final String TAG = "ReportsBatchHelper";

  public BatchData updateBatchIfRequired(Context context, BatchData batch) {
    try {
      JSONObject batchJson = batch.batchDataJson;
      if (batchJson.has(MoEConstants.REQUEST_HEADER_REQUEST_ID)) {
        Logger.v(TAG + " updateBatchIfRequired() : Batch already updated. No update required.");
        return batch;
      }
      Logger.v(TAG + " updateBatchIfRequired() : Batch does not have request id and time will add"
          + " it now.");
      batch.batchDataJson = updateBatch(batchJson,
          MoEDAO.getInstance(context).getSDKIdentifiers());
      if (batch._id != -1) {
        MoEDAO.getInstance(context).updateBatch(batch);
      }
    } catch (Exception e) {
      Logger.e(TAG + " updateBatchIfRequired() : Exception: ", e);
    }
    return batch;
  }

  JSONObject updateBatch(JSONObject batchJson,
      SDKIdentifiers identifiers) throws JSONException {
    ReportBatchMeta batchMeta = batchMetaFromJson(batchJson);

    JSONObject metaJson = new JSONObject();
    metaJson.put(MoEConstants.ATTR_BATCH_ID, batchMeta.batchId);
    metaJson.put(MoEConstants.REQUEST_ATTR_REQUEST_TIME, batchMeta.requestTime);
    JSONObject preferencesJson = devicePreferencesJson(batchMeta.preferences);
    if (preferencesJson != null) {
      metaJson.put(MoEConstants.REQUEST_ATTR_DEVICE_PREFERENCE, preferencesJson);
    }
    batchJson.put(MoEConstants.ATTR_SDK_META, metaJson);
    String requestId =
        MoEUtils.getSha1ForString(
            batchMeta.batchId + batchMeta.requestTime + identifiers.sdkUniqueId);
    batchJson.put(MoEConstants.REQUEST_HEADER_REQUEST_ID, requestId);
    return batchJson;
  }

  ReportBatchMeta batchMetaFromJson(JSONObject batchJson) {
    ReportBatchMeta batchMeta = savedBatchMeta(batchJson);
    if (batchMeta == null) {
      batchMeta = new ReportBatchMeta(null,
          MoEUtils.getRequestId(), MoEUtils.currentISOTime());
    } else {
      if (MoEUtils.isEmptyString(batchMeta.batchId)) {
        batchMeta.batchId = MoEUtils.getRequestId();
      }
      if (MoEUtils.isEmptyString(batchMeta.requestTime)) {
        batchMeta.requestTime = MoEUtils.currentISOTime();
      }
    }
    return batchMeta;
  }

  @Nullable ReportBatchMeta savedBatchMeta(JSONObject batchJson) {
    try {
      if (!batchJson.has(MoEConstants.ATTR_SDK_META)) return null;
      JSONObject metaJson = batchJson.getJSONObject(MoEConstants.ATTR_SDK_META);
      DevicePreferences preferences = null;
      if (metaJson.has(MoEConstants.REQUEST_ATTR_DEVICE_PREFERENCE)) {
        JSONObject devicePreferences =
            metaJson.getJSONObject(MoEConstants.REQUEST_ATTR_DEVICE_PREFERENCE);
        preferences = new DevicePreferences(
            devicePreferences.has(MoEConstants.REQUEST_ATTR_DATA_TRACKING_PREFERENCE),
            devicePreferences.has(MoEConstants.REQUEST_ATTR_IN_APP_PREFERENCE),
            devicePreferences.has(MoEConstants.REQUEST_ATTR_PUSH_PREFERENCE)
        );
      }
      return new ReportBatchMeta(
          preferences,
          metaJson.optString(MoEConstants.ATTR_BATCH_ID, null),
          metaJson.optString(MoEConstants.REQUEST_ATTR_REQUEST_TIME, null)
      );
    } catch (Exception e) {
      Logger.e(TAG + " batchMetaFromJson() : Exception: ", e);
    }
    return null;
  }

  @Nullable JSONObject devicePreferencesJson(DevicePreferences preferences) {
    JSONObject preferencesJson = new JSONObject();
    try {
      if (preferences == null) return null;
      if (preferences.isDataTrackingOptedOut)
        preferencesJson.put(MoEConstants.REQUEST_ATTR_DATA_TRACKING_PREFERENCE, false);

      if (preferences.isPushOptedOut)
        preferencesJson.put(MoEConstants.REQUEST_ATTR_PUSH_PREFERENCE, false);
      if (preferences.isInAppOptedOut)
        preferencesJson.put(MoEConstants.REQUEST_ATTR_IN_APP_PREFERENCE, false);
    } catch (Exception e) {
      Logger.e(TAG + " devicePreferencesJson() : Exception: ", e);
    }
    return preferencesJson.length() > 0 ? preferencesJson : null;
  }

  @Nullable JSONObject identifierJson(SDKIdentifiers identifiers) {
    JSONObject identifiersJson = new JSONObject();
    try {
      if (identifiers.userAttributeUniqueId != null) {
        identifiersJson.put(ATTR_MOE_USER_ID, identifiers.userAttributeUniqueId);
      }
      if (identifiers.segmentAnonymousId != null) {
        identifiersJson.put(ATTR_SEGMENT_ID, identifiers.segmentAnonymousId);
      }
    } catch (Exception e) {
      Logger.e(TAG + " getIdentifiers() : Exception: ", e);
    }
    return identifiersJson.length() > 0 ? identifiersJson : null;
  }

  private Object lock = new Object();

  public void createAndSaveBatches(Context context, UserSession session) {
    synchronized (lock) {
      ArrayList<Event> eventList = null;
      MoEDAO dao = MoEDAO.getInstance(context);
      DevicePreferences devicePreferences = dao.getDevicePreferences();
      for (; ; ) {
        //read events from data-points table
        eventList = dao.getInteractionData(100);
        Logger.d(TAG +
            ": createAndSaveBatches() :Fetching interaction data in batches");
        if (eventList == null || eventList.isEmpty()) {
          Logger.d(TAG + "createAndSaveBatches(): Found Nothing to send");
          return;
        }
        // create event batches
        ReportBatch reportBatch = new ReportBatch(eventList,
            new ReportBatchMeta(devicePreferences, MoEUtils.getRequestId(),
                MoEUtils.currentISOTime(), session,
                !MoEDAO.getInstance(context).isDeviceRegistered()),
            dao.getSDKIdentifiers());

        String interactionData = createBatch(reportBatch);
        if (interactionData == null) {
          return;
        }
        //write to batch table
        dao.writeBatch(interactionData);
        //delete events from data-point table
        dao.deleteInteractionData(eventList, context);
        eventList.clear();
      }
    }
  }

  @Nullable
  String createBatch(ReportBatch reportBatch) {
    try {
      JSONObject batch = new JSONObject();
      //add events and its count
      JSONArray viewsInfoArray = new JSONArray();
      for (Event event : reportBatch.getEventList()) {
        viewsInfoArray.put(new JSONObject(event.details));
      }
      batch.put(MoEConstants.ATTR_INTERACTION_VIEWS_COUNT, viewsInfoArray.length())
          .put(MoEConstants.ATTR_INTERACTION_VIEWS_INFO, viewsInfoArray);
      // add meta if required
      JSONObject sdkMeta = metaJson(reportBatch.getBatchMeta());
      if (sdkMeta != null) {
        batch.put(MoEConstants.ATTR_SDK_META, sdkMeta);
      }
      // add identifiers if required
      JSONObject sdkIdentifiers = identifierJson(reportBatch.getSdkIdentifiers());
      if (sdkIdentifiers != null) {
        batch.put(MoEConstants.ATTR_SDK_IDENTIFIERS, sdkIdentifiers);
      }
      // adding request
      batch.put(MoEConstants.REQUEST_HEADER_REQUEST_ID, MoEUtils.getSha1ForString(
          reportBatch.getBatchMeta().batchId
              + reportBatch.getBatchMeta().requestTime
              + reportBatch.getSdkIdentifiers()
              .sdkUniqueId));
      return batch.toString();
    } catch (Exception e) {
      Logger.e(TAG + " createBatch() : Exception: ", e);
    }
    return null;
  }

  JSONObject metaJson(ReportBatchMeta meta) {
    JSONObject metaJson = new JSONObject();
    try {
      metaJson.put(MoEConstants.ATTR_BATCH_ID, meta.batchId);
      metaJson.put(MoEConstants.REQUEST_ATTR_REQUEST_TIME, meta.requestTime);

      JSONObject devicePreferences = devicePreferencesJson(meta.preferences);
      if (devicePreferences != null) {
        metaJson.put(MoEConstants.REQUEST_ATTR_DEVICE_PREFERENCE, devicePreferences);
      }
      if (meta.isDeviceAddPending){
        metaJson.put(ATTR_DEVICE_ADD_RESPONSE, "failure");
      }
      UserSession userSession = meta.userSession;
      if (userSession != null) {
        JSONArray sourceArray = new JSONArray();
        if (userSession.trafficSource != null && !TrafficSource.isEmpty(
            userSession.trafficSource)) {
          JSONObject trafficJson = TrafficSource.toJson(userSession.trafficSource);
          if (MoEUtils.hasKeys(trafficJson)) {
            sourceArray.put(trafficJson);
          }
        }
        metaJson.put(REQUEST_ATTR_SOURCE, sourceArray);
        JSONObject sessionJson = UserSession.toJson(userSession);
        if (sessionJson != null) {
          if (sessionJson.has(UserSession.SOURCE_ARRAY)) {
            sessionJson.remove(UserSession.SOURCE_ARRAY);
          }
          if (sessionJson.has(UserSession.LAST_INTERACTION_TIME)) {
            sessionJson.remove(UserSession.LAST_INTERACTION_TIME);
          }
          if (sessionJson.has(UserSession.INITIATED_IN_BACKGROUND) && sessionJson.getInt(UserSession.INITIATED_IN_BACKGROUND) != 1){
            sessionJson.remove(UserSession.INITIATED_IN_BACKGROUND);
          }
          metaJson.put(REQUEST_ATTR_SESSION, sessionJson);
        }
      }

      return metaJson;
    } catch (Exception e) {
      Logger.e(TAG + " metaJson() : Exception: ", e);
    }
    return metaJson;
  }

  private static final String REQUEST_ATTR_SOURCE = "source";
  private static final String REQUEST_ATTR_SESSION = "session";
  private static final String ATTR_MOE_USER_ID = "moe_user_id";
  private static final String ATTR_SEGMENT_ID = "segment_id";
  private static final String ATTR_DEVICE_ADD_RESPONSE = "dev_add_res";

}
