/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import android.support.annotation.NonNull;
import com.moengage.core.listeners.AppBackgroundListener;
import com.moengage.core.listeners.OnAppBackgroundListener;
import com.moengage.core.listeners.OnLogoutCompleteListener;
import java.util.ArrayList;
import java.util.List;

/**
 * Callback manager for MoEngage Core module.
 *
 * @author Umang Chamaria
 * Date: 26/03/20
 */
public class MoECallbacks {

  private MoECallbacks() {
    backgroundListenerList = new ArrayList<>();
    logoutCompleteListenerList = new ArrayList<>();
    appBackgroundListeners = new ArrayList<>();
  }

  private static MoECallbacks instance;

  private List<OnAppBackgroundListener> backgroundListenerList;
  private List<OnLogoutCompleteListener> logoutCompleteListenerList;
  private List<AppBackgroundListener> appBackgroundListeners;

  public static MoECallbacks getInstance() {
    if (instance == null) {
      synchronized (MoECallbacks.class) {
        if (instance == null) instance = new MoECallbacks();
      }
    }
    return instance;
  }

  /**
   * Adds a callback listener for {@link OnAppBackgroundListener}
   *
   * @param listener Instance of {@link OnAppBackgroundListener}
   */
  @SuppressWarnings("ConstantConditions")
  public void addAppBackgroundListener(@NonNull OnAppBackgroundListener listener) {
    if (listener == null) return;
    backgroundListenerList.add(listener);
  }

  /**
   * Removes a callback listener for {@link OnAppBackgroundListener}
   *
   * @param listener Instance of {@link OnAppBackgroundListener}
   */
  @SuppressWarnings("ConstantConditions")
  public void removeAppBackgroundListener(@NonNull OnAppBackgroundListener listener) {
    if (listener == null) return;
    backgroundListenerList.remove(listener);
  }

  List<OnAppBackgroundListener> getBackgroundListenerList() {
    return backgroundListenerList;
  }

  /**
   * Adds a callback listener for {@link OnLogoutCompleteListener}
   *
   * @param listener Instance of {@link OnLogoutCompleteListener}
   */
  @SuppressWarnings("ConstantConditions")
  public void addLogoutCompleteListener(@NonNull OnLogoutCompleteListener listener) {
    if (listener == null) return;
    logoutCompleteListenerList.add(listener);
  }

  /**
   * Removes a callback listener for {@link OnAppBackgroundListener}
   *
   * @param listener Instance of {@link OnLogoutCompleteListener}
   */
  @SuppressWarnings("ConstantConditions")
  public void removeLogoutListener(@NonNull OnLogoutCompleteListener listener) {
    if (listener == null) return;
    logoutCompleteListenerList.remove(listener);
  }

  List<OnLogoutCompleteListener> getLogoutListenerList() {
    return logoutCompleteListenerList;
  }

  @SuppressWarnings("ConstantConditions")
  public void addAppBackgroundListener(@NonNull AppBackgroundListener listener){
    if (listener == null) return;
    appBackgroundListeners.add(listener);
  }

  @SuppressWarnings("ConstantConditions")
  public void removeAppBackgroundListener(@NonNull AppBackgroundListener listener){
    if (listener == null) return;
    appBackgroundListeners.add(listener);
  }

  public List<AppBackgroundListener> getAppBackgroundListeners(){
    return appBackgroundListeners;
  }

}
