/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.integrations.segment;

import android.content.Context;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.PayloadBuilder;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import java.util.Iterator;
import org.json.JSONObject;

/**
 * Transform Segment's event attribute format to MoEngage format and track an event.
 * @author Umang Chamaria
 * Date: 29/04/19
 */
public class TrackSegmentEventTask extends SDKTask {

  private String eventName;
  private JSONObject eventAttributes;

  private static final String TAG = "TrackSegmentEventTask";

  public TrackSegmentEventTask(Context context, String eventName, JSONObject eventAttributes) {
    super(context);
    this.eventName =  eventName;
    this.eventAttributes = eventAttributes;
  }

  @Override public TaskResult execute() {
    try{
      Logger.v(TAG + " execute() : Started Execution");
      MoEHelper.getInstance(context).trackEvent(eventName,
          transformEventAttributesToMoEngageFormat(eventAttributes));
      Logger.v(TAG + " execute() : Completed Execution");
    }catch (Exception e){
      Logger.e( TAG + " execute() : Exception: ", e);
    }
    return null;
  }

  @Override public String getTaskTag() {
    return TAG_TRACK_EVENT_SEGMENT;
  }

  @Override public boolean isSynchronous() {
    return false;
  }

  /**
   * Converts the Segment Properties to a format understood by MoEngage SDK.
   *
   * @param eventAttributes Properties passed by the Segment SDK.
   * @return {@link JSONObject} of properties converted to MoEngage Format.
   */
  private JSONObject transformEventAttributesToMoEngageFormat(JSONObject eventAttributes) {
    try {
      Logger.v(TAG + " transformEventAttributesToMoEngageFormat() : Transforming track properties "
          + " to MoEngage format");
      PayloadBuilder builder = new PayloadBuilder();
      Iterator<String> iterator = eventAttributes.keys();
      while (iterator.hasNext()) {
        String key = iterator.next();
        Object value = eventAttributes.get(key);
        if (value instanceof String) {
          if (!MoEUtils.isDate((String) value)) {
            builder.putAttrObject(key, value);
          } else {
            builder.putAttrISO8601Date(key, (String) value);
          }
        } else {
          builder.putAttrObject(key, value);
        }
      }
      return builder.build();
    } catch (Exception e) {
      Logger.e(TAG + " transformEventAttributesToMoEngageFormat() : ");
      return eventAttributes;
    }
  }
}
