/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.executor;

import android.support.annotation.NonNull;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import java.util.HashSet;
import java.util.Set;

/**
 * Helper class to manage tasks running on background thread.
 *
 * @author Umang Chamaria
 * Date: 09/06/20
 */
public class TaskManager {

  private static final String TAG = MoEConstants.MODULE_TAG + "TaskManager";

  private Set<String> runningTaskList;

  private TaskManager() {
    runningTaskList = new HashSet<>();
  }

  private static TaskManager instance;

  public static TaskManager getInstance() {
    if (instance == null) {
      synchronized (TaskManager.class) {
        if (instance == null) instance = new TaskManager();
      }
    }
    return instance;
  }

  /**
   * Attempts to add a task to the processing queue.
   *
   * @param task instance of {@link ITask}
   * @return true if the task is added to the queue else false.
   */
  @SuppressWarnings("ConstantConditions") public boolean addTaskToQueue(@NonNull ITask task) {
    if (task == null) return false;
    Logger.v(TAG + " addTaskToQueue() Trying to add " + task.getTaskTag() + " to the queue");
    if (canAddTaskToQueue(task)) {
      Logger.v(task.getTaskTag() + " added to queue");
      runningTaskList.add(task.getTaskTag());
      TaskProcessor.getInstance().addTask(task);
      return true;
    } else {
      Logger.v(TAG
          + " addTaskToQueue() Task is already queued. Cannot add it to queue. Task : "
          + task.getTaskTag());
      return false;
    }
  }

  /**
   * Attempts to add task to the beginning of the processing queue.
   *
   * @param task instance of {@link ITask}
   * @return true if the task is added to the queue else false.
   */
  @SuppressWarnings("ConstantConditions") public boolean addTaskToQueueBeginning(
      @NonNull ITask task) {
    if (task == null) return false;
    Logger.v(TAG + " addTaskToQueueBeginning() Trying to add " + task.getTaskTag() + " to the "
        + "queue");
    if (canAddTaskToQueue(task)) {
      Logger.v(
          TAG + " addTaskToQueueBeginning() " + task.getTaskTag() + " added to beginning of queue");
      runningTaskList.add(task.getTaskTag());
      TaskProcessor.getInstance().addTaskToFront(task);
      return true;
    } else {
      Logger.v(
          TAG + " addTaskToQueueBeginning() : Task is already queued. Cannot add it to queue.");
      return false;
    }
  }

  /**
   * Tries to execute the task immediately.
   *
   * @param task instance of {@link ITask}
   * @return true if the task can be executed immediately else false.
   */
  @SuppressWarnings("ConstantConditions") public boolean startTask(@NonNull ITask task) {
    if (task == null) return false;
    Logger.v(TAG + " startTask() : Try to start task " + task.getTaskTag());
    if (canAddTaskToQueue(task)) {
      Logger.v(TAG + " Starting task " + task.getTaskTag());
      runningTaskList.add(task.getTaskTag());
      TaskProcessor.getInstance().runTask(task);
      return true;
    } else {
      Logger.v(
          TAG
              + " startTask() : Cannot start task. Task is already in progress or queued. "
              + task.getTaskTag());
      return false;
    }
  }

  public void execute(@NonNull Task task) {
    TaskProcessor.getInstance().execute(task);
  }

  private boolean canAddTaskToQueue(ITask task) {
    if (!task.isSynchronous()) return true;
    return !runningTaskList.contains(task.getTaskTag());
  }

  void removeTaskFromList(String tag) {
    Logger.v(TAG + " removeTaskFromList() : Removing tag from list: " + tag);
    runningTaskList.remove(tag);
  }
}
