/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.support.annotation.ColorRes;
import android.support.annotation.DrawableRes;
import android.support.annotation.IntegerRes;
import android.support.annotation.NonNull;
import android.text.TextUtils;
import com.moe.pushlibrary.MoEHelper;
import com.moengage.core.executor.TaskManager;
import com.moengage.core.logger.LogcatLogAdapter;
import java.util.ArrayList;
import java.util.List;

/**
 * Initialise MoEngage SDK using this class.
 *
 * Please follow the guidelines mentioned in <a
 * href="https://docs.moengage.com/docs/sdk-installation"
 * >MoEngage Integration Docs</a> for integrating the
 * MoEngage SDK.<br>
 *
 * This initializer should be called in in the `onCreate()` of the {@link Application} class.
 * Below is a sample of how the SDK should be initialised.
 * <pre>
 *   <code>
 *     MoEngage moEngage = new MoEngage.Builder(this, "XXXXXXXXXXXXXXX")
 *     .setSenderId("615448685370")
 *     .setNotificationSmallIcon(R.drawable.icon)
 *     .setNotificationLargeIcon(R.drawable.ic_launcher)
 *     .setLogLevel(Logger.VERBOSE)
 *     .build();
 *    MoEngage.initialise(moEngage);
 *   </code>
 * </pre>
 *
 * <em>Replace XXXXXXXXXXXXXXX with your app-id on the MoEngage Dashboard.</em><br>
 * <p><p/>
 *
 * <b>Note: The above initialisation example does not contain all the methods. It only includes a
 * few commonly used methods. Please refer to the documentation of {@link MoEngage.Builder}
 * for all methods and add them according to your needs.</b>
 *
 * @author Umang Chamaria
 * @since 9.0.00
 */

public class MoEngage {

  public enum DATA_REGION {
    /**
     * Enum to redirect data to Indian Cluster
     *
     * @since 10.0.02
     */
    REGION_SERV3,
    /**
     * Enum to redirect data to European Cluster
     */
    REGION_EU,
    /**
     * Enum to redirect data to the default region. This is the default value need be specified
     * explicitly.
     */
    REGION_DEFAULT;
  }

  private static final String TAG = MoEConstants.MODULE_TAG + "MoEngage";

  private MoEngage.Builder builder;

  private MoEngage(MoEngage.Builder builder) {
    this.builder = builder;
  }

  /**
   * Checks whether the installed build is debuggable or not.
   *
   * @return true is the build is debuggable else false.
   */
  public static boolean isDebugBuild() {
    return isDebugBuild;
  }

  private static boolean isDebugBuild = false;

  static void setBuildStatus(boolean buildStatus) {
    isDebugBuild = buildStatus;
  }

  /**
   * Initialise MoEngage SDK with {@link MoEngage} object returned by {@link MoEngage.Builder}
   *
   * @param moEngage object of {@link MoEngage}
   * @since 9.0.00
   */
  @SuppressWarnings("ConstantConditions") public static void initialise(
      @NonNull MoEngage moEngage) {
    Logger.addLogAdapter(new LogcatLogAdapter());
    if (moEngage == null) {
      Logger.e(TAG + " Object instance is null cannot initialise");
      return;
    }
    MoEngage.Builder builder = moEngage.builder;
    if (builder == null || builder.application == null) {
      Logger.e(TAG + "initialise() Builder/Context/Application is null. Cannot initialise SDK.");
      return;
    }
    Context context = builder.application.getApplicationContext();
    isDebugBuild = MoEUtils.isDebugBuild(context);
    //setting log level
    Logger.setLogLevel(builder.sdkConfig.logLevel);
    // loading default config
    RemoteConfig.setRemoteConfig(new RemoteConfig());
    // loading config from disk
    TaskManager.getInstance().addTaskToQueueBeginning(new LoadConfigurationFromDiskTask(context));
    //setting app id
    if (!builder.sdkConfig.isSegmentIntegration) {
      if (!TextUtils.isEmpty(builder.appId)) {
        builder.sdkConfig.appId = MoEUtils.formatAppId(builder.appId);
      } else {
        Logger.e(TAG + "initialise() : " + "App-id not passed. Cannot use MoEngage Platform");
        return;
      }
    } else {
      Logger.v(TAG + " initialise() : Segment integration enabled will not use app id");
    }
    // set large icon
    if (builder.sdkConfig.pushConfig.largeIcon == -1) {
      Logger.e(TAG + "initialise() : " + "Large icon not set");
    }
    //set small icon
    if (builder.sdkConfig.pushConfig.smallIcon == -1) {
      Logger.e(TAG + " initialise() : " + "Small icon not set will not show notification");
    }

    //notification tone
    if (!TextUtils.isEmpty(builder.sdkConfig.pushConfig.tone)) {
      String tone = builder.sdkConfig.pushConfig.tone;
      if (tone.contains(".")) {
        tone = tone.substring(0, tone.lastIndexOf("."));
      }
      builder.sdkConfig.pushConfig.tone = tone;
    }
    // inAppOptOut activity
    ArrayList<String> activityNameListInApp = new ArrayList<>();
    if (builder.inAppOptOutList != null) {
      try {
        for (Class clazz : builder.inAppOptOutList) {
          activityNameListInApp.add(clazz.getName());
        }
      } catch (Exception e) {
        Logger.e(TAG + "initialise() : " + "Activity Opt out ", e);
      }
    }
    activityNameListInApp.add("com.moengage.pushbase.activities.PushTracker");
    activityNameListInApp.add("com.moengage.pushbase.activities.SnoozeTracker");
    activityNameListInApp.add("com.moengage.integrationverifier.IntegrationVerificationActivity");
    builder.sdkConfig.inAppOptOutList = activityNameListInApp;
    //activity tracking opt-out
    if (builder.activityTrackingOptOutList != null) {
      try {
        ArrayList<String> activityNameList =
            new ArrayList<>(builder.activityTrackingOptOutList.size());
        for (Class clazz : builder.activityTrackingOptOutList) {
          activityNameList.add(clazz.getName());
        }
        builder.sdkConfig.activityTrackingOptOutList = activityNameList;
      } catch (Exception e) {
        Logger.e(TAG + "initialise() : " + "Activity Opt out ", e);
      }
    }
    MoEHelper.getInstance(context).setApplication(builder.application);
    // register lifecycle callback
    if (builder.application != null && !builder.sdkConfig.isSegmentIntegration) {
      MoEHelper.getInstance(context).registerActivityLifecycle(builder.application);
    } else {
      Logger.v(TAG + " initialise() : Segment integration is enabled. Will not register for "
          + "lifecycle callbacks.");
    }

    SdkConfig.setSdkConfig(builder.sdkConfig);
    //printing config
    try {
      if ((SdkConfig.getConfig().isLogEnabledForSignedBuild || isDebugBuild())
          && SdkConfig.getConfig().logLevel >= Logger.VERBOSE) {
        Logger.v(TAG + " initialise() : Config: \n" + SdkConfig.getConfig());
      }
    } catch (Exception e) {
      Logger.e(TAG + " initialise() : ", e);
    }
  }

  /**
   * Optionally opt-out of data tracking. When data tracking is opted no event or user
   * attribute is tracked on MoEngage Platform.
   *
   * @param context Application context.
   * @param shouldOptOutDataTracking true if you don't want to track user data, else false.
   * @since 9.1.00
   */
  public static void optOutDataTracking(Context context, boolean shouldOptOutDataTracking) {
    Logger.d(
        TAG + " optOutDataTracking() : Opt Out Called with value: " + shouldOptOutDataTracking);
    ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
    boolean savedState = provider.isDataTrackingOptedOut();
    provider.optOutOfDataTracking(shouldOptOutDataTracking);
    if (savedState != shouldOptOutDataTracking) {
      MoEDispatcher.getInstance(context).getDeviceAddManager().registerGdprOptOut(context);
    }
    if (shouldOptOutDataTracking) {
      // gaid optOut
      SdkConfig.getConfig().isGaidTrackingOptedOut = true;
      //androidId optOut
      SdkConfig.getConfig().isAndroidIdTrackingOptedOut = true;
      //location optOut
      SdkConfig.getConfig().isLocationTrackingOptedOut = true;
      //geo-fence optOut
      SdkConfig.getConfig().isGeofenceTrackingOptedOut = true;
      //device attributes optOut
      SdkConfig.getConfig().isDeviceAttributeTrackingOptedOut = true;
      //setting location services state
      SdkConfig.getConfig().isLocationServiceEnabled = false;
      //clean data
      TaskManager.getInstance().addTaskToQueue(new DataTrackingOptOutTask(context));
    }
  }

  /**
   * Optionally opt-out of push campaigns. No push campaigns will be shown once this is
   * opted out.
   *
   * @param context Application context.
   * @param shouldOptOutPushNotification true if you don't want users to receive push
   * notification, else false.
   * @since 9.1.00
   */
  public static void optOutPushNotification(Context context, boolean
      shouldOptOutPushNotification) {
    Logger.d(
        TAG + "optOutPushNotification: Opt Out Called with value: " + shouldOptOutPushNotification);
    ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
    boolean savedState = provider.isPushNotificationOptedOut();
    provider.optOutOfPushNotification(shouldOptOutPushNotification);
    if (shouldOptOutPushNotification) {
      provider.clearPushToken();
    }
    if (savedState != shouldOptOutPushNotification) {
      MoEDispatcher.getInstance(context).getDeviceAddManager().registerGdprOptOut(context);
    }
  }

  /**
   * Optionally opt-out of in-app campaigns. No in-app campaigns will be shown once this is
   * opted out.
   *
   * @param context Application context.
   * @param shouldOptOutInApp true if you don't want users to receive in-app
   * notification, else false.
   * @since 9.1.00
   */
  public static void optOutInAppNotification(Context context, boolean
      shouldOptOutInApp) {
    Logger.d(TAG + " optOutInAppNotification() : Opt out called with value: " + shouldOptOutInApp);
    ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
    boolean savedState = provider.isInAppOptedOut();
    provider.optOutOfInAppNotification(shouldOptOutInApp);
    if (savedState != shouldOptOutInApp) {
      MoEDispatcher.getInstance(context).getDeviceAddManager().registerGdprOptOut(context);
    }
  }

  /**
   * Initialise MoEngage SDK using this class.
   *
   * Please follow the guidelines mentioned in <a
   * href="https://docs.moengage.com/docs/sdk-installation"
   * >MoEngage Integration Docs</a> for integrating the
   * MoEngage SDK.<br>
   *
   * This initializer should be called in in the `onCreate()` of the {@link Application} class.
   * Below is a sample of how the SDK should be initialised.
   * <pre>
   *   <code>
   *     MoEngage moEngage = new MoEngage.Builder(this, "XXXXXXXXXXXXXXX")
   *     .setSenderId("615448685370")
   *     .setNotificationSmallIcon(R.drawable.icon)
   *     .setNotificationLargeIcon(R.drawable.ic_launcher)
   *     .setLogLevel(Logger.VERBOSE)
   *     .build();
   *    MoEngage.initialise(moEngage);
   *   </code>
   * </pre>
   *
   * <em>Replace XXXXXXXXXXXXXXX with your app-id on the MoEngage Dashboard.</em><br>
   * <p><p/>
   *
   * <b>Note: The above initialisation example does not contain all the methods. It only includes a
   * few commonly used methods. Please refer to the documentation of {@link MoEngage.Builder}
   * for all methods and add them according to your needs.</b>
   *
   * @author Umang Chamaria
   * @since 9.0.00
   */
  public static class Builder {

    private String appId;
    private Application application;
    private List<Class> inAppOptOutList;
    private List<Class> activityTrackingOptOutList;
    private SdkConfig sdkConfig;

    /**
     * Builder to initialise MoEngage SDK.
     * App Id can we found on the
     * <a href="https://app.moengage.com/v3/#/settings/app/general">Settings Page</a> of MoEngage
     * Dashboard
     *
     * @param application Instance of {@link Application}
     * @param appId App ID from MoEngage Dashboard
     * @since 9.0.00
     */
    public Builder(@NonNull Application application, @NonNull String appId) {
      this.application = application;
      this.appId = appId;
      sdkConfig = new SdkConfig();
    }

    /**
     * Set the large icon to be shown in the notification.
     *
     * @param largeIcon resource link to the large icon.
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder setNotificationLargeIcon(@DrawableRes int largeIcon) {
      sdkConfig.pushConfig.largeIcon = largeIcon;
      return this;
    }

    /**
     * Set the small icon to be shown in the notification.<br>
     * <b>This icon is a mandate for showing notifications</b><br>
     * Make sure the small icon follows the notification icon guidelines.
     * Guideline: Notification small icon should be flat, pictured face on, and must be white on
     * a transparent background. For more details refer to the
     * <a href="https://developer.android.com/guide/practices/ui_guidelines/icon_design_status_bar.html#style11">official
     * docs</a>
     * for more details.
     *
     * @param smallIcon resource link to the small icon.
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder setNotificationSmallIcon(@DrawableRes int smallIcon) {
      sdkConfig.pushConfig.smallIcon = smallIcon;
      return this;
    }

    /**
     * Set the sender-id/project id for the GCM/FCM project.<br>
     * <b>Sender id is a mandate when you are using GCM or Instant apps and MoEngage SDK is
     * registering for push.</b>
     *
     * @param senderId sender id/project id for your GCM/FCM project.
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder setSenderId(@NonNull String senderId) {
      sdkConfig.pushConfig.senderId = senderId;
      return this;
    }

    /**
     * Set the color for notification. Currently color is given more precedence over large icon.
     *
     * @param color color resource id
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder setNotificationColor(@ColorRes int color) {
      sdkConfig.pushConfig.notificationColor = color;
      return this;
    }

    /**
     * Set the file name for notification tone.<br>
     * <b>Note:</b> This will work only below Android Oreo Devices.
     * <b>File should be present in the raw folder. Do not pass the file extension, only file
     * name should be passed.</b>
     *
     * @param tone file name for the notification tone.
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder setNotificationTone(String tone) {
      sdkConfig.pushConfig.tone = tone;
      return this;
    }

    /**
     * By default, the SDK shows only one notification at a given point in time, existing
     * notification is updated with the new notification. In case you want to
     * show multiple notification use this API.
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 10.0.00
     */
    public Builder enableMultipleNotificationInDrawer() {
      sdkConfig.pushConfig.shouldShowMultiplePushInDrawer = true;
      return this;
    }

    /**
     * By default, the MoEngage SDK will try to show in-app messages on all the activities. If you
     * want block in-apps on any specific screen(s) use this API to pass a list of screens on
     * which in-apps should not be shown.
     *
     * @param inAppOptOutClassList list of {@link Class} on which in-app should not be shown.
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutInAppOnActivity(List<Class> inAppOptOutClassList) {
      inAppOptOutList = inAppOptOutClassList;
      return this;
    }

    /**
     * By default, the MoEngage SDK would track screen-name for all the activities. In case you
     * don't want to track any specific screen(s) use this API to pass the list of screens to the
     * SDK which should not be tracked.
     *
     * @param trackingOptOut list of {@link Class} which should not be tracked.
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutActivityTracking(List<Class> trackingOptOut) {
      activityTrackingOptOutList = trackingOptOut;
      return this;
    }

    /**
     * By default MoEngage SDK synthesis the back-stack from the manifest when redirecting user on
     * push click. In case you don't want back-stack synthesis use this opt-out
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutBackStackBuilder() {
      sdkConfig.pushConfig.isBackStackBuilderOptedOut = true;
      return this;
    }

    /**
     * By default, the MoEngage SDK hides the top navigation bar when showing in-apps. If your
     * application has navigation drawer and fragments some-times
     * the fragment is not rendered after in-app is dismissed. Opt-Out of navigation bar
     * manipulation if your app is built using navigation drawer and fragments.
     *
     * @return instance of {@link MoEngage.Builder}
     */
    public Builder optOutNavBar() {
      sdkConfig.isNavBarOptedOut = true;
      return this;
    }

    /**
     * By default MoEngage SDK collects Google Advertising Identifier(GAID). In case you want to
     * restrict this GAID collection use this this opt-out.
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutGAIDCollection() {
      sdkConfig.isGaidTrackingOptedOut = true;
      return this;
    }

    /**
     * By Default MoEngage SDK collects Android id. In case you want to restrict Android id
     * collection use this opt-out.
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutAndroidIdCollection() {
      sdkConfig.isAndroidIdTrackingOptedOut = true;
      return this;
    }

    /**
     * By default MoEngage SDK tracks user location. In case you want to restrict collection of
     * user location use this opt-out.
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutLocationTracking() {
      sdkConfig.isLocationTrackingOptedOut = true;
      return this;
    }

    /**
     * By default the MoEngage SDK tries to set geo-fence based on the geo-fence campaigns created
     * on MoEngage Dashboard. In case you want don't want to set geo-fence for use this opt-out<br>
     * Note: If this opt-out is used no geo-fence campaign created on MoEngage Dashboard will
     * be delivered.
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutGeoFence() {
      sdkConfig.isGeofenceTrackingOptedOut = true;
      return this;
    }

    /**
     * MoEngage SDK tries to collect the operator name if permission is granted.
     * In case you want to restrict collection of operator name use this opt-out.
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutCarrierNameCollection() {
      sdkConfig.isCarrierTrackingOptedOut = true;
      return this;
    }

    /**
     * MoEngage SDK collects device attributes like height, width,screen density. In case you
     * want to restrict device attribute collection use this op-out.
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutDeviceAttributeCollection() {
      sdkConfig.isDeviceAttributeTrackingOptedOut = true;
      return this;
    }

    /**
     * By default the MoEngage SDK registers for push token. In case you don't want the SDK to
     * register for push use this opt-out.
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutTokenRegistration() {
      sdkConfig.pushConfig.isFcmPushRegistrationEnabled = false;
      return this;
    }

    /**
     * Redirect data to different regions if required.<br>
     *
     * @param regionConstant one of the possible values based on the region you want to redirect
     * to.
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder redirectDataToRegion(DATA_REGION regionConstant) {
      sdkConfig.dataRegion = regionConstant;
      return this;
    }

    /**
     * Set the interval(in seconds) at which the SDK flushes out the data to the server.<br>
     * Note: Value will only be accepted if it is greater than the SDK defined flush interval.
     *
     * @param interval time interval to flush data. Value should be in seconds.
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder setFlushInterval(long interval) {
      sdkConfig.flushInterval = interval;
      return this;
    }

    /**
     * By default MoEngage SDK sync's data every few minutes when the app is in foreground. In
     * case you want to disable periodic flush use this opt-out.
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutPeriodicFlush() {
      sdkConfig.isPeriodicFlushEnabled = false;
      return this;
    }

    /**
     * Sets log level for MoEngage SDK's logs. Default level is INFO
     *
     * @param logLevel log Level
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder setLogLevel(int logLevel) {
      sdkConfig.logLevel = logLevel;
      return this;
    }

    /**
     * Enables/Disables MoEngage logs.<br><b>Note : This API should be used only if logs are
     * required in production/signed builds.</b>
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder enableLogsForSignedBuild() {
      sdkConfig.isLogEnabledForSignedBuild = true;
      return this;
    }

    /**
     * Enable location services for the SDK. By default SDK does not track location or set
     * geo-fence.
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder enableLocationServices() {
      sdkConfig.isLocationServiceEnabled = true;
      return this;
    }

    /**
     * Optionally remove notification large icon
     *
     * @return instance of {@link MoEngage.Builder}
     * @since 9.0.00
     */
    public Builder optOutNotificationLargeIcon() {
      sdkConfig.pushConfig.isLargeIconOptedOut = true;
      return this;
    }

    /**
     * Builds the configuration to initialise MoEngage SDK.
     * Call {@link MoEngage#initialise(MoEngage)} with the returned object to
     *
     * @return instance of {@link MoEngage}
     * @since 9.0.00
     */
    public MoEngage build() {
      return new MoEngage(this);
    }

    /**
     * MoEngage SDK by default syncs data in the background at a regular interval.
     * In case you don't want the SDK to sync data in background use this opt-out.<br>
     * <b>Note:</b> This can lead some un-intended delay in data appearing on MoEngage Dashboard.
     *
     * @return instance of {@link MoEngage}
     * @since 9.1.05
     */
    public Builder optOutBackgroundSync() {
      sdkConfig.isBackgroundSyncEnabled = false;
      return this;
    }

    /**
     * MoEngage SDK by default syncs Real Time Trigger campaigns in the background at a regular
     * interval. In case you don't want the SDK to sync campaign in the background use this
     * opt-out.<br>
     * <b>Note:</b> This can lead to reduced delivery of Real Time Trigger Campaigns.
     *
     * @return instance of {@link MoEngage}
     * @since 9.1.05
     */
    public Builder optOutRealTimeTriggerBackgroundSync() {
      sdkConfig.isRealTimeTriggerBackgroundSyncEnabled = false;
      return this;
    }

    /**
     * This API it to notify MoEngage SDK that the app uses MoEngage SDK along with Segment's
     * SDK as a bundle.<br>
     * <b>Note:</b> Do not use this API if you are not using MoEngage via Segment as a bundled
     * SDK. Calling this API without using Segment bundle will result in malfunction of the SDK.
     *
     * @return instance of {@link MoEngage}
     * @since 9.1.06
     */
    public Builder enableSegmentIntegration() {
      sdkConfig.isSegmentIntegration = true;
      return this;
    }

    /**
     * By default MoEngage SDK shows in-apps using the {@link Activity} lifecycle callbacks. In
     * case you want more granular control to show in-apps you can opt-out of the default
     * implementation of showing in-apps on lifecycle callbacks using this opt-out.
     *
     * @return instance of {@link MoEngage}
     * @since 9.4.00
     */
    public Builder optOutDefaultInAppDisplay() {
      sdkConfig.isLifecycleInAppOptedOut = true;
      return this;
    }

    /**
     * Use {@link Builder#enableMultipleNotificationInDrawer()}
     */
    @Deprecated public Builder setNotificationType(@IntegerRes int notificationType) {
      if (application == null) return this;
      int count = application.getApplicationContext().getResources().getInteger(notificationType);
      if (count > 1) sdkConfig.pushConfig.shouldShowMultiplePushInDrawer = true;
      return this;
    }

    /**
     * If enabled MoEngage SDK will attempt to fetch user location even if the application is in
     * background to set/update Geo-fences accurately.<br/>
     * <b>Note:</b> On Android 10 and above background location fetch triggers a notification to the
     * user. Make sure you have informed the user about this background location fetch else they
     * might completely revoke location permission.
     *
     * @return instance of {@link MoEngage}
     * @since 9.8.04
     */
    public Builder enableBackgroundLocationFetch() {
      sdkConfig.isBackgroundLocationFetchEnabled = true;
      return this;
    }

    /**
     * By default, SDK updates Geofences even in background. In case you don't want the SDK to
     * update fences in the background use this opt out.
     *
     * @return instance of {@link MoEngage}
     * @since 9.8.04
     */
    public Builder optOutGeofenceBackgroundSync() {
      sdkConfig.isGeofenceBackgroundSyncEnabled = false;
      return this;
    }

    /**
     * By default the MoEngage SDK does not register for Huawei PushKit push token. In
     * case you want the MoEngage SDK to handle the Huawei PushKit push registration use this API.
     * @return instance of {@link MoEngage}
     * @since 10.2.00
     */
    public Builder enablePushKitTokenRegistration() {
      sdkConfig.pushConfig.isPushKitRegistrationEnabled = true;
      return this;
    }

    /**
     * By default, SDK shows a placeholder image in case the image download fails. In case you
     * want to change the placeholder image use this API to provide the image.
     *
     * @param resourceId resource id of the image.
     * @return instance of {@link MoEngage}
     * @since 10.1.00
     */
    public Builder setPlaceHolderImageForCard(@DrawableRes int resourceId) {
      sdkConfig.cardConfig.cardPlaceHolderImage = resourceId;
      return this;
    }

    /**
     * By default, SDK shows an image and text indicating inbox is empty. In case you want to
     * change the image use this API to provide the image.
     *
     * @param resourceId resource id of the image.
     * @return instance of {@link MoEngage}
     * @since 10.1.00
     */
    public Builder setEmptyInboxImageForCard(@DrawableRes int resourceId) {
      sdkConfig.cardConfig.inboxEmptyImage = resourceId;
      return this;
    }

    /**
     * By default SDK
     *
     * @param dateFormat String for formatting date.
     * @return instance of {@link MoEngage}
     * @since 10.1.00
     */
    public Builder setDateFormatForCard(String dateFormat) {
      sdkConfig.cardConfig.cardsDateFormat = dateFormat;
      return this;
    }

    /**
     * Time interval after which SDK retry generating the push token in case of registration
     * failure.<br/>
     * <b>Note:</b> Token registration retry will only work if the SDK is registering for Push
     * Token.
     *
     * @param tokenRetryInterval time interval after which sdk should retry. Unit - SECONDS
     * @return instance of {@link MoEngage}
     * @since 10.1.00
     */
    public Builder setTokenRetryInterval(long tokenRetryInterval) {
      if (tokenRetryInterval > 5) {
        sdkConfig.pushConfig.tokenRetryInterval = tokenRetryInterval;
      }
      return this;
    }

    /**
     * By default the MoEngage SDK does not add additional encryption on top of https to REST API
     * calls.
     * In case you want the MoEngage SDK to add additional encryption use this API.
     * @return instance of {@link MoEngage}
     * @since 10.2.01
     */
    public Builder enableEncryption() {
      sdkConfig.isEncryptionEnabled = true;
      return this;
    }

    /**
     * Configure Mi Push.
     *
     * @param appId App-Id from the Mi Dashboard.
     * @param appKey App-Key from the Mi Dashboard.
     * @param enableTokenRegistration true if you want the MoEngage SDK to register for Push
     * Token else false.
     * @return instance of {@link MoEngage}
     * @since 10.3.00
     */
    public Builder configureMiPush(String appId, String appKey, boolean enableTokenRegistration){
      sdkConfig.pushConfig.miPushConfig.appId = appId;
      sdkConfig.pushConfig.miPushConfig.appKey = appKey;
      sdkConfig.pushConfig.miPushConfig.isRegistrationEnabled = enableTokenRegistration;
      return this;
    }
  }
}
