/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.executor;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingDeque;

/**
 * @author Umang Chamaria on 11/16/15.
 */
public class TaskProcessor {
  private static final String TAG = MoEConstants.MODULE_TAG + "TaskProcessor";

  private BlockingDeque<ITask> taskQueue;

  private ExecutorService executorService = Executors.newCachedThreadPool();

  private ArrayList<WeakReference<OnTaskCompleteListener>> taskCompleteListeners;

  private ITask active;

  private final Object lock = new Object();

  private static TaskProcessor instance = null;

  private TaskProcessor() {
    taskQueue = new LinkedBlockingDeque<>();
    taskCompleteListeners = new ArrayList<>();
  }

  public static TaskProcessor getInstance() {
    if (instance == null) {
      synchronized (TaskProcessor.class) {
        if (instance == null) instance = new TaskProcessor();
      }
    }
    return instance;
  }

  public void addTask(ITask task) {
    if (task != null) {
      taskQueue.add(task);
      startExecution();
    }
  }

  public void addTaskToFront(ITask task) {
    if (task != null) {
      taskQueue.addFirst(task);
      startExecution();
    }
  }

  private void startExecution() {
    if (active == null) {
      scheduleNext();
    }
  }

  private void scheduleNext() {
    if ((active = taskQueue.poll()) != null) {
      executorService.submit(new Runnable() {
        @Override
        public void run() {
          executeTask(active);
          scheduleNext();
        }
      });
    }
  }

  public void runTask(final ITask task) {
    executorService.submit(new Runnable() {
      @Override public void run() {
        executeTask(task);
      }
    });
  }

  private void executeTask(ITask task) {
    TaskResult result = task.execute();
    String taskTag = task.getTaskTag();
    if (!TextUtils.isEmpty(taskTag)) {
      TaskManager.getInstance().removeTaskFromList(taskTag);
      notifyListener(taskTag, result);
    }
  }

  @SuppressWarnings("ConstantConditions") public void removeOnTaskCompleteListener(
      @NonNull WeakReference<OnTaskCompleteListener> listener) {
    if (taskCompleteListeners != null && listener != null) {
      int index = taskCompleteListeners.indexOf(listener);
      if (index != -1) taskCompleteListeners.remove(index);
    }
  }

  @SuppressWarnings("ConstantConditions")
  @Nullable public WeakReference<OnTaskCompleteListener> setOnTaskCompleteListener(
      @NonNull OnTaskCompleteListener listener) {
    if (listener == null) return null;
    WeakReference<OnTaskCompleteListener> reference = new WeakReference<>(listener);
    taskCompleteListeners.add(reference);
    return reference;
  }

  private void notifyListener(String tag, TaskResult result) {
    if (taskCompleteListeners == null) return;
    synchronized (lock) {
      for (WeakReference<OnTaskCompleteListener> taskCompleteListener : taskCompleteListeners) {
        if (taskCompleteListener.get() != null) {
          taskCompleteListener.get().onTaskComplete(tag, result);
        }
      }
    }
  }

  /**
   * Use {@link TaskManager#execute(Task)}
   */
  @SuppressWarnings("ConstantConditions") void execute(@NonNull final Task task) {
    if (task == null) return;
    executorService.execute(new Runnable() {
      @Override public void run() {
        try {
          task.execute();
        } catch (Exception e) {
          Logger.e(TAG + " run() : ", e);
        }
      }
    });
  }
}