/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import android.content.Context;
import android.support.annotation.NonNull;
import com.moengage.core.executor.TaskManager;
import com.moengage.core.integrations.segment.TrackSegmentEventTask;
import com.moengage.core.integrations.segment.TrackSegmentUserAttributeTask;
import com.moengage.core.model.IntegrationMeta;
import com.moengage.push.PushManager;
import java.util.Map;
import org.json.JSONObject;

/**
 * Internal Helper class for interaction between Hybrid and Partner SDKs
 *
 * @author Umang Chamaria
 * Date: 26/06/20
 */
public class MoEIntegrationHelper {

  private static final String TAG = MoEConstants.MODULE_TAG + "MoEIntegrationHelper";

  @SuppressWarnings("ConstantConditions")
  public static void setIntegrationMeta(@NonNull IntegrationMeta meta) {
    try {
      if (meta == null) return;
      Logger.v(TAG + " setIntegrationMeta() : Meta: " + meta);
      ConfigurationCache.getInstance().setIntegrationType(meta.integrationType);
      ConfigurationCache.getInstance().setIntegrationVersion(meta.integrationVersion);
    } catch (Exception e) {
      Logger.e(TAG + " setIntegrationMeta() : ", e);
    }
  }

  /**
   * API to track event via MoEngage-Segment Bundled Integration.<br>
   * <b>Note:</b> Do not call this method. This is only for internal usage.
   *
   * @param eventName Event Name
   * @param eventAttributes Event Attributes
   * @since
   */
  @SuppressWarnings("ConstantConditions")
  public static void trackEventFromSegment(@NonNull Context context, @NonNull String eventName,
      JSONObject eventAttributes) {
    try {
      if (context == null) return;
      Logger.v(TAG + " trackEventFromSegment() : Event Name: " + eventName);
      Logger.v(TAG + " trackEventFromSegment() : Attributes: " + eventAttributes);
      if (MoEUtils.isEmptyString(eventName)) {
        Logger.e(TAG + " trackEventFromSegment() : Event name cannot be empty");
        return;
      }
      TaskManager.getInstance()
          .addTaskToQueue(new TrackSegmentEventTask(context, eventName, eventAttributes));
    } catch (Exception e) {
      Logger.e(TAG + " trackEventFromSegment() : ", e);
    }
  }

  /**
   * API to track user attributes via MoEngage-Segment Bundled Integration.<br>
   * <b>Note:</b> Do not call this method. This is only for internal usage.
   *
   * @param context instance of {@link Context}
   * @param userAttributesMap attribute map.
   * @since
   */
  @SuppressWarnings("ConstantConditions")
  public static void trackUserAttributeFromSegment(@NonNull Context context,
      Map<String, Object> userAttributesMap) {
    try {
      if (context == null || userAttributesMap == null) return;
      TaskManager.getInstance()
          .addTaskToQueue(new TrackSegmentUserAttributeTask(context, userAttributesMap));
    } catch (Exception e) {
      Logger.e(TAG + " trackUserAttributeFromSegment() : ", e);
    }
  }

  /**
   * Initialises the MoEngage SDK.
   *
   * @param context instance of {@link Context}
   * @param appId App-id for the MoEngage Platform.
   * @param senderId Sender-id for the FCM project, optional
   */
  public static void initialize(Context context, String appId, String senderId) {
    try {
      Logger.v(TAG + " initialize() : App-id: " + appId);
      Logger.v(TAG + " initialize() : Sender-Id: " + senderId);
      MoEngage.setBuildStatus(MoEUtils.isDebugBuild(context));
      if (MoEUtils.isEmptyString(appId)) return;
      SdkConfig.getConfig().appId = MoEUtils.formatAppId(appId);
      if (!MoEUtils.isEmptyString(senderId)) SdkConfig.getConfig().pushConfig.senderId = senderId;
      //force registration for push
      if (SdkConfig.getConfig().pushConfig.isFcmPushRegistrationEnabled) {
        PushManager.getInstance().registerForPush(context);
      }
    } catch (Exception e) {
      Logger.e(TAG + " initialize() : ", e);
    }
  }

  @SuppressWarnings("ConstantConditions")
  public static void trackSegmentAnonymousId(@NonNull Context context, String anonymousId) {
    try {
      if (context == null || MoEUtils.isEmptyString(anonymousId)) return;
      ConfigurationProvider.getInstance(context).saveSegmentAnonymousId(anonymousId);
    } catch (Exception e) {
      Logger.e(TAG + " trackSegmentAnonymousId() : ", e);
    }
  }
}
