/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.userattributes;

import android.content.Context;
import com.moe.pushlibrary.models.Event;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEDAO;
import com.moengage.core.MoEDispatcher;
import com.moengage.core.MoEUtils;
import com.moengage.core.RemoteConfig;
import com.moengage.core.events.MoEEventManager;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import com.moengage.core.model.MoEAttribute;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */
public class TrackAttributeTask extends SDKTask {
  private static final String TAG = "TrackAttributeTask";

  private JSONObject attributeJson;
  private boolean isCustomAttribute;

  TrackAttributeTask(Context context, JSONObject attributeJson, boolean isCustomAttribute) {
    super(context);
    this.attributeJson = attributeJson;
    this.isCustomAttribute = isCustomAttribute;
  }

  @Override public TaskResult execute() {
    try {
      Logger.v(TAG + " executing Task : ");
      if (isCustomAttribute) {
        Logger.v(TAG + " execute() : No need to cache custom attributes, will track attribute.");
        writeUserAttributeToStorage(attributeJson);
        taskResult.setIsSuccess(true);
        return taskResult;
      }
      MoEAttribute trackedAttribute = MoEUtils.convertJsonToAttributeObject(attributeJson);
      if (trackedAttribute == null) {
        Logger.v(TAG
            + " execute() : Conversion from Json to object failed cannot decide whether to track "
            + "attribute or not, will track attribute.");
        writeUserAttributeToStorage(attributeJson);
        taskResult.setIsSuccess(true);
        return taskResult;
      }
      Logger.v(TAG
          + " execute() : Will try to send attribute to server. Attribute: "
          + trackedAttribute.toString());
      // saved attribute value
      MoEAttribute savedAttribute =
          MoEDAO.getInstance(context).getAttributeByName(trackedAttribute.getName());
      //check for unique id change
      if (trackedAttribute.getName().equals(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID)) {
        if (!MoEDispatcher.getInstance(context)
            .getCoreEvaluator()
            .isValidUniqueId(RemoteConfig.getConfig().blockedUniqueIdRegex,
                trackedAttribute.getValue())) {
          Logger.e(TAG
              + " execute() : Not a valid unique id. tracked value: "
              + trackedAttribute.getValue());
          taskResult.setIsSuccess(true);
          return taskResult;
        }
        String currentUniqueId = MoEUtils.getUserAttributeUniqueId(context);
        if (currentUniqueId != null && !trackedAttribute.getValue().equals(currentUniqueId)) {
          Logger.v(TAG
              + " execute(): User Attribute Unique Id has changed will trigger force "
              + "logout.");
          MoEDispatcher.getInstance(context).handleLogout(true);
          return createTaskResult(attributeJson, false);
        } else {
          trackUserAttributeIfRequired(trackedAttribute, savedAttribute);
        }
        taskResult.setIsSuccess(true);
        return taskResult;
      }
      // generating sha1 of attribute value
      trackedAttribute.setValue(MoEUtils.getSha1ForString(trackedAttribute.getValue()));
      if (savedAttribute != null) {
        Logger.v(TAG + " execute(): Saved user attribute: " + savedAttribute.toString());
      }
      trackUserAttributeIfRequired(trackedAttribute, savedAttribute);
      Logger.v(TAG + " execute() : completed execution");
    } catch (Exception e) {
      Logger.e(TAG + " execute() : Exception ", e);
    }
    taskResult.setIsSuccess(true);
    return taskResult;
  }

  private void trackUserAttributeIfRequired(MoEAttribute trackedAttribute,
      MoEAttribute savedAttribute) {
    if (shouldTrackAttribute(trackedAttribute, savedAttribute,
        RemoteConfig.getConfig().userAttributeCachingTime)) {
      writeUserAttributeToStorage(attributeJson);
      cacheAttribute(trackedAttribute);
    } else {
      Logger.v(TAG
          + " trackUserAttributeIfRequired() : Attribute Already tracked. Will not be sent to "
          + "server.");
    }
  }

  @Override public String getTaskTag() {
    return TAG_TRACK_ATTRIBUTE;
  }

  @Override public boolean isSynchronous() {
    return false;
  }

  private void writeUserAttributeToStorage(JSONObject attributeJson) {
    Event event = new Event(MoEConstants.EVENT_ACTION_USER_ATTRIBUTE, attributeJson);
    MoEEventManager.getInstance(context).writeDataPointToStorage(event);
    syncIfRequired(event);
  }

  private void syncIfRequired(Event event) {
    if (event.details.contains(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID)) {
      Logger.v(TAG + "syncIfRequired() Unique Id set, So will try to send data");
      MoEDispatcher.getInstance(context).sendInteractionData();
    }
  }

  private boolean shouldTrackAttribute(MoEAttribute trackedAttribute, MoEAttribute savedAttribute,
      long sendDelay) {
    if (savedAttribute == null || trackedAttribute == null) return true;
    if (!trackedAttribute.getName().equals(savedAttribute.getName())) return true;
    if (!trackedAttribute.getValue().equals(savedAttribute.getValue())) return true;
    if (!trackedAttribute.getDataType().equals(savedAttribute.getDataType())) return true;
    return savedAttribute.getLastTrackedTime() + sendDelay < trackedAttribute.getLastTrackedTime();
  }

  private void cacheAttribute(MoEAttribute attribute) {
    Logger.v(TAG + " cacheAttribute(): Will cache attribute: " + attribute.toString());
    if (attribute == null) {
      Logger.e(TAG + " cacheAttribute() : Attribute object is null cannot cache it.");
      return;
    }
    if (attribute.getName().equals(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID)) {
      Logger.v(TAG
          + " cacheAttribute(): Attribute to cache is USER_ATTRIBUTE_UNIQUE_ID will copy"
          + " it to shared preference as well");
      MoEDAO.getInstance(context).saveUserAttributeUniqueId(attribute);
      return;
    }
    MoEDAO.getInstance(context).addOrUpdateAttributeToCache(attribute);
  }
}
