/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 * All rights reserved.
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.analytics;

import android.app.Activity;
import android.content.Context;
import android.net.Uri;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.annotation.WorkerThread;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.models.Event;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.ActionMapperConstants;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoECoreEvaluator;
import com.moengage.core.MoEDAO;
import com.moengage.core.MoEDispatcher;
import com.moengage.core.MoEUtils;
import com.moengage.core.RemoteConfig;
import com.moengage.core.model.TrafficSource;
import com.moengage.core.model.UserSession;
import java.util.UUID;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 * Date: 2019-05-21
 */
public class AnalyticsHelper {

  private static final String TAG = "AnalyticsHelper";

  private static AnalyticsHelper instance = null;

  private UserSession session;

  private MoECoreEvaluator evaluator;

  private SourceProcessor sourceProcessor;

  private AnalyticsHelper(Context context) {
    session = MoEDAO.getInstance(context).getLastSavedSession();
    evaluator = MoEDispatcher.getInstance(context)
        .getCoreEvaluator();
    sourceProcessor = new SourceProcessor();
  }

  public static AnalyticsHelper getInstance(Context context) {
    if (instance == null) {
      synchronized (AnalyticsHelper.class) {
        if (instance == null) instance = new AnalyticsHelper(context);
      }
    }
    return instance;
  }

  public void onEventTracked(Event event, Context context) {
    try {
      Logger.v(TAG
          + " onEventTracked() : Will update last interaction time if required. Event: "
          + event.details);
      // check if event is non-interactive
      if (!event.isInteractiveEvent) {
        Logger.v(
            TAG + " onEventTracked() : No operation required. Tracked event is non-interactive");
        return;
      }
      // user attribute tracked no action required.
      if (event.eventName.equals(MoEConstants.EVENT_ACTION_USER_ATTRIBUTE)) {
        Logger.v(TAG + " updateSession() : Need not update session info since user attribute is "
            + "tracked.");
        return;
      }
      // app in foreground no action required.
      if (MoEHelper.isAppInForeground()) {
        Logger.v(TAG + " updateSession() : App is in foreground no action required.");
        updateLastInteractionTime(MoEUtils.currentTime());
        return;
      }
      // no sessions created till now
      if (session == null) {
        Logger.v(TAG + " onEventTracked() : No previous session. Will create a new session");
        batchPreviousDataAndCreateNewSession(context, null, true);
        return;
      }
      // existing session expired create a new session.
      if (evaluator.hasSessionExpired(session.getLastInteractionTime(),
          RemoteConfig.getConfig().sessionInActiveTime,
          MoEUtils.currentTime())) {
        Logger.v(TAG + " onEventTracked() : Session has expired.");
        batchPreviousDataAndCreateNewSession(context, null, true);
        return;
      }
      //update last interaction
      updateLastInteractionTime(MoEUtils.currentTime());
    } catch (Exception e) {
      Logger.e(TAG + " onEventTracked() : Exception: ", e);
    }
  }

  void updateLastInteractionTime(long time) {
    if (session != null) {
      session.setLastInteractionTime(time);
    }
  }

  // update session if required
  @WorkerThread public void onAppOpen(Activity activity) {
    if (session != null) {
      Logger.v(TAG + " onAppOpen() : Current Session " + session.toString());
    }
    updateUserSessionIfRequired(activity);
  }

  public void onNotificationClicked(Context context, Bundle pushPayload,
      boolean isAppInBackground) {
    try {
      TrafficSource source = null;
      Logger.v(TAG + " onNotificationClicked() : ");
      MoEHelperUtils.dumpIntentExtras(pushPayload);
      String deepLink = getDeepLinkFromPayload(pushPayload);
      if (!MoEUtils.isEmptyString(deepLink)) {
        Uri uri = Uri.parse(deepLink);
        source = sourceProcessor.getTrafficSourceFromUrl(uri,
            RemoteConfig.getConfig().additionalSourceIdentifiers);
      }
      if (source == null || TrafficSource.isEmpty(source)) {
        Bundle extras = pushPayload;
        if (pushPayload.containsKey(ActionMapperConstants.KEY_ACTION_PAYLOAD)) {
          Bundle actionButtonExtras = getActionButtonExtras(pushPayload);
          if (actionButtonExtras != null) {
            extras = actionButtonExtras;
          }
        }
        source = sourceProcessor.getTrafficSourceFromExtras(extras,
            RemoteConfig.getConfig().additionalSourceIdentifiers);
      }
      updateSessionIfRequired(context, source, isAppInBackground);
    } catch (Exception e) {
      Logger.e(TAG + " onNotificationClicked() : ", e);
    }
  }

  private Bundle getActionButtonExtras(Bundle pushPayload) {
    try {
      String actionString = pushPayload.getString(ActionMapperConstants.KEY_ACTION_PAYLOAD);
      JSONObject actionJson = new JSONObject(actionString);
      if (ActionMapperConstants.ACTION_NAVIGATE.equals(
          actionJson.getString(ActionMapperConstants.KEY_ACTION_TAG)) && actionJson.has(
          ActionMapperConstants.KEY_EXTRA)) {
        return MoEHelperUtils.convertJSONObjecttoBundle(actionJson.getJSONObject(ActionMapperConstants.KEY_EXTRA));
      }
    } catch (Exception e) {
      Logger.e(TAG + " getActionButtonExtras() : ", e);
    }
    return null;
  }

  @Nullable private String getDeepLinkFromPayload(Bundle pushPayload) throws JSONException {
    if (pushPayload.containsKey(MoEHelperConstants.MOE_WEB_URL)) {
      return pushPayload.getString(MoEHelperConstants.MOE_WEB_URL);
    } else if (pushPayload.containsKey(MoEHelperConstants.GCM_EXTRA_WEB_URL)) {
      return pushPayload.getString(MoEHelperConstants.GCM_EXTRA_WEB_URL);
    } else if (pushPayload.containsKey(ActionMapperConstants.KEY_ACTION_PAYLOAD)) {
      String actionPayload = pushPayload.getString(ActionMapperConstants.KEY_ACTION_PAYLOAD);
      JSONObject actionJson = new JSONObject(actionPayload);
      if (actionJson.has(ActionMapperConstants.KEY_URI)) {
        return actionJson.getString(ActionMapperConstants.KEY_URI);
      }
    }
    return null;
  }

  private void updateUserSessionIfRequired(Activity activity) {
    try {
      Context context = activity.getApplicationContext();
      TrafficSource currentSource = sourceProcessor.getTrafficSourceFromActivity(activity,
          RemoteConfig.getConfig().additionalSourceIdentifiers);
      updateSessionIfRequired(context, currentSource, MoEHelper.isAppInBackground());
    } catch (Exception e) {
      Logger.e(TAG + " onAppOpen() : Exception: ", e);
    }
  }

  private void updateSessionIfRequired(Context context, TrafficSource currentSource,
      boolean isAppInBackground) {
    if (session == null) {
      Logger.v(TAG
          + " updateSessionIfRequired() : No saved session for user will create a new session.");
      batchPreviousDataAndCreateNewSession(context, currentSource,
          MoEHelper.isAppInBackground());
      return;
    }
    // previous session expired
    boolean hasSessionExpired =
        evaluator.hasSessionExpired(session.getLastInteractionTime(),
            RemoteConfig.getConfig().sessionInActiveTime,
            MoEUtils.currentTime());
    if (hasSessionExpired) {
      Logger.v(TAG
          + " updateSessionIfRequired() : Previous session has expired. Will create a new session"
          + ".");
      batchPreviousDataAndCreateNewSession(context, currentSource,
          MoEHelper.isAppInBackground());
      return;
    }
    // source changed
    TrafficSource savedSource = session.getSource();
    boolean hasSourceChanged = evaluator.hasSourceChanged(savedSource, currentSource);
    if (hasSourceChanged) {
      Logger.v(TAG + " updateSessionIfRequired() : Source changed. will create a new session");
      batchPreviousDataAndCreateNewSession(context, currentSource,
          isAppInBackground);
    }
  }

  @WorkerThread private void batchPreviousDataAndCreateNewSession(Context context,
      TrafficSource currentSource, boolean isFromBackground) {
    MoEDispatcher.getInstance(context).getBatchHelper().createAndSaveBatches(context, session);
    MoEDispatcher.getInstance(context).sendInteractionData();
    createAndPersistNewSession(context, currentSource, isFromBackground);
  }

  private UserSession createAndPersistNewSession(Context context,
      TrafficSource currentSource, boolean isFromBackground) {
    session = createNewSession(currentSource, isFromBackground);
    Logger.v(TAG + " createAndPersistNewSession() : New session: " + session.toString());
    persistUserSession(context, session);
    return session;
  }

  private UserSession createNewSession(@Nullable TrafficSource currentSource,
      boolean isFromBackground) {
    long currentTime = MoEUtils.currentTime();
    UserSession session = new UserSession();
    session.setSessionId(UUID.randomUUID().toString());
    session.setStartTime(MoEUtils.getTimeInISO(currentTime));
    if (currentSource != null) {
      session.setSource(currentSource);
    }
    session.setLastInteractionTime(currentTime);
    session.setBackgroundInitiated(isFromBackground);
    return session;
  }

  private void persistUserSession(Context context, UserSession userSession) {
    MoEDAO.getInstance(context).saveUserSession(userSession);
  }

  public void onAppClose(Context context) {
    updateLastInteractionTime(MoEUtils.currentTime());
    persistUserSession(context, session);
  }

  @WorkerThread public void onLogout(Context context) {
    createAndPersistNewSession(context, null, MoEHelper.isAppInBackground());
  }

  @Nullable public UserSession getSession() {
    return session;
  }
}
