/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import android.content.Context;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import com.moengage.core.model.AppStatus;
import java.util.Date;

/**
 * @author Umang Chamaria
 * Date: 16/01/20
 */
public class TrackInstallUpdateTask extends SDKTask {

  private static final String TAG = "TrackInstallUpdateTask";

  private AppStatus appStatus;

  public TrackInstallUpdateTask(Context context, AppStatus appStatus) {
    super(context);
    this.appStatus = appStatus;
  }

  @Override public TaskResult execute() {
    try {
      Logger.v(TAG + " execute() : executing task.");
      if (!RemoteConfig.getConfig().isAppEnabled) return taskResult;
      Logger.v(TAG + " execute() : Tracked App Status: " + appStatus);
      int currentVersion = ConfigurationProvider.getInstance(context).getAppVersion();
      switch (appStatus) {
        case UPDATE:
          trackUpdateEventIfRequired(currentVersion);
          break;
        case INSTALL:
          trackInstallEventIfRequired(currentVersion);
          break;
      }
      ConfigurationProvider.getInstance(context).storeAppVersion(currentVersion);
      Logger.v(TAG + " execute() : completed task.");
    } catch (Exception e) {
      Logger.e(TAG + " execute() : ", e);
    }
    return taskResult;
  }

  private void trackInstallEventIfRequired(int currentVersion) {
    if (ConfigurationProvider.getInstance(context).isInstallRegistered()) {
      Logger.e(TAG + " execute() : Install event is already tracked will not track again.");
      return;
    }
    Logger.v(TAG + " execute() : Will track install.");
    Properties attributes = new Properties();
    attributes.addAttribute(MoEHelperConstants.VERSION, currentVersion)
        .addAttribute(MoEConstants.GENERIC_PARAM_V2_KEY_SDK_VERSION,
            MoEHelperConstants.LIB_VERSION)
        .addAttribute(MoEHelperConstants.TIME_OF_INSTALL, System.currentTimeMillis())
        .addAttribute(MoEConstants.GENERIC_PARAM_V2_KEY_OS,
            MoEConstants.GENERIC_PARAM_V2_VALUE_OS);
    MoEHelper.getInstance(context)
        .trackEvent(MoEHelperConstants.EVENT_APP_INSTALL, attributes);
    ConfigurationProvider.getInstance(context).setInstallRegistered();
    taskResult.setIsSuccess(true);
  }

  private void trackUpdateEventIfRequired(int currentVersion) {
    int prevVersion = ConfigurationProvider.getInstance(context).getStoredAppVersion();
    if (currentVersion == prevVersion) {
      Logger.e(TAG + " trackUpdateEventIfRequired() : Update is already tracked for this version.");
      return;
    }
    Logger.v(TAG + " trackUpdateEventIfRequired() : Will track update.");
    Properties eventObj =
        new Properties().addAttribute(MoEHelperConstants.FROM_VERSION, prevVersion)
            .addAttribute(MoEHelperConstants.TO_VERSION, currentVersion)
            .addAttribute(MoEHelperConstants.TIME_OF_UPDATE, new Date());
    MoEHelper.getInstance(context)
        .trackEvent(MoEHelperConstants.EVENT_APP_UPDATE, eventObj);
    taskResult.setIsSuccess(true);
  }

  @Override public String getTaskTag() {
    return TAG_INSTALL_UPDATE_TASK;
  }

  @Override public boolean isSynchronous() {
    return true;
  }
}
