/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moe.pushlibrary.providers;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import com.moe.pushlibrary.providers.MoEDataContract.AttributeCacheEntity;
import com.moe.pushlibrary.providers.MoEDataContract.BatchDataEntity;
import com.moe.pushlibrary.providers.MoEDataContract.CampaignListEntity;
import com.moe.pushlibrary.providers.MoEDataContract.DTEntity;
import com.moe.pushlibrary.providers.MoEDataContract.DatapointEntity;
import com.moe.pushlibrary.providers.MoEDataContract.InAppMessageEntity;
import com.moe.pushlibrary.providers.MoEDataContract.InAppStatsEntity;
import com.moe.pushlibrary.providers.MoEDataContract.InAppV3Entity;
import com.moe.pushlibrary.providers.MoEDataContract.MessageEntity;
import com.moe.pushlibrary.providers.MoEDataContract.UserAttributeEntity;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import com.moengage.core.MoEUtils;
import com.moengage.core.inapp.InAppManager;
import com.moengage.core.model.DataTypes;
import com.moengage.core.model.InAppV2Meta;
import com.moengage.core.model.InAppV3Meta;
import com.moengage.core.storage.CardsDataContract.CardsEntity;
import java.util.ArrayList;
import org.json.JSONObject;

class MoEDatabaseHelper extends SQLiteOpenHelper {

  private static final String TAG = "DatabaseHelper";

  /**
   * The database name
   */
  private static final String DATABASE_NAME = "MOEInteractions";
  /**
   * The current database version
   */
  private static final int DATABASE_VERSION = 17;

  private Context context;

  /**
   * @param context Application Context
   */
  public MoEDatabaseHelper(Context context) {
    super(context, DATABASE_NAME, null, DATABASE_VERSION);
    this.context = context;
  }

  /**
   * Table containing the Inbox messages
   */
  static final String TABLE_NAME_MSGS = "MESSAGES";
  /**
   * Table containing events
   */
  static final String TABLE_NAME_DATAPOINTS = "DATAPOINTS";
  /**
   * Table containing in app messages
   */
  static final String TABLE_NAME_INAPPS = "INAPPMSG";
  /**
   * Old table containing in app messages
   */
  static final String TABLE_NAME_OLD_INAPPS = "INAPPS";
  /**
   * Table to save user attributes
   */
  static final String TABLE_NAME_USER_ATTRIBUTES = "USERATTRIBUTES";
  /**
   * Table to save campaign list
   */
  static final String TABLE_NAME_CAMPAIGN_LIST = "CAMPAIGNLIST";
  /**
   * Table to save batched data
   */
  static final String TABLE_NAME_BATCH_DATA = "BATCH_DATA";
  /**
   * Table to save device triggers
   */
  static final String TABLE_NAME_DEVICE_TRIGGERS = "DEVICE_TRIGGERS";

  static final String TABLE_NAME_ATTRIBUTE_CACHE = "ATTRIBUTE_CACHE";

  static final String TABLE_NAME_INAPP_V3 = "INAPP_V3";

  static final String TABLE_NAME_INAPP_STATS = "INAPP_STATS";

  static final String TABLE_NAME_CARD_CAMPAIGNS = "CARDS";

  @Override public void onCreate(SQLiteDatabase db) {
    // SQL statement to create interactions table
    String DDL_DATAPOINTS = "CREATE TABLE IF NOT EXISTS "
        + TABLE_NAME_DATAPOINTS
        + " ( "
        + DatapointEntity._ID
        + " INTEGER PRIMARY KEY, "
        + DatapointEntity.GTIME
        + " INTEGER, "
        + DatapointEntity.DETAILS
        + " TEXT ); ";

    createMessagesTable(db);

    createInAppTable(db);
    createUserAttributeTable(db);
    createCampaignListTable(db);
    db.execSQL(DDL_DATAPOINTS);
    createBatchDataTable(db);
    createDeviceTriggerTable(db);
    createAttributeCacheTableIfRequired(db);
    createInAppStatsTable(db);
    createInAppV3Table(db);
    createCardsTable(db);
    Logger.i(TAG + ": Database created");
  }

  private boolean tableExists(SQLiteDatabase db, final String tableName) {
    Cursor cursor = null;
    try {
      cursor = db.rawQuery(
          "select DISTINCT tbl_name from sqlite_master where tbl_name = '" + tableName + "'",
          null);
      return (cursor != null && cursor.getCount() > 0);
    } finally {
      if (null != cursor) {
        cursor.close();
      }
    }
  }

  public boolean isTableExists(SQLiteDatabase db, String tableName) {
    int count = 0;
    try {
      if (tableName == null || db == null || !db.isOpen()) {
        return false;
      }
      Cursor cursor = db.rawQuery("SELECT COUNT(*) FROM sqlite_master WHERE type = ? AND name = ?",
          new String[] { "table", tableName });
      if (!cursor.moveToFirst()) {
        return false;
      }
      count = cursor.getInt(0);
      cursor.close();
    } catch (Exception e) {
      Logger.e(TAG + ": isTableExists Exception " + e.toString());
    }
    return count > 0;
  }

  /*
   * (non-Javadoc)
   *
   * @see
   * android.database.sqlite.SQLiteOpenHelper#onUpgrade(android.database
   * .sqlite.SQLiteDatabase, int, int)
   */
  @Override public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
    Logger.v(TAG + ": Provider upgrading DB ");

    int upgradeTo = oldVersion + 1;
    while (upgradeTo <= newVersion) {
      switch (upgradeTo) {
        case 3:
          db.beginTransaction();
          try {
            createDataPointsTable(db);
            createMessagesTable(db);
            //creating the in-appv2 table here as the old table was dropped.
            createInAppTable(db);
            portDataFromv2(db);
            db.execSQL("DROP TABLE IF EXISTS moeints");
            db.execSQL("DROP TABLE IF EXISTS moemsgs");
            db.execSQL("DROP TABLE IF EXISTS moeinappmsgs");
            db.setTransactionSuccessful();
          } catch (Exception e) {
            Logger.e(TAG + ": failed to port data. FROM V2.", e);
          } finally {
            db.endTransaction();
          }
          break;
        case 4:
          // DBv4 Never released
          break;
        case 5:
          db.beginTransaction();
          try {
            db.execSQL("DROP TABLE IF EXISTS CHATS");
            db.setTransactionSuccessful();
          } catch (Exception e) {
            Logger.e(": failed to port data.. FOR UBOX", e);
          } finally {
            db.endTransaction();
          }
          break;
        case 6:
          db.beginTransaction();
          try {
            if (isTableExists(db, TABLE_NAME_OLD_INAPPS)) {
              String ALTER_INAPP =
                  " ALTER TABLE " + TABLE_NAME_OLD_INAPPS + " ADD COLUMN TYPE INTEGER";
              db.execSQL(ALTER_INAPP);
            }
            db.setTransactionSuccessful();
          } catch (Exception e) {
            Logger.e(TAG + ": failed to add column " + TABLE_NAME_OLD_INAPPS, e);
          } finally {
            db.endTransaction();
          }
          break;
        case 7:
          db.beginTransaction();
          try {
            createDataPointsTable(db);
            populateDatapoints(db);
            db.execSQL("DROP TABLE IF EXISTS EVENTS");
            db.setTransactionSuccessful();
          } catch (Exception e) {
            Logger.e(TAG + ": failed to populate Datapoints ", e);
          } finally {
            db.endTransaction();
          }
          break;
        case 8:
          db.beginTransaction();
          try {
            if (!isFieldExist(TABLE_NAME_MSGS, MessageEntity.MSG_TAG, db)) {
              String ALTER_INBOX = " ALTER TABLE " + TABLE_NAME_MSGS + " ADD COLUMN " +
                  MessageEntity.MSG_TAG + " TEXT";
              db.execSQL(ALTER_INBOX);
            }
            db.setTransactionSuccessful();
          } catch (Exception e) {
            Logger.e(TAG + " failed to add columns to " + TABLE_NAME_MSGS, e);
          } finally {
            db.endTransaction();
          }
          break;
        case 9:
          db.execSQL("DROP TABLE IF EXISTS INAPPS");
          createInAppTable(db);
          //this check is added for the mess up made by big basket.
          if (!isFieldExist(TABLE_NAME_MSGS, MessageEntity.MSG_TAG, db)) {
            String ALTER_INBOX = " ALTER TABLE " + TABLE_NAME_MSGS + " ADD COLUMN " +
                MessageEntity.MSG_TAG + " TEXT";
            db.execSQL(ALTER_INBOX);
          }
          break;
        case 10:
          createUserAttributeTable(db);
          break;
        case 11:
          createCampaignListTable(db);
          break;
        case 12:
          createBatchDataTable(db);
          break;
        case 13:
          createDeviceTriggerTable(db);
          break;
        case 14:
          createAttributeCacheTableIfRequired(db);
          portUserAttributeUniqueId(db);
          break;
        case 15:
          addCampaignIdToMessageTable(db);
          break;
        case 16:
          createInAppV3Table(db);
          portInAppV2ToV3(db);
          createInAppStatsTable(db);
          break;
        case 17:
          createCardsTable(db);
          break;
        default:
          Logger.v(TAG + "Failed to upgrade from DB version" + oldVersion + "to DB version" +
              newVersion);
      }
      upgradeTo++;
    }
    Logger.v(TAG + " Database Upgraded");
  }

  private void createDataPointsTable(SQLiteDatabase db) {
    String DDL_DATAPOINTS = "CREATE TABLE IF NOT EXISTS "
        + TABLE_NAME_DATAPOINTS
        + " ( "
        + DatapointEntity._ID
        + " INTEGER PRIMARY KEY, "
        + DatapointEntity.GTIME
        + " INTEGER, "
        + DatapointEntity.DETAILS
        + " TEXT ); ";
    db.execSQL(DDL_DATAPOINTS);
  }

  private void createMessagesTable(SQLiteDatabase db) {
    String DDL_MSGS = "CREATE TABLE IF NOT EXISTS "
        + TABLE_NAME_MSGS
        + " ( "
        + MessageEntity._ID
        + " INTEGER PRIMARY KEY, "
        + MessageEntity.MSG_DETAILS
        + " TEXT, "
        + MessageEntity.MSG_CLICKED
        + " INTEGER DEFAULT 0, "
        + MessageEntity.MSG_TTL
        + " INTEGER, "
        + MessageEntity.GTIME
        + " INTEGER, "
        + MessageEntity.MSG_TAG
        + " TEXT, "
        + MessageEntity.CAMPAIGN_ID
        + " TEXT "
        + ")";
    db.execSQL(DDL_MSGS);
  }

  private void createInAppTable(SQLiteDatabase db) {
    String DDL_INAPP = "CREATE TABLE IF NOT EXISTS "
        + TABLE_NAME_INAPPS
        + " ( "
        + InAppMessageEntity._ID
        + " INTEGER PRIMARY KEY, "
        + InAppMessageEntity.GTIME
        + " INTEGER, "
        + InAppMessageEntity.MSG_CAMPAIGN_ID
        + " TEXT, "
        + InAppMessageEntity.MSG_ALIGN_TYPE
        + " TEXT, "
        + InAppMessageEntity.MSG_INAPP_TYPE
        + " TEXT, "
        + InAppMessageEntity.MSG_TTL
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_MIN_DELAY
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_MAX_TIMES
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_SHOWN_COUNT
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_PERSISTENT
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_PRIORITY
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_CONTEXT
        + " TEXT, "
        + InAppMessageEntity.MSG_LAST_SHOWN
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_IS_CLICKED
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_HAS_ERRORS
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_AUTODISMISS_TIME
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_CANCELABLE
        + " INTEGER DEFAULT 0, "
        + InAppMessageEntity.MSG_CONTENT
        + " TEXT, "
        + InAppMessageEntity.MSG_SHOW_ONLY_IN
        + " TEXT, "
        + InAppMessageEntity.MSG_STATUS
        + " TEXT, "
        + InAppMessageEntity.MSG_CONTAINER_STYLE
        + " TEXT );";
    db.execSQL(DDL_INAPP);
  }

  public void addMSGTagIfRequiredInbox(SQLiteDatabase db) {
    if (!isFieldExist(TABLE_NAME_MSGS, MessageEntity.MSG_TAG, db)) {
      Logger.v(TAG + " : addMSGTagIfRequiredInbox : updating inbox table");
      String ALTER_UBOX = " ALTER TABLE " + TABLE_NAME_MSGS + " ADD COLUMN " +
          MessageEntity.MSG_TAG + " TEXT";
      db.execSQL(ALTER_UBOX);
    }
  }

  private void createUserAttributeTable(SQLiteDatabase db) {
    String DDL_USER_ATTRIBUTES = "CREATE TABLE IF NOT EXISTS "
        + TABLE_NAME_USER_ATTRIBUTES
        + " ( "
        + UserAttributeEntity._ID
        + " INTEGER PRIMARY KEY, "
        + UserAttributeEntity.ATTRIBUTE_NAME
        + " TEXT, "
        + UserAttributeEntity.ATTRIBUTE_VALUE
        + " TEXT"
        + " ); ";
    db.execSQL(DDL_USER_ATTRIBUTES);
  }

  private void createCampaignListTable(SQLiteDatabase db) {
    String DDL_CAMPAIGN_LIST = "CREATE TABLE IF NOT EXISTS "
        + TABLE_NAME_CAMPAIGN_LIST
        + " ( "
        + CampaignListEntity._ID
        + " INTEGER PRIMARY KEY, "
        + CampaignListEntity.CAMPAIGN_ID
        + " TEXT, "
        + CampaignListEntity.CAMPAIGN_ID_TTL
        + " INTEGER"
        + " );";
    db.execSQL(DDL_CAMPAIGN_LIST);
  }

  private void createBatchDataTable(SQLiteDatabase db) {
    String DDL_BATCH_DATA = "CREATE TABLE IF NOT EXISTS "
        + TABLE_NAME_BATCH_DATA
        + " ( "
        + BatchDataEntity._ID
        + " INTEGER PRIMARY KEY, "
        + BatchDataEntity.BATCHED_DATA
        + " TEXT"
        + " );";
    db.execSQL(DDL_BATCH_DATA);
  }

  void createDeviceTriggerTable(SQLiteDatabase db) {
    String DDL_DEVICE_TRIGGERS = "CREATE TABLE IF NOT EXISTS "
        + TABLE_NAME_DEVICE_TRIGGERS
        + " ( "
        + DTEntity._ID
        + " INTEGER PRIMARY KEY, "
        + DTEntity.CAMPAIGN_ID
        + " TEXT, "
        + DTEntity.TRIGGER_EVENT_NAME
        + " TEXT, "
        + DTEntity.PAYLOAD
        + " TEXT, "
        + DTEntity.CAMPAIGN_PAYLOAD
        + " TEXT, "
        + DTEntity.CAMPAIGN_TYPE
        + " TEXT, "
        + DTEntity.MAX_COUNT
        + " INTEGER DEFAULT 0, "
        + DTEntity.MINIMUM_DELAY
        + " INTEGER DEFAULT 0, "
        + DTEntity.SHOULD_SHOW_OFFLINE
        + " INTEGER DEFAULT 0, "
        + DTEntity.MAX_SYNC_DELAY_TIME
        + " INTEGER DEFAULT 0, "
        + DTEntity.EXPIRY_TIME
        + " INTEGER, "
        + DTEntity.PRIORITY
        + " INTEGER DEFAULT 3, "
        + DTEntity.LAST_SHOW_TIME
        + " INTEGER DEFAULT 0, "
        + DTEntity.SHOW_COUNT
        + " INTEGER DEFAULT 0, "
        + DTEntity.LAST_UPDATED_TIME
        + " INTEGER DEFAULT 0, "
        + DTEntity.STATUS
        + " TEXT, "
        + DTEntity.SHOULD_IGNORE_DND
        + " INTEGER DEFAULT 0, "
        + DTEntity.DELAY_BEFORE_SHOWING_NOTIFICATION
        + " INTEGER DEFAULT 0 "
        + " ) ";
    db.execSQL(DDL_DEVICE_TRIGGERS);
  }

  void createAttributeCacheTableIfRequired(SQLiteDatabase database) {
    String DDL_ATTRIBUTE_CACHE = "CREATE TABLE IF NOT EXISTS "
        + TABLE_NAME_ATTRIBUTE_CACHE
        + " ( "
        + AttributeCacheEntity._ID
        + " INTEGER PRIMARY KEY, "
        + AttributeCacheEntity.ATTRIBUTE_NAME
        + " TEXT, "
        + AttributeCacheEntity.ATTRIBUTE_VALUE
        + " TEXT, "
        + AttributeCacheEntity.LAST_TRACKED_TIME
        + " INTEGER DEFAULT 0, "
        + AttributeCacheEntity.DATA_TYPE
        + " TEXT "
        + " ) ";
    database.execSQL(DDL_ATTRIBUTE_CACHE);
  }

  void createInAppV3Table(SQLiteDatabase db) {
    String DDL_INAPP_V3 = "CREATE TABLE IF NOT EXISTS "
        + TABLE_NAME_INAPP_V3
        + " ( "
        + InAppV3Entity._ID
        + " INTEGER PRIMARY KEY, "
        + InAppV3Entity.CAMPAIGN_ID
        + " TEXT, "
        + InAppV3Entity.CAMPAIGN_TYPE
        + " TEXT, "
        + InAppV3Entity.CAMPAIGN_STATUS
        + " TEXT, "
        + InAppV3Entity.CAMPAIGN_STATE
        + " TEXT, "
        + InAppV3Entity.PRIORITY
        + " INTEGER, "
        + InAppV3Entity.LAST_UPDATED_TIME
        + " INTEGER, "
        + InAppV3Entity.TEMPLATE_TYPE
        + " TEXT, "
        + InAppV3Entity.DELETION_TIME
        + " INTEGER, "
        + InAppV3Entity.LAST_RECEIVED_TIME
        + " INTEGER DEFAULT 0, "
        + InAppV3Entity.CAMPAIGN_META
        + " TEXT "
        + " ) ";
    db.execSQL(DDL_INAPP_V3);
  }

  void createInAppStatsTable(SQLiteDatabase database) {
    String DDL_INAPP_STATS = "CREATE TABLE IF NOT EXISTS "
        + TABLE_NAME_INAPP_STATS
        + " ( "
        + InAppStatsEntity._ID
        + " INTEGER PRIMARY KEY, "
        + InAppStatsEntity.TIMESTAMP
        + " INTEGER, "
        + InAppStatsEntity.PAYLOAD
        + " TEXT, "
        + InAppStatsEntity.REQUEST_ID
        + " TEXT "
        + " ) ";
    database.execSQL(DDL_INAPP_STATS);
  }

  void createCardsTable(SQLiteDatabase database){
    String DDL_CARDS = "CREATE TABLE IF NOT EXISTS "
        + TABLE_NAME_CARD_CAMPAIGNS
        + " ( "
        + CardsEntity._ID
        + " INTEGER PRIMARY KEY, "
        + CardsEntity.CARD_ID
        + " TEXT, "
        + CardsEntity.CATEGORY
        + " TEXT, "
        + CardsEntity.CAMPAIGN_STATE
        + " TEXT, "
        + CardsEntity.VISIBILITY_STATUS
        + " TEXT, "
        + CardsEntity.LAST_UPDATED_TIME
        + " INTEGER, "
        + CardsEntity.CAMPAIGN_PAYLOAD
        + " TEXT, "
        + CardsEntity.IS_PINNED
        + " INTEGER, "
        + CardsEntity.DELETION_TIME
        + " INTEGER, "
        + CardsEntity.IS_NEW_CARD
        + " INTEGER, "
        + CardsEntity.IS_DELETED
        + " INTEGER DEFAULT 0, "
        + CardsEntity.PRIORITY
        + " INTEGER DEFAULT 0 "
        + " ) ";
    database.execSQL(DDL_CARDS);
  }

  void addCampaignIdToMessageTable(SQLiteDatabase db) {
    if (!isFieldExist(TABLE_NAME_MSGS, MessageEntity.CAMPAIGN_ID, db)) {
      Logger.v(TAG + " addMSGTagIfRequiredInbox()  updating inbox table");
      String ALTER_MESSAGES_TABLE = " ALTER TABLE " + TABLE_NAME_MSGS + " ADD COLUMN " +
          MessageEntity.CAMPAIGN_ID + " TEXT";
      db.execSQL(ALTER_MESSAGES_TABLE);
    }
  }

  private void portUserAttributeUniqueId(SQLiteDatabase db) {
    try {
      Logger.v(TAG + " portUserAttributeUniqueId(): Will try to port USER_ATTRIBUTE_UNIQUE_ID "
          + "to another storage.");
      if (!tableExists(db, TABLE_NAME_USER_ATTRIBUTES)) {
        Logger.v(TAG + " portUserAttributeUniqueId() : User attribute table does not exist");
        return;
      }
      // fetch value from table
      db.beginTransaction();
      String QUERY_GET_USER_ATTRIBUTE_UNIQUE_ID = "SELECT "
          + UserAttributeEntity.ATTRIBUTE_NAME + ", "
          + UserAttributeEntity.ATTRIBUTE_VALUE + " "
          + "FROM " + TABLE_NAME_USER_ATTRIBUTES
          + " WHERE " + UserAttributeEntity.ATTRIBUTE_NAME + " = ?";
      Cursor cursor = db.rawQuery(QUERY_GET_USER_ATTRIBUTE_UNIQUE_ID,
          new String[] { MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID });
      if (cursor == null) {
        Logger.v(TAG + " portUserAttributeUniqueId(): USER_ATTRIBUTE_UNIQUE_ID not present. "
            + "Cursor is null");
        return;
      }
      if (!cursor.moveToFirst()) {
        Logger.v(TAG + " portUserAttributeUniqueId(): USER_ATTRIBUTE_UNIQUE_ID not present. "
            + "Cursor does not have a 1st element.");
        return;
      }
      String uniqueId = cursor.getString(1);
      cursor.close();
      // copy existing value to shared preference
      ConfigurationProvider.getInstance(context).saveUserAttributeUniqueId(uniqueId);
      // copy existing value to attribute cache table
      if (!tableExists(db, TABLE_NAME_ATTRIBUTE_CACHE)) {
        Logger.v(TAG + " portUserAttributeUniqueId() : Attribute Cache does not exist. Cannot "
            + "port data");
        return;
      }
      ContentValues contentValue = new ContentValues();
      contentValue.put(AttributeCacheEntity.ATTRIBUTE_NAME,
          MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID);
      contentValue.put(AttributeCacheEntity.ATTRIBUTE_VALUE, uniqueId);
      contentValue.put(AttributeCacheEntity.LAST_TRACKED_TIME, 0);
      contentValue.put(AttributeCacheEntity.DATA_TYPE, DataTypes.STRING.toString());
      db.insert(TABLE_NAME_ATTRIBUTE_CACHE, null, contentValue);
      db.setTransactionSuccessful();
    } catch (Exception e) {
      Logger.e(TAG + " portUserAttributeUniqueId(): Exception ", e);
    } finally {
      db.endTransaction();
    }
  }

  private void portDataFromv2(SQLiteDatabase db) {
    try {
      db.beginTransaction();

      //no need to port events data might cause issues

      String QUERY_GET_MSGS = "SELECT * FROM moemsgs";
      Cursor cursor = db.rawQuery(QUERY_GET_MSGS, null);
      if (null != cursor && cursor.moveToFirst()) {
        do {
          ContentValues values = new ContentValues();
          values.put(MessageEntity.MSG_DETAILS, cursor.getString(1));
          values.put(MessageEntity.MSG_CLICKED, cursor.getInt(2));
          values.put(MessageEntity.MSG_TTL, Long.parseLong(cursor.getString(3)));
          values.put(MessageEntity.GTIME, Long.parseLong(cursor.getString(4)));
          long row = db.insert(TABLE_NAME_MSGS, null, values);
          Logger.v(":onUpgrade: Porting message data: " + row);
        } while (cursor.moveToNext());
        cursor.close();
        cursor = null;
      }

      //removed porting code for inapps
      db.setTransactionSuccessful();
    } catch (Exception e) {
      Logger.e(TAG + " portDatafromv2", e);
    } finally {
      if (null != db) {
        db.endTransaction();
      }
    }
  }

  private void populateDatapoints(SQLiteDatabase db) {
    if (null == db) {
      return;
    }
    try {
      Logger.d(TAG + "Started porting DATA - FOR DATAPOINTS");
      db.beginTransaction();
      String QUERY_GET_EVENTS = "SELECT _id, action, attrs, gtime, ltime FROM EVENTS";
      Cursor cursor = db.rawQuery(QUERY_GET_EVENTS, null);
      if (null != cursor && cursor.moveToFirst()) {
        do {
          ContentValues values = new ContentValues();
          JSONObject details = MoEHelperUtils.getDatapointJSON(cursor.getString(1),
              new JSONObject(cursor.getString(2)), Long.toString(cursor.getLong(3)),
              cursor.getString(4));
          if (null != details) {
            values.put(DatapointEntity.DETAILS, details.toString());
            values.put(DatapointEntity.GTIME, cursor.getLong(3));
            long row = db.insert(MoEDatabaseHelper.TABLE_NAME_DATAPOINTS, null, values);
            Logger.v(TAG + " Porting event data: " + row);
          }
        } while (cursor.moveToNext());
        cursor.close();
      }
      db.setTransactionSuccessful();
    } catch (Exception e) {
      Logger.e(TAG + ": populateDatapoints", e);
    } finally {
      db.endTransaction();
    }
  }

  public boolean isFieldExist(String tableName, String fieldName, SQLiteDatabase db) {
    boolean isExist = false;
    Cursor res = db.rawQuery("PRAGMA table_info(" + tableName + ")", null);
    ArrayList<String> columnNameList = new ArrayList<>();
    try {
      if (res != null && res.moveToFirst()) {
        do {
          String columnName = res.getString(res.getColumnIndex("name"));
          columnNameList.add(columnName);
        } while (res.moveToNext());
      }
    } finally {
      if (res != null) {
        res.close();
      }
    }

    if (columnNameList.contains(fieldName)) {
      isExist = true;
    }
    return isExist;
  }

  private void portInAppV2ToV3(SQLiteDatabase db) {
    try {
      ConfigurationProvider.getInstance(context)
          .setLastInAppShownTime(
              ConfigurationProvider.getInstance(context).getLastInAppShownTime() / 1000);
      if (!tableExists(db, TABLE_NAME_INAPP_V3)) {
        Logger.e(TAG + " portInAppV2ToV3() : InAppV3 table does not exist. Cannot "
            + "migrate data");
        return;
      }
      if (!tableExists(db, TABLE_NAME_INAPPS)) {
        Logger.e(TAG + " portInAppV2ToV3(): InAppV2 table does not exist. Cannot migrate"
            + " data");
        return;
      }
      String QUERY_IN_APP_V2 = "SELECT "
          + InAppMessageEntity.MSG_CAMPAIGN_ID
          + ", "
          + InAppMessageEntity.MSG_SHOWN_COUNT
          + ", "
          + InAppMessageEntity.MSG_LAST_SHOWN
          + ", "
          + InAppMessageEntity.MSG_IS_CLICKED
          + ", "
          + InAppMessageEntity.MSG_STATUS
          + ", "
          + InAppMessageEntity.MSG_TTL
          + ", "
          + InAppMessageEntity.MSG_PRIORITY
          + " FROM "
          + TABLE_NAME_INAPPS;
      db.beginTransaction();

      Cursor cursor = db.rawQuery(QUERY_IN_APP_V2, null);
      if (cursor != null && cursor.moveToFirst()) {
        do {
          ContentValues values = new ContentValues();

          String campaignId = cursor.getString(0);
          long expiryTime = cursor.getLong(5) / 1000;

          InAppV2Meta inAppV2Meta =
              new InAppV2Meta(campaignId, cursor.getLong(5), cursor.getLong(6),
                  cursor.getLong(1), cursor.getLong(2), cursor.getInt(3));

          InAppV3Meta inAppV3Meta =
              InAppManager.getInstance().generateMetaForV2Campaign(inAppV2Meta);

          if (inAppV3Meta == null || inAppV3Meta.campaignState == null || inAppV3Meta.campaignMeta
              == null) {
            Logger.v(TAG + " portInAppV2ToV3() : Could generate in-appv3 payload. Will not "
                + "migrate the campaign.");
            continue;
          }

          long deletionTime = expiryTime > MoEUtils.currentSeconds() + 5184000L ? expiryTime
              : MoEUtils.currentSeconds() + 5184000L;

          values.put(InAppV3Entity.CAMPAIGN_STATE, inAppV3Meta.campaignState.toString());
          values.put(InAppV3Entity.CAMPAIGN_META, inAppV3Meta.campaignMeta.toString());
          values.put(InAppV3Entity.CAMPAIGN_ID, campaignId);
          values.put(InAppV3Entity.CAMPAIGN_STATUS, "IN_ACTIVE");
          values.put(InAppV3Entity.CAMPAIGN_TYPE, "general");
          values.put(InAppV3Entity.DELETION_TIME, deletionTime);

          db.insert(TABLE_NAME_INAPP_V3, null, values);
        } while (cursor.moveToNext());
      }
      if (cursor != null) {
        cursor.close();
      }
      db.setTransactionSuccessful();
    } catch (Exception e) {
      Logger.e(TAG + " portInAppV2ToV3() : Exception ", e);
    } finally {
      db.endTransaction();
    }
  }

  void addTableIfRequired(SQLiteDatabase db, String tableName){
    try {
      if (isTableExists(db, tableName)){
        Logger.v(TAG + " addTableIfRequired() : creating missing table: " + tableName);
        createTable(db, tableName);
      }else {
        Logger.v(TAG + " addTableIfRequired() : " + tableName + " already present.");
      }
    } catch (Exception e) {
      Logger.e( TAG + " addTableIfRequired() : ", e);
    }
  }

  private void createTable(SQLiteDatabase db, String tableName){
    switch (tableName){
      case TABLE_NAME_USER_ATTRIBUTES:
        createUserAttributeTable(db);
        break;
      case TABLE_NAME_CAMPAIGN_LIST:
        createCampaignListTable(db);
        break;
      case TABLE_NAME_BATCH_DATA:
        createBatchDataTable(db);
        break;
      case TABLE_NAME_DEVICE_TRIGGERS:
        createDeviceTriggerTable(db);
        break;
      case TABLE_NAME_INAPP_V3:
        createInAppV3Table(db);
        break;
      case TABLE_NAME_ATTRIBUTE_CACHE:
        createAttributeCacheTableIfRequired(db);
        break;
      case TABLE_NAME_INAPP_STATS:
        createInAppStatsTable(db);
        break;
      case TABLE_NAME_CARD_CAMPAIGNS:
        createCardsTable(db);
        break;
    }
  }

}