/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moe.pushlibrary;

import android.app.Activity;
import android.app.Application;
import android.content.Context;
import android.content.Intent;
import android.database.Cursor;
import android.location.Location;
import android.os.Bundle;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.RestrictTo;
import android.support.annotation.RestrictTo.Scope;
import android.support.annotation.WorkerThread;
import android.text.TextUtils;
import com.moe.pushlibrary.models.GeoLocation;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.core.Logger;
import com.moengage.core.MoECallbacks;
import com.moengage.core.MoEDTManager;
import com.moengage.core.MoEDispatcher;
import com.moengage.core.MoEUtils;
import com.moengage.core.Properties;
import com.moengage.core.RemoteConfig;
import com.moengage.core.TrackInstallUpdateTask;
import com.moengage.core.events.MoEEventManager;
import com.moengage.core.inapp.InAppManager;
import com.moengage.core.integrations.segment.TrackSegmentEventTask;
import com.moengage.core.integrations.segment.TrackSegmentUserAttributeTask;
import com.moengage.core.listeners.OnAppBackgroundListener;
import com.moengage.core.listeners.OnLogoutCompleteListener;
import com.moengage.core.model.AppStatus;
import com.moengage.core.model.UserGender;
import com.moengage.core.userattributes.MoEAttributeManager;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.json.JSONObject;

/**
 * Interaction class for MoEngage SDK. Has helper methods to track events and user attributes.
 *
 * @author MoEngage (abhishek@moengage.com)
 * @version 5.0
 * @since 1.0
 */
public class MoEHelper {

  /**
   * The entire task is delegated to the controller
   */
  private MoEDispatcher dispatcher = null;
  /**
   * Bundle extra which denotes that the activity was recreated for
   * configuration change and this should be accounted for when tracking
   */
  private String BUNDLE_EXTRA_RESTORING = "EXTRA_RESTORING";
  /**
   * Denotes that the activity is being restored from a previous state and it
   * should be tracked again for start
   */
  private boolean isActivityBeingRestored = false;
  /**
   * An instance of the application {@link Context}
   */
  private Context context;
  /**
   * Maintains the visible activity count
   */
  private static int activityCounter = 0;

  private static String TAG = "MoEHelper";

  private boolean isFirstScreen;

  public static int getActivityCounter(){
    return activityCounter;
  }

  /**
   * Increment the activity counter.
   */
  private static synchronized void incrementCounter() {
    activityCounter++;
  }

  /**
   * Decrement the activity counter.
   */
  private static synchronized void decrementCounter() {
    activityCounter--;
  }

  /**
   * Checks whether the app is in foreground or not.
   * If any screen is not being tracked then,
   * on that screen it would not return the correct value
   *
   * @return true if the app is in foreground else returns false
   */
  public static boolean isAppInForeground() {
    return activityCounter > 0;
  }

  private MoEAttributeManager attributeManager = null;

  /**
   * Instantiates an instance of {@link MoEHelper}.
   * Use {@link MoEHelper#getInstance(Context)}
   * @param con Reference to the Application {@link Context}
   */
  private MoEHelper(Context con) {
    context = con.getApplicationContext();
    if (null == dispatcher) {
      dispatcher = getDelegate();
    }
    attributeManager = getAttributeManager();
    instance = this;
  }

  /*
   * This is for backward porting the exposed API to reduce the number of
   * object created.
   */
  private static MoEHelper instance = null;

  /**
   * Returns a new Instance of {@link MoEHelper} for every call.
   *
   * @param context Application Context Reference. Should be a not NULL value
   * @return a new instance of MoEHelper
   */
  public static MoEHelper getInstance(@NonNull Context context) {
    if (instance == null) {
      synchronized (MoEHelper.class) {
        if (instance == null) instance = new MoEHelper(context);
      }
    }
    return instance;
  }

  @RestrictTo(Scope.LIBRARY_GROUP)
  public void initialize(String senderId, String appId) {
    dispatcher.initialize(senderId, appId);
  }

	/* ********************** Activity call backs ************************** */

  /**
   * Integration point for on start of the {@link Activity} to which
   * {@link MoEHelper} is attached to.
   *
   * @param activity The current {@link Activity} to which this instance is
   * attached to
   * @param intent The new intent delivered to this activity
   */
  public void onNewIntent(Activity activity, Intent intent) {
    if(!isActivityBeingRestored){
      dispatcher.onStart(activity, intent);
    }
    InAppManager.getInstance().registerInAppManager(activity);
  }

  @RestrictTo(Scope.LIBRARY_GROUP)
  public void onStartInternal(@NonNull Activity activity){
    if (getActivityCounter() == 0){
      dispatcher.onAppOpen();
      setFirstScreen(true);
    }
    incrementCounter();
    context = activity.getApplicationContext();
    onNewIntent(activity, null);
  }

  @RestrictTo(Scope.LIBRARY_GROUP)
  public void onStopInternal(@NonNull Activity activity) {
    Logger.v("Activity onStop called for " + activity.toString());
    decrementCounter();
    InAppManager.getInstance().unregisterInAppManager(activity);
    dispatcher.onStop(activity);
  }

  @RestrictTo(Scope.LIBRARY_GROUP)
  public void onResumeInternal(Activity activity){
    if (null == context) {
      context = activity.getApplicationContext();
    }
    Logger.v("Activity onResume called for " + activity.toString());
    dispatcher.onResume(activity, isActivityBeingRestored);
    isActivityBeingRestored = false;
  }

	/* ********************** Handling config change ************************ */

  /**
   * Called to retrieve per-instance state from an activity before being
   * killed so that the activity is not tracked twice in MoEngage Platform.
   * Also helps to show the In App dialog even after the Configuration change
   *
   * @param outState Bundle in which to place your saved state.
   */
  public void onSaveInstanceState(Bundle outState) {
      Logger.v(TAG + " onSaveInstanceState-- saving state");
      if (null == outState) return;
      outState.putBoolean(BUNDLE_EXTRA_RESTORING, true);
   }

  /**
   * Called to restore the per instance state of the Activity so that it is
   * not tracked twice and also to restore the In App PromotionalMessage if it was being
   * shown before the configuration change occurred
   *
   * @param savedInstanceState the data most recently supplied in
   * {@link #onSaveInstanceState}.
   */
  public void onRestoreInstanceState(Bundle savedInstanceState) {
    Logger.v(TAG + " onRestoreInstanceState-- restoring state");
    if (null == savedInstanceState) return;
    if (savedInstanceState.containsKey(BUNDLE_EXTRA_RESTORING)) {
      isActivityBeingRestored = true;
      savedInstanceState.remove(BUNDLE_EXTRA_RESTORING);
    }
  }

  /**
   * Use {@link MoEHelper#trackEvent(String, PayloadBuilder)}
   */
  @Deprecated public MoEHelper trackEvent(@NonNull final String action, JSONObject attrs) {
    if (TextUtils.isEmpty(action)) return this;
    MoEEventManager.getInstance(context).trackEvent(action, attrs);
    return this;
  }

  /**
   * Use {@link MoEHelper#trackEvent(String, PayloadBuilder)}
   */
  @Deprecated
  public MoEHelper trackEvent(@NonNull final String action, @NonNull Map<String, String> attrs) {
    Set<String> keySet = attrs.keySet();
    if (keySet.isEmpty()) {
      return this;
    }
    PayloadBuilder newAttrs = new PayloadBuilder();
    for (String key : keySet) {
      try {
        newAttrs.putAttrString(key, attrs.get(key));
      } catch (Exception e) {
        Logger.e(TAG + " trackEvent", e);
      }
    }
    trackEvent(action, newAttrs);
    return this;
  }

  /**
   * Use {@link MoEHelper#trackEvent(String, PayloadBuilder)}
   */
  @Deprecated
  public MoEHelper trackEvent(@NonNull String action){
    if (TextUtils.isEmpty(action)) return this;
    trackEvent(action.trim(), new PayloadBuilder());
    return this;
  }

  /**
   * Use {@link MoEHelper#trackEvent(String, Properties)}
   */
  @Deprecated public MoEHelper trackEvent(@NonNull String action, PayloadBuilder payloadBuilder) {
    if (TextUtils.isEmpty(action)) {
      Logger.e(TAG + " trackEvent() : Action name cannot be null");
      return this;
    }
    if (payloadBuilder == null) {
      payloadBuilder = new PayloadBuilder();
    } else {
      MoEEventManager.getInstance(context).trackEvent(action, payloadBuilder);
    }
    return this;
  }

  public void trackEvent(@NonNull String eventName, Properties properties){
    if (MoEUtils.isEmptyString(eventName)) return;
    if (properties == null) properties = new Properties();
    MoEEventManager.getInstance(context).trackEvent(eventName, properties.getPayload());
  }

  /**
   * Set an user attribute. This can be used to segment users on MoEngage
   * platform.
   *
   * @param userAttribute The user attribute name which needs to be added. This field
   * cannot be null
   * @param attributeValue The value associated with the specified user attribute
   * @return {@link MoEHelper} instance
   */
  public MoEHelper setUserAttribute(@NonNull String userAttribute, @NonNull String attributeValue) {
    if (userAttribute == null) {
      Logger.e(TAG + " User attribute value cannot be null");
      return this;
    }
    try {
      if (attributeValue == null) {
        attributeValue = "";
      } else if (MoEHelperConstants.USER_ATTRIBUTE_USER_BDAY.equals(userAttribute)) {
        attributeValue = URLEncoder.encode(attributeValue, "UTF-8");
      }
    } catch (UnsupportedEncodingException e1) {
      Logger.e(TAG + " setUserAttribute", e1);
    }catch (Exception e){
      Logger.e(TAG + " setUserAttribute", e);
    }
    JSONObject userJson = new JSONObject();
    try {
      userJson.put(userAttribute.trim(), attributeValue);
      attributeManager.setUserAttribute(userJson);
    } catch (Exception e) {
      Logger.e(TAG + " setUserAttribute", e);
    }
    return this;
  }

  /**
   * Set a user attribute for the current user
   *
   * @param userAttribute The attribute which needs to be set
   * @param attributeValue The attribute value corresponding to the userAttribute
   * @return {@link MoEHelper} instance
   */
  public MoEHelper setUserAttribute(@NonNull String userAttribute, int attributeValue) {
    if (userAttribute == null) {
      Logger.e(TAG + " User attribute value cannot be null");
      return this;
    }
    JSONObject userJson = new JSONObject();
    try {
      userJson.put(userAttribute.trim(), attributeValue);
      attributeManager.setUserAttribute(userJson);
    } catch (Exception e) {
      Logger.e(TAG + " setUserAttribute", e);
    }
    return this;
  }

  /**
   * Set a user attribute for the current user
   *
   * @param userAttribute The attribute which needs to be set
   * @param attributeValue The attribute value corresponding to the userAttribute
   * @return {@link MoEHelper} instance
   */
  public MoEHelper setUserAttribute(@NonNull String userAttribute, boolean
      attributeValue) {
    if (userAttribute == null) {
      Logger.e(TAG + " User attribute value cannot be null");
      return this;
    }
    JSONObject userJson = new JSONObject();
    try {
      userJson.put(userAttribute.trim(), attributeValue);
      attributeManager.setUserAttribute(userJson);
    } catch (Exception e) {
      Logger.e(TAG + " setUserAttribute", e);
    }
    return this;
  }

  /**
   * Set a user attribute for the current user
   *
   * @param userAttribute The attribute which needs to be set
   * @param attributeValue The attribute value corresponding to the userAttribute
   * @return {@link MoEHelper} instance
   */
  @SuppressWarnings("ConstantConditions")
  public MoEHelper setUserAttribute(@NonNull String userAttribute, double attributeValue) {
    if (userAttribute == null) {
      Logger.e(TAG + " User attribute value cannot be null");
      return this;
    }
    JSONObject userJson = new JSONObject();
    try {
      userJson.put(userAttribute.trim(), attributeValue);
      attributeManager.setUserAttribute(userJson);
    } catch (Exception e) {
      Logger.e(TAG + " setUserAttribute", e);
    }
    return this;
  }

  /**
   * Set a user attribute for the current user
   *
   * @param userAttribute The attribute which needs to be set
   * @param attributeValue The attribute value corresponding to the userAttribute
   * @return {@link MoEHelper} instance
   */
  @SuppressWarnings("ConstantConditions")
  public MoEHelper setUserAttribute(@NonNull String userAttribute, float attributeValue) {
    if (userAttribute == null) {
      Logger.e(TAG + " User attribute value cannot be null");
      return this;
    }
    JSONObject userJson = new JSONObject();
    try {
      userJson.put(userAttribute.trim(), attributeValue);
      attributeManager.setUserAttribute(userJson);
    } catch (Exception e) {
      Logger.e(TAG + " setUserAttribute", e);
    }
    return this;
  }

  /**
   * Set a user attribute for the current user
   *
   * @param userAttribute The attribute which needs to be set
   * @param attributeValue The attribute value corresponding to the userAttribute
   * @return {@link MoEHelper} instance
   */
  @SuppressWarnings("ConstantConditions")
  public MoEHelper setUserAttribute(@NonNull String userAttribute, long attributeValue) {
    if (userAttribute == null) {
      Logger.e(TAG + " User attribute value cannot be null");
      return this;
    }
    JSONObject userJson = new JSONObject();
    try {
      userJson.put(userAttribute.trim(), attributeValue);
      attributeManager.setUserAttribute(userJson);
    } catch (Exception e) {
      Logger.e(TAG + " setUserAttribute", e);
    }
    return this;
  }

  /**
   * Sets the given epoch time for the given user.
   *
   * @param userAttribute user attribute name
   * @param attributeValue user attribute value
   * @return {@link MoEHelper} instance
   */
  @SuppressWarnings("ConstantConditions")
  public MoEHelper setUserAttributeEpochTime(@NonNull String userAttribute, long attributeValue){
    if (userAttribute == null) {
      Logger.e(TAG + " User attribute value cannot be null");
      return this;
    }
    attributeManager.setCustomUserAttribute(new PayloadBuilder().putAttrDateEpoch(userAttribute,
        attributeValue).build());
    return this;
  }

  /**
   * Set the user attributes with the attributes specified in the HashMap
   *
   * @param attributeMap The attribute map which needs to be set for the user
   * @return {@link MoEHelper} instance
   */
  public MoEHelper setUserAttribute(@NonNull Map<String, Object> attributeMap) {
    if (attributeMap.isEmpty()) {
      Logger.e(TAG + " User attribute map cannot be null or empty");
      return this;
    }
    Set<String> keySet = attributeMap.keySet();
    if ( keySet.isEmpty() ) {
      return this;
    }
    for (String key : keySet) {
      try {
        if(TextUtils.isEmpty(key))continue;
        Object attrValue = attributeMap.get(key);
        if( attrValue instanceof Date){
          setUserAttribute(key.trim(), (Date)attrValue);
          continue;
        }else if( attrValue instanceof GeoLocation ){
          setUserAttribute(key.trim(), (GeoLocation)attrValue);
          continue;
        }else if( attrValue instanceof Location ){
          setUserAttribute(key.trim(), (Location)attrValue);
          continue;
        }
        JSONObject userJson = new JSONObject();
        userJson.put(key.trim(), attrValue);
        attributeManager.setUserAttribute(userJson);
      } catch (Exception e) {
        Logger.e(TAG + " setUserAttribute", e);
      }
    }
    return this;
  }

  /**
   * Set a date user attribute for the current user.
   *
   * @param name user attribute name.
   * @param value user attribute value.
   * @return {@link MoEHelper} instance
   */
  public MoEHelper setUserAttribute(@NonNull String name, @NonNull Date value){
    if (MoEUtils.isEmptyString(name)) {
      Logger.e(TAG + " User attribute value cannot be null");
      return this;
    }
    attributeManager.setCustomUserAttribute(new PayloadBuilder().putAttrDate(name, value).build());
    return this;
  }

  public MoEHelper setUserAttribute(@NonNull String name, @NonNull String date, String dateFormat){
    if (MoEUtils.isEmptyString(name)) {
      Logger.e(TAG + " User attribute value cannot be null");
      return this;
    }
    attributeManager.setCustomUserAttribute(new PayloadBuilder().putAttrDate(name, date, dateFormat).build());
    return this;
  }

  public MoEHelper setUserAttribute(@NonNull String name, @NonNull Location location){
    if (MoEUtils.isEmptyString(name)) {
      Logger.e(TAG + " User attribute value cannot be null");
      return this;
    }
    attributeManager.setCustomUserAttribute(new PayloadBuilder().putAttrLocation(name, location).build());
    return this;
  }

  public MoEHelper setUserAttribute(@NonNull String name, @NonNull GeoLocation location){
    if (MoEUtils.isEmptyString(name)) {
      Logger.e(TAG + " User attribute value cannot be null");
      return this;
    }
    attributeManager.setCustomUserAttribute(new PayloadBuilder().putAttrLocation(name, location).build());
    return this;
  }

  /**
   * To set user location, call this method
   *
   * @param lat The latitude of the user location
   * @param lng The longitude of the user location
   */
  public void setUserLocation(double lat, double lng) {
    setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_USER_LOCATION, new GeoLocation(lat, lng));
  }

  /**
   * Returns the Inbox unread message count Don't call on main thread
   *
   * @return The count of unread messages
   */
  @WorkerThread
  public int getUnreadMessagesCount() {
    return dispatcher.getUnreadMessageCount();
  }


  /**
   * Retrieves all Inbox messages from the DB<br>
   * <b>Note : Don't call on UI Thread</b>.
   * @param context An instance of the application {@link Context}
   * @return A {@link Cursor} with all the existing messages
   */
  @Nullable @WorkerThread
  public static Cursor getAllMessages(Context context) {
    return MoEDispatcher.getInstance(context).getAllMessages();
  }

  /**
   * Logs that the message with the provided id was clicked
   *
   * @param context An instance of the application {@link Context}
   * @param id The id associated with the message which was clicked
   */
  @WorkerThread
  public static void setMessageClicked(Context context, long id) {
    MoEDispatcher.getInstance(context).setInboxMessageClicked(id);
  }

  /**
   * Invalidates the existing sessions and user attributes and treats all
   * actions performed by the user as a new user after this method is called<br>
   * If the Application is doing a self registration for gcm token then invalidate the token on
   * logout
   */
  public void logoutUser() {
    if (null == context) return;
    dispatcher.logoutUser(false);
  }

  /**
   * Will request a data sync with MoEngage Platform.
   */
  public void syncInteractionDataNow() {
    dispatcher.sendInteractionData();
  }


  public void trackNotificationClickedByTime(long gtime){
    dispatcher.trackNotificationClicked(gtime);
  }

  @RestrictTo(Scope.LIBRARY)
  public void registerActivityLifecycle(Application application) {

    Logger.d("MoEHelper: Auto integration is enabled");
    if (application == null) {
      Logger.e(TAG
          + " registerActivityLifecycle() : Cannot resiter for lifecycle callbacks. Application "
          + "instance is null.");
      return;
    }
    this.application = application;
    if (lifeCycleCallBacks == null) {
      lifeCycleCallBacks = new MoEActivityLifeCycleCallBacks();
      application.registerActivityLifecycleCallbacks(lifeCycleCallBacks);
    }
  }

  private MoEActivityLifeCycleCallBacks lifeCycleCallBacks;

  /**
   * Method to unregister lifecycle callbacks
   *
   * @param application reference to {@link Application}
   */
  @SuppressWarnings("ConstantConditions")
  public void unregisterLifecycleCallbacks(@NonNull Application application) {
    if (lifeCycleCallBacks != null && application != null) {
      Logger.v(TAG + " unregisterLifecycleCallbacks() : un-registering for lifecycle callbacks");
      application.unregisterActivityLifecycleCallbacks(lifeCycleCallBacks);
    } else {
      Logger.v(TAG + " unregisterLifecycleCallbacks() : either lifecycle callbacks or "
          + "application class instance is null cannot unregister.");
    }
  }


  public MoEDispatcher getDelegate(){
    if( null == dispatcher){
      dispatcher = MoEDispatcher.getInstance(context);
    }
    return dispatcher;
  }

  /**
   * Helper method to set user email
   *
   * @param value email of User passed by user
   */
  public void setEmail(@NonNull String value) {
      setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_USER_EMAIL, value);
  }

  /**
   * Helper method to set user full name
   *
   * @param value Full Name of user passed by user
   */
  public void setFullName(@NonNull String value) {
      setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_USER_NAME, value);
  }

  /**
   * Use {@link MoEHelper#setGender(UserGender)}
   */
  @Deprecated
  public void setGender(@NonNull String value) {
    if (MoEUtils.isEmptyString(value)){
      Logger.e( TAG + " setGender() : Cannot set null gender value.");
      return;
    }
    setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_USER_GENDER, value.toLowerCase());
  }

  /**
   * Helper method to set user gender
   *
   * @param gender Gender value passed by user
   * @since 10.2.00
   */
  public void setGender(@NonNull UserGender gender) {
    setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_USER_GENDER, gender.toString().toLowerCase());
  }

  /**
   * Helper method to set User first Name
   *
   * @param value First Name value passed by user
   */
  public void setFirstName(@NonNull String value) {
      setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_USER_FIRST_NAME, value);
  }

  /**
   * Helper method to set User Last Name
   *
   * @param value Last Name value passed by user
   */
  public void setLastName(@NonNull String value) {
      setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_USER_LAST_NAME, value);
  }

  /**
   * Helper method to set User BirthDate<br>
   *
   * @param birthDate BirthDate value passed by user
   */
  public void setBirthDate(@NonNull Date birthDate){
    attributeManager.setCustomUserAttribute(new PayloadBuilder().putAttrDate(MoEHelperConstants
        .USER_ATTRIBUTE_USER_BDAY, birthDate).build());
  }

  /**
   * Helper method to set User Unique Id
   *
   * @param value Unique Id passed by Client
   */
  public void setUniqueId(@NonNull String value) {
    if (MoEUtils.isEmptyString(value)){
      Logger.e( TAG + " setUniqueId() : Cannot set null unique id.");
      return;
    }
    setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID, value);
  }

  /**
   * Helper method to set User Unique Id
   *
   * @param value Unique Id passed by Client
   */
  public void setUniqueId(int value) {
    setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID, value);
  }

  /**
   * Helper method to set User Unique Id
   *
   * @param value Unique Id passed by Client
   */
  public void setUniqueId(long value) {
    setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID, value);
  }

  /**
   * Helper method to set User Unique Id
   *
   * @param value Unique Id passed by Client
   */
  public void setUniqueId(float value) {
    setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID, value);
  }

  /**
   * Helper method to set User Unique Id
   *
   * @param value Unique Id passed by Client
   */
  public void setUniqueId(double value) {
    setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID, value);
  }

  /**
   * Helper method to set User Number
   *
   * @param value Number passed by user
   */
  public void setNumber(String value) {
    if (!TextUtils.isEmpty(value)) {
      setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_USER_MOBILE, value);
    }
  }

  /**
   * Tracks device locale.
   * Usage : Call the method in the application class of the application
   */
  public void trackDeviceLocale(){
    dispatcher.trackDeviceLocale();
  }

  public void setAlias(double currentId){
    try{
      JSONObject userJSON = new JSONObject();
      userJSON.put(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID, currentId);
      dispatcher.setAlias(userJSON);
    } catch(Exception e){
      Logger.e(TAG + " setAlias() ", e);
    }
  }

  /**
   * Update user's unique id which was previously set by {@link MoEHelper#setUniqueId(String)}
   * @param currentId updated unique id.
   */
  public void setAlias(String currentId){
    try{
      if (TextUtils.isEmpty(currentId)){
        Logger.e(TAG + "Updated id cannot be null");
        return;
      }
      JSONObject userJSON = new JSONObject();
      userJSON.put(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID, currentId.trim());
      dispatcher.setAlias(userJSON);
    } catch(Exception e){
      Logger.e(TAG + " setAlias() ", e);
    }
  }

  /**
   * Update user's unique id which was previously set by {@link MoEHelper#setUniqueId(long)}
   * @param currentId updated unique id.
   */
  public void setAlias(long currentId){
    try{
      JSONObject userJSON = new JSONObject();
      userJSON.put(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID, currentId);
      dispatcher.setAlias(userJSON);
    } catch(Exception e){
      Logger.e(TAG + " setAlias() ", e);
    }
  }

  /**
   * Update user's unique id which was previously set by {@link MoEHelper#setUniqueId(int)}
   * @param currentId updated unique id.
   */
  public void setAlias(int currentId) {
    try {
      JSONObject userJSON = new JSONObject();
      userJSON.put(MoEHelperConstants.USER_ATTRIBUTE_UNIQUE_ID, currentId);
      dispatcher.setAlias(userJSON);
    } catch (Exception e) {
      Logger.e(TAG + " setAlias() ", e);
    }
  }

  /**
   * Set push preference for the user.
   * @param pushPreference push preference value true or false
   */
  public void trackUserPushPreference(boolean pushPreference){
    setUserAttribute(MoEHelperConstants.USER_ATTRIBUTE_USER_PUSH_PREFERENCE, pushPreference);
  }

  /**
   * Fetches device triggers
   *
   * @since 8.5.00
   */
  public void fetchDeviceTriggersIfRequired(){
    MoEDTManager.getInstance().syncDeviceTriggersIfRequired(context);
  }

  /**
   * Use {@link MoECallbacks#addLogoutCompleteListener(OnLogoutCompleteListener)}
   */
  @Deprecated public void setOnLogoutCompleteListener(OnLogoutCompleteListener listener) {
    MoECallbacks.getInstance().addLogoutCompleteListener(listener);
  }

  /**
   * No-op method Use {@link MoECallbacks#removeLogoutListener(OnLogoutCompleteListener)}
   */
  @Deprecated public void removeLogoutCompleteListener() {

  }

  public MoEHelper setUserAttributeISODate(@NonNull String attributeName,
      @NonNull String attributeValue) {
    dispatcher.setCustomUserAttribute(
        new PayloadBuilder().putAttrISO8601Date(attributeName, attributeValue).build());
    return this;
  }

  /**
   * Use {@link MoECallbacks#addAppBackgroundListener(OnAppBackgroundListener)}
   */
  @Deprecated public void registerAppBackgroundListener(@NonNull OnAppBackgroundListener listener) {
    Logger.v(TAG + " registerAppBackgroundListener() : Registering OnAppBackgroundListener");
    MoECallbacks.getInstance().addAppBackgroundListener(listener);
  }

  /**
   * No-op method use {@link MoECallbacks#removeAppBackgroundListener(OnAppBackgroundListener)}
   */
  @Deprecated public void unregisterAppBackgroundListener() {
  }

  public MoEAttributeManager getAttributeManager() {
    return getDelegate().getAttributeManager();
  }

  /**
   * API to track event via MoEngage-Segment Bundled Integration.<br>
   * <b>Note:</b> Do not call this method. This is only for internal usage.
   *
   * @param eventName Event Name
   * @param eventAttributes Event Attributes
   * @since 9.4.02
   */
  public void trackEventFromSegment(String eventName, JSONObject eventAttributes) {
    getDelegate().addTaskToQueue(new TrackSegmentEventTask(context, eventName, eventAttributes));
  }

  /**
   * API to track user attributes via MoEngage-Segment Bundled Integration.<br>
   * <b>Note:</b> Do not call this method. This is only for internal usage.
   *
   * @param userAttributesMap attribute map.
   * @since 9.4.02
   */
  public void trackUserAttributeFromSegment(Map<String, Object> userAttributesMap) {
    getDelegate().addTaskToQueue(new TrackSegmentUserAttributeTask(context, userAttributesMap));
  }

  public boolean isFirstScreen(){
    return isFirstScreen;
  }

  @RestrictTo(Scope.LIBRARY)
  public void setFirstScreen(boolean firstScreen) {
    isFirstScreen = firstScreen;
  }

  public static boolean isAppInBackground() {
      return activityCounter <= 0;
    }

  private Application application;

  @Nullable
  public Application getApplication(){
    return application;
  }

  public void setApplication(Application application){
    this.application = application;
  }

  /**
   * This API tells the SDK whether it is a fresh install or an existing application was updated.
   *
   * @param status app status, either install or update.
   */
  public void setAppStatus(AppStatus status) {
    if (!RemoteConfig.getConfig().isAppEnabled) return;
    MoEDispatcher.getInstance(context).addTaskToQueue(new TrackInstallUpdateTask(context, status));
  }

  private List<String> appContext;

  /**
   * Set the current context for the given user.
   *
   * @param appContext instance of {@link List}
   * @since 10.0.00
   */
  public void setAppContext(List<String> appContext) {
    this.appContext = appContext;
  }

  @Nullable public List<String> getAppContext() {
    return appContext;
  }

  /**
   * Resets the current context of the user
   *
   * @since 10.0.00
   */
  public void resetAppContext() {
    this.appContext = null;
  }
}
