/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import android.annotation.TargetApi;
import android.app.AlarmManager;
import android.app.PendingIntent;
import android.app.job.JobInfo;
import android.app.job.JobParameters;
import android.app.job.JobScheduler;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.os.Build;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import com.moe.pushlibrary.models.BatchData;
import com.moengage.core.analytics.AnalyticsHelper;
import com.moengage.core.events.MoEEventManager;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskResult;
import com.moengage.core.model.DevicePreferences;
import com.moengage.core.model.PushTokens;
import com.moengage.core.model.RemoteConfig;
import com.moengage.core.reports.ReportsBatchHelper;
import com.moengage.core.rest.Response;
import com.moengage.core.utils.JsonBuilder;
import com.moengage.core.utils.RestUtils;
import java.util.ArrayList;
import java.util.TimeZone;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Send report batches to the server.
 *
 * @author Umang Chamaria
 */
class SendInteractionDataTask extends SDKTask {
  @Nullable private OnJobComplete jobCompleteCallback;
  @Nullable private JobParameters jobParameters;
  private ConfigurationProvider provider;

  private final String TAG = "SendInteractionDataTask";

  SendInteractionDataTask(Context context) {
    this(context, null, null);
  }

  SendInteractionDataTask(Context context, @Nullable OnJobComplete jobCompleteCallback,
      @Nullable JobParameters parameters) {
    super(context);
    this.jobCompleteCallback = jobCompleteCallback;
    this.jobParameters = parameters;
    provider = ConfigurationProvider.getInstance(this.context);
  }

  @Override public TaskResult execute() {
    try {
      if (!RemoteConfig.getConfig().isAppEnabled) return null;
      Logger.v(TAG + " executing task");
      // creating and saving batches
      MoEDispatcher.getInstance(context).getBatchHelper().createAndSaveBatches(context,
          AnalyticsHelper.getInstance(context).getSession());
      // set cached event counter to zero
      MoEEventManager.getInstance(context).setEventCounter(0);

      String appId = MoEUtils.getAppId();
      if (TextUtils.isEmpty(appId)) {
        Logger.e(TAG + " execute: App-id not present cannot make report add call.");
        return null;
      }
      sendInteractionData(appId);
      schedulePeriodicRetryIfRequired();
      releaseJobLockIfRequired();
      Logger.v(TAG + " : completed task");
      return null;
    } catch (Exception e) {
      Logger.f(TAG + " : execute() ", e);
    }
    return null;
  }

  private void releaseJobLockIfRequired() {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP
        && jobCompleteCallback != null
        && jobParameters != null) {
      Logger.v(TAG + " releaseJobLockIfRequired() : Trying to release job lock.");
      jobCompleteCallback.jobCompleted(jobParameters, false);
    }
  }

  @Override public String getTaskTag() {
    return TAG_SEND_INTERACTION_DATA;
  }

  @Override public boolean isSynchronous() {
    return true;
  }

  private void sendInteractionData(String appId) {
    boolean result = false;
    ArrayList<BatchData> batchedData = null;

    String endpoint =
        MoEConstants.API_ENDPOINT_REPORT_ADD + "/" + appId;
    if (shouldSendDataToTestServer()) {
      endpoint = MoEConstants.API_ENDPOINT_INTEGRATION_VERIFICATION_REPORT_ADD;
    }

    ReportsBatchHelper batchHelper = new ReportsBatchHelper();
    for (; ; ) {
      batchedData = MoEDAO.getInstance(context).getBatchedData(100);
      Logger.d(
          "SendInteractionDataTask : sendInteractionData:Fetching interaction data in batches");
      if (batchedData == null || batchedData.isEmpty()) {
        Logger.d("SendInteractionDataTask : sendInteractionData: Found Nothing to send");
        return;
      }
      for (BatchData data : batchedData) {
        BatchData batch = batchHelper.updateBatchIfRequired(context, data);
        try {
          String requestId = getRequestID(data.batchDataJson);
          JSONObject requestBody =
              cleanBody(data.batchDataJson)
                  .put(MoEConstants.REQUEST_ATTR_QUERY_PARAMS, getQueryParams());
          Response response = APIManager.reportAdd(appId, endpoint, requestId,
              requestBody);
          if (response != null && response.responseCode == 200) result = true;
        } catch (Exception e) {
          Logger.f("SendInteractionDataTask : API failed", e);
          result = false;
        }
        if (result) {
          Logger.d("SendInteractionDataTask : Batch sent successfully deleting batch");
          MoEDAO.getInstance(context).deleteBatch(batch);
        } else {
          scheduleRetryIfRequired();
          break;
        }
      }
      if (!result) break;
      batchedData.clear();
    }
  }

  private void scheduleRetryIfRequired() {
    int retryCount = provider.getImmediateRetryCount();
    switch (retryCount) {
      case 0:
        scheduleImmediateRetry(1);
        provider.setImmediateRetryCount(++retryCount);
        break;
      case 1:
        scheduleImmediateRetry(3);
        provider.setImmediateRetryCount(++retryCount);
        break;
      default:
        provider.setImmediateRetryCount(0);
    }
  }

  /**
   * Checks whether device is registered as a test device or not
   *
   * @return true if registered as test device, else false
   */
  private boolean shouldSendDataToTestServer() {
    boolean isDeviceRegisteredForVerification =
        provider.isDeviceRegisteredForVerification();
    long registrationTime =
        provider.getVerificationRegistrationTime();
    return isDeviceRegisteredForVerification && ((registrationTime + 3600 * 1000)
        > MoEUtils.currentTime());
  }

  private void scheduleImmediateRetry(int minutes) {
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
      scheduleSyncJob(666666, minutes * 60 * 1000);
    } else {
      scheduleSyncAlarm(55555, minutes * 60 * 1000);
    }
  }

  @Nullable private String getRequestID(JSONObject batchData) {
    try {
      return batchData.getString(MoEConstants.REQUEST_HEADER_REQUEST_ID);
    } catch (JSONException e) {
      Logger.e("SendInteractionDataTask: getRequestID(): Exception ", e);
    }
    return null;
  }

  private JSONObject cleanBody(JSONObject batchData) {
    batchData.remove(MoEConstants.REQUEST_HEADER_REQUEST_ID);
    return batchData;
  }

  private void schedulePeriodicRetryIfRequired() {
    if (!SdkConfig.getConfig().isBackgroundSyncEnabled) return;
    long syncInterval =
        RemoteConfig.getConfig().dataSyncRetryInterval;
    Logger.v(TAG + " schedulePeriodicRetryIfRequired() : Will schedule background retry.");
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
      scheduleSyncJob(77777, syncInterval);
    } else {
      scheduleSyncAlarm(88888, syncInterval);
    }
  }

  private void scheduleSyncAlarm(int requestCode, long timeDelayMillis) {
    Intent alarmIntent = new Intent(context, MoEAlarmReceiver.class);
    PendingIntent pendingIntent =
        PendingIntent.getBroadcast(context, requestCode, alarmIntent,
            PendingIntent.FLAG_UPDATE_CURRENT);
    AlarmManager alarmManager = (AlarmManager) context.getSystemService(Context.ALARM_SERVICE);
    if (alarmManager != null) {
      alarmManager.set(AlarmManager.RTC_WAKEUP, MoEUtils.currentTime() + timeDelayMillis,
          pendingIntent);
    }
  }

  @TargetApi(Build.VERSION_CODES.LOLLIPOP) private void scheduleSyncJob(int requestCode,
      long timeDelayMillis) {
    ComponentName serviceComponent = new ComponentName(context, DataSyncJob.class);
    JobInfo.Builder builder = new JobInfo.Builder(requestCode, serviceComponent);
    builder.setRequiredNetworkType(JobInfo.NETWORK_TYPE_ANY)
        .setOverrideDeadline(MoEUtils.currentTime() + 2 * timeDelayMillis)
        .setMinimumLatency(timeDelayMillis);
    JobScheduler jobScheduler =
        (JobScheduler) context.getSystemService(Context.JOB_SCHEDULER_SERVICE);
    if (jobScheduler != null) {
      jobScheduler.schedule(builder.build());
    }
  }

  private JSONObject getQueryParams() throws JSONException {
    JsonBuilder builder = RestUtils.getDefaultParams(context);
    builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_TIMEZONE, TimeZone.getDefault().getID());

    DevicePreferences devicePreferences = MoEDAO.getInstance(context).getDevicePreferences();

    if (!devicePreferences.isPushOptedOut) {
      PushTokens pushTokens = MoEDAO.getInstance(context).getPushTokens();
      if (!MoEUtils.isEmptyString(pushTokens.fcmToken)) {
        builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_FCM_PUSH_ID, pushTokens.fcmToken);
      }
      if (!MoEUtils.isEmptyString(pushTokens.secondaryToken)) {
        builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_OEM_TOKEN, pushTokens.secondaryToken);
      }
    }

    if (!devicePreferences.isDataTrackingOptedOut) {
      String androidId = MoEUtils.getAndroidID(context);
      if (!TextUtils.isEmpty(androidId)) {
        builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_ANDROID_ID, androidId);
      }

      if (!SdkConfig.getConfig().isGaidTrackingOptedOut) {
        String gaid = provider.getStoredGAID();
        if (TextUtils.isEmpty(gaid)) {
          AdvertisingIdClient.AdInfo adInfo = MoEUtils.getAdvertisementInfo(context);
          if (adInfo != null) {
            gaid = adInfo.getId();
            provider.storeGAID(gaid);
          }
        }
        if (!TextUtils.isEmpty(gaid)) {
          builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_GAID, gaid);
        }
      }
      builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_OS_VERSION,
          String.valueOf(Build.VERSION.SDK_INT));
      builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_MODEL, Build.MODEL);
      builder.putString(MoEConstants.GENERIC_PARAM_V2_KEY_APP_VERSION_NAME,
          provider.getAppVersionName());

      String nwType = MoEUtils.getNetworkType(context);
      if (!TextUtils.isEmpty(nwType)) {
        builder.putString(MoEConstants.GENERIC_PARAM_KEY_NW_TYPE, nwType);
      }
    }

    return builder.build();
  }
}
