/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.core;

import android.content.ContentProviderOperation;
import android.content.ContentValues;
import android.content.Context;
import android.content.OperationApplicationException;
import android.database.Cursor;
import android.net.Uri;
import android.os.RemoteException;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.RestrictTo;
import android.support.annotation.RestrictTo.Scope;
import android.support.annotation.WorkerThread;
import android.text.TextUtils;
import com.moe.pushlibrary.models.BatchData;
import com.moe.pushlibrary.models.Event;
import com.moe.pushlibrary.models.UserAttribute;
import com.moe.pushlibrary.providers.MoEDataContract;
import com.moe.pushlibrary.providers.MoEDataContract.AttributeCacheEntity;
import com.moe.pushlibrary.providers.MoEDataContract.BatchDataEntity;
import com.moe.pushlibrary.providers.MoEDataContract.CampaignListEntity;
import com.moe.pushlibrary.providers.MoEDataContract.DatapointEntity;
import com.moe.pushlibrary.providers.MoEDataContract.InAppMessageEntity;
import com.moe.pushlibrary.providers.MoEDataContract.MessageEntity;
import com.moe.pushlibrary.providers.MoEDataContract.UserAttributeEntity;
import com.moengage.ConfigDefault;
import com.moengage.core.model.DevicePreferences;
import com.moengage.core.model.MoEAttribute;
import com.moengage.core.model.PushTokens;
import com.moengage.core.model.SDKIdentifiers;
import com.moengage.core.model.UserSession;
import java.util.ArrayList;
import org.json.JSONObject;

/**
 * @author MoEngage (abhishek@moengage.com)
 * @version 1.3
 * @since 5.0
 */
public final class MoEDAO {

  private static final String TAG = "MoEDAO";

  public static MoEDAO getInstance(Context context) {
    if (instance == null) {
      synchronized (MoEDAO.class){
        instance = new MoEDAO(context);
      }
    }
    return instance;
  }

  private static MoEDAO instance = null;

  private Uri MESSAGES_CONTENT_URI = null;
  private Uri INAPP_CONTENT_URI = null;
  private Uri DATAPOINTS_CONTENT_URI = null;
  private Uri USER_ATTRIBUTES_URI = null;
  private Uri CAMPAIGN_LIST_URI = null;
  private Uri BATCHED_DATA_URI = null;
  private Uri ATTRIBUTE_CACHE_URI = null;

  private String AUTHORITY = null;
  private Context context;

  private MoEDAO(Context context) {
    MESSAGES_CONTENT_URI = MessageEntity.getContentUri(context);
    INAPP_CONTENT_URI = InAppMessageEntity.getContentUri(context);
    DATAPOINTS_CONTENT_URI = DatapointEntity.getContentUri(context);
    USER_ATTRIBUTES_URI = UserAttributeEntity.getContentUri(context);
    CAMPAIGN_LIST_URI = CampaignListEntity.getContentUri(context);
    BATCHED_DATA_URI = BatchDataEntity.getContentUri(context);
    ATTRIBUTE_CACHE_URI = AttributeCacheEntity.getContentUri(context);
    AUTHORITY = MoEDataContract.getAuthority(context);
    this.context = context;
  }

  int getUnreadMessageCount() {
    int unReadCount = 0;
    Cursor cursor = null;
    try {
      cursor = context.getContentResolver()
          .query(MESSAGES_CONTENT_URI, MessageEntity.PROJECTION, MessageEntity.MSG_CLICKED + " = ?",
              new String[] { "0" }, MessageEntity.DEFAULT_SORT_ORDER);
      if (null != cursor) {
        unReadCount = cursor.getCount();
      }
      Logger.v("Getting Unread PromotionalMessage Count: count=" + unReadCount);
    } catch (Exception e) {
      Logger.f( TAG + " getUnreadMessageCount() : Exception: ", e);
    }finally {
      closeCursor(cursor);
    }

    return unReadCount;
  }

  public void addEvent(Event event) {
    try {
      if (null == event) {
        Logger.v("Null event passed, skipping it");
        return;
      }
      Logger.v("Event : " + event.details);
      ContentValues values = new ContentValues();
      values.put(DatapointEntity.GTIME, event.time);
      values.put(DatapointEntity.DETAILS, event.details);
      Uri newRecord = context.getContentResolver().insert(DATAPOINTS_CONTENT_URI, values);
      if (null != newRecord) {
        Logger.v("New Event added with Uri: " + newRecord.toString());
      } else {
        Logger.v("Unable to add event");
      }
    } catch (Exception e) {
      Logger.f( TAG + " addEvent() : Exception: ", e);
    }
  }

  /**
   * Gets the events from datapoints table, if present.
   * @param batchSize maximum number of events to be returned
   * @return user events if present else null
   */
  @RestrictTo(Scope.LIBRARY_GROUP)
  @Nullable public ArrayList<Event> getInteractionData(int batchSize) {
    Cursor cur = null;
    try {
      Uri CONTENT_URI = DATAPOINTS_CONTENT_URI.buildUpon()
          .appendQueryParameter(MoEDataContract.QUERY_PARAMETER_LIMIT, String.valueOf(batchSize))
          .build();
      cur = context.getContentResolver()
          .query(CONTENT_URI, DatapointEntity.PROJECTION, null, null,
              DatapointEntity.GTIME + " ASC");
      if (null == cur || cur.getCount() == 0) {
        Logger.v("Empty cursor");
        closeCursor(cur);
        return null;
      }

      ArrayList<Event> eventList = new ArrayList<Event>();
      while (cur.moveToNext()) {
        eventList.add(new Event(cur.getInt(DatapointEntity.COLUMN_INDEX_ID),
            cur.getString(DatapointEntity.COLUMN_INDEX_DETAILS)));
      }
      return eventList;
    } catch (Exception e) {
      Logger.f(TAG + " getInteractionData() : Exception: ", e);
    }finally {
      closeCursor(cur);
    }
    return null;
  }

  /**
   * Deletes passed events from the datapoints table
   * @param events events to be deleted
   * @param context application context
   */
  @RestrictTo(Scope.LIBRARY_GROUP)
  public void deleteInteractionData(ArrayList<Event> events, Context context) {
    ArrayList<ContentProviderOperation> operations = new ArrayList<ContentProviderOperation>();
    ContentProviderOperation operation;
    for (Event item : events) {
      operation = ContentProviderOperation.newDelete(DATAPOINTS_CONTENT_URI)
          .withSelection(DatapointEntity._ID + " = ?", new String[] { String.valueOf(item._id) })
          .build();
      operations.add(operation);
    }

    try {
      context.getContentResolver().applyBatch(AUTHORITY, operations);
    } catch (RemoteException e) {
      Logger.f("MoEDAO: deleteInteractionData", e);
    } catch (OperationApplicationException e) {
      Logger.f("MoEDAO: deleteInteractionData", e);
    }catch (Exception e){
      Logger.f("MoEDAO: deleteInteractionData", e);
    }
  }

  @Nullable
  Cursor getMessages(Context context) {
    try {
      return context.getContentResolver()
          .query(MESSAGES_CONTENT_URI, MessageEntity.PROJECTION, null, null,
              MessageEntity.DEFAULT_SORT_ORDER);
    } catch (Exception e) {
      Logger.f( TAG + " getMessages() : Exception: ", e);
    }
    return null;
  }

  boolean setMessageClicked(final long id) {
    int rowCount = -1;
    try {
      Uri updateRec = MESSAGES_CONTENT_URI.buildUpon().appendPath(String.valueOf(id)).build();
      ContentValues values = new ContentValues();
      values.put(MessageEntity.MSG_CLICKED, 1);
      rowCount = context.getContentResolver().update(updateRec, values, null, null);
      context.getContentResolver().notifyChange(updateRec, null);
    } catch (Exception e) {
      Logger.f( TAG + " setMessageClicked() : Exception: ", e);
    }
    return rowCount > 0;
  }

  void removeExpiredData() {
    try {
      String currTime = Long.toString(System.currentTimeMillis());
      int rows = context.getContentResolver()
          .delete(INAPP_CONTENT_URI, InAppMessageEntity.MSG_TTL + " < ?" + " AND " +
                  InAppMessageEntity.MSG_STATUS + " = ?",
              new String[] { Long.toString(System.currentTimeMillis() / 1000), "expired" });
      Logger.v("MoEDAO:removeExpiredData: Number of IN APP records deleted: " + rows);
      rows = context.getContentResolver()
          .delete(MESSAGES_CONTENT_URI, MessageEntity.MSG_TTL + " < ?", new String[] { currTime });
      Logger.v("MoEDAO:removeExpiredData: Number of PromotionalMessage records deleted: " + rows);
      rows = context.getContentResolver()
          .delete(CAMPAIGN_LIST_URI, CampaignListEntity.CAMPAIGN_ID_TTL + " < ?", new String[] { currTime });
      Logger.v("MoEDAO:removeExpiredData: Number of CampaignList records deleted: " + rows);
      context.getContentResolver().notifyChange(INAPP_CONTENT_URI, null);
      context.getContentResolver().notifyChange(MESSAGES_CONTENT_URI, null);
    } catch (Exception e) {
      Logger.f( TAG + " removeExpiredData() : Exception: ", e);
    }
  }

  boolean setMessageClickedByTime(final long gtime) {
    int rowCount = -1;
    try {
      ContentValues values = new ContentValues();
      values.put(MessageEntity.MSG_CLICKED, 1);
      rowCount = context.getContentResolver()
          .update(MESSAGES_CONTENT_URI, values, MessageEntity.GTIME + " = ? ",
              new String[] { String.valueOf(gtime) });
      context.getContentResolver().notifyChange(MESSAGES_CONTENT_URI, null);
    } catch (Exception e) {
      Logger.f( TAG + " setMessageClickedByTime() : Exception: ", e);
    }
    return rowCount > 0;
  }

  void addOrUpdateUserAttribute(@NonNull UserAttribute userAttribute) {
    if (userAttribute == null) return;
    Logger.v("User Attribute -->"
        + userAttribute.getUserAttributeName()
        + ":"
        + userAttribute.getUserAttributeValue());
    ContentValues contentValues = new ContentValues();
    contentValues.put(UserAttributeEntity.ATTRIBUTE_NAME, userAttribute.userAttributeName);
    contentValues.put(UserAttributeEntity.ATTRIBUTE_VALUE, userAttribute.userAttributeValue);
    Cursor cursor = null;
    try {
      //check if the user attribute already exists
      cursor = context.getContentResolver()
          .query(USER_ATTRIBUTES_URI, UserAttributeEntity.PROJECTION,
              UserAttributeEntity.ATTRIBUTE_NAME + "=?", new String[] {
                  userAttribute.userAttributeName
              }, null);
      if (cursor != null && cursor.moveToFirst()) {
        //update attribute value if exists
        updateUserAttribute(userAttribute, contentValues);
      } else {
        //add attribute value
        addUserAttribute(contentValues);
      }
    }catch (Exception e){
      Logger.e("MoEDAO: addOrUpdateUserAttribute()", e);
    }finally {
      if (cursor != null) {
        cursor.close();
      }
    }
  }

  private void addUserAttribute(ContentValues contentValues) {
    Uri newRecord = context.getContentResolver().insert(USER_ATTRIBUTES_URI, contentValues);
    if (null != newRecord) {
      Logger.v("New user attribute added with Uri: " + newRecord.toString());
    } else {
      Logger.v("Unable to user attribute");
    }
  }

  private void updateUserAttribute(@NonNull UserAttribute userAttribute,
      ContentValues contentValues) {
    int updateCount = context.getContentResolver()
        .update(USER_ATTRIBUTES_URI, contentValues, UserAttributeEntity.ATTRIBUTE_NAME + "=?",
            new String[] { userAttribute.userAttributeName });
    if (updateCount > 0) {
      Logger.v("New user attribute updated, count: " + updateCount);
    } else {
      Logger.v("Unable to user attribute");
    }
  }

  @Nullable UserAttribute getUserAttributeByName(@NonNull String attributeName){
    if (TextUtils.isEmpty(attributeName)) return null;
    Cursor cursor = null;
    UserAttribute userAttribute = null;
    try{
      cursor = context.getContentResolver()
          .query(USER_ATTRIBUTES_URI, UserAttributeEntity.PROJECTION,
              UserAttributeEntity.ATTRIBUTE_NAME + "=?", new String[] {
                  attributeName
              }, null);
      if (cursor != null && cursor.moveToFirst()){
        userAttribute = new UserAttribute();
        userAttribute.userAttributeName =
            cursor.getString(UserAttributeEntity.COLUMN_INDEX_ATTRIBUTE_NAME);
        userAttribute.userAttributeValue =
            cursor.getString(UserAttributeEntity.COLUMN_INDEX_ATTRIBUTE_VALUE);
      }
    } catch (Exception e){
      Logger.f( TAG + " getUserAttributeByName() : Exception: ", e);
    }finally{
      closeCursor(cursor);
    }
    return userAttribute;
  }

  /**
   * Gets batches of interaction data from batchdata table
   * @param batchSize max number of batches that should be returned
   * @return batched data if present, else null
   */
  @Nullable ArrayList<BatchData> getBatchedData(int batchSize) {
    Uri CONTENT_URI = BATCHED_DATA_URI.buildUpon()
        .appendQueryParameter(MoEDataContract.QUERY_PARAMETER_LIMIT, String.valueOf(batchSize))
        .build();
    Cursor cursor = null;
    ArrayList<BatchData> batchList = null;
    try {
      cursor = context.getContentResolver()
          .query(CONTENT_URI, BatchDataEntity.PROJECTION, null, null, null);
      if (cursor == null || cursor.getCount() == 0) {
        Logger.v("Empty cursor");
        closeCursor(cursor);
        return null;
      }
      batchList = new ArrayList<>(cursor.getCount());
      if (cursor.moveToFirst()) {
        do {
          long _id = cursor.getLong(cursor.getColumnIndex(BatchDataEntity._ID));
          String data = cursor.getString(cursor.getColumnIndex(BatchDataEntity.BATCHED_DATA));
          try {
            batchList.add(new BatchData(_id, new JSONObject(data)));
          } catch (Exception e) {
            Logger.f( "MoEDAO getBatchedData() : ", e);
          }
        }while (cursor.moveToNext());
      }
    } catch (Exception e){
      Logger.f( "MoEDAO getBatchedData() :exception ", e);
    } finally{
      closeCursor(cursor);
    }
    return batchList;
  }

  /**
   * Writes batched interaction data to batchdata table
   * @param batch batched data
   */
  @RestrictTo(Scope.LIBRARY_GROUP)
  public void writeBatch(@NonNull String batch){
    try {
      if (batch == null) return;
      ContentValues values = new ContentValues();
      values.put(BatchDataEntity.BATCHED_DATA, batch);
      Uri newBatch = context.getContentResolver().insert(BATCHED_DATA_URI, values);
      if (newBatch != null){
        Logger.v("MoEDAO: writeBatch() New batch added : uri " + newBatch.toString());
      }else {
        Logger.f("MoEDAO: writeBatch() unable to add batch");
      }
    } catch (Exception e) {
      Logger.f( TAG + " writeBatch() : Exception: ", e);
    }
  }

  /**
   * Deletes passed batch from the batchdata table
   * @param batch batch to be deleted
   */
  void deleteBatch(BatchData batch) {
    ArrayList<ContentProviderOperation> operations = new ArrayList<>();
    ContentProviderOperation operation;
      operation = ContentProviderOperation.newDelete(BATCHED_DATA_URI)
          .withSelection(BatchDataEntity._ID + " = ?", new String[] { String.valueOf(batch._id) })
          .build();
      operations.add(operation);
    try {
      context.getContentResolver().applyBatch(AUTHORITY, operations);
    } catch (RemoteException e) {
      Logger.f("MoEDAO: deleteInteractionData", e);
    } catch (OperationApplicationException e) {
      Logger.f("MoEDAO: deleteInteractionData", e);
    }catch (Exception e){
      Logger.f("MoEDAO: deleteInteractionData", e);
    }
  }

  /**
   * Deletes all events from data-points table.
   */
  @WorkerThread
  void deleteAllEvents(){
    try {
      context.getContentResolver()
          .delete(DatapointEntity.getContentUri(context), null, null);
    } catch (Exception e) {
      Logger.f( TAG + " deleteAllEvents() : Exception: ", e);
    }
  }

  /**
   * Deletes all pending bathes from batches table.
   */
  @WorkerThread
  void deleteAllBatches(){
    try {
      context.getContentResolver().delete(BatchDataEntity.getContentUri(context),
          null, null);
    } catch (Exception e) {
      Logger.f( TAG + " deleteAllBatches() : Exception: ", e);
    }
  }

  private void closeCursor(Cursor cursor){
    if (cursor != null){
      cursor.close();
    }
  }

  @RestrictTo(Scope.LIBRARY)
  @Nullable public MoEAttribute getAttributeByName(String attributeName){
    Cursor cursor = null;
    try {
      cursor = context.getContentResolver()
          .query(ATTRIBUTE_CACHE_URI, AttributeCacheEntity.PROJECTION,
              AttributeCacheEntity.ATTRIBUTE_NAME + "=?", new String[] {
                  attributeName
              }, null);
      if (cursor != null && cursor.moveToFirst()){
        return cachedAttributeFromCursor(cursor);
      }
    }catch (Exception e){
      Logger.f( TAG + " getAttributeByName() : ");
    }finally {
      closeCursor(cursor);
    }
    return null;
  }

  @RestrictTo(Scope.LIBRARY)
  public void addOrUpdateAttributeToCache(@NonNull MoEAttribute attribute){
    if (isAttributePresentInCache(attribute.getName())){
      //update user attribute
      updateAttributeCache(attribute);
    }else {
      //add user attribute
      addAttributeToCache(attribute);
    }
  }

  private int updateAttributeCache(@NonNull MoEAttribute attribute) {
    int updateCount = -1;
    try {
      updateCount = context.getContentResolver()
          .update(ATTRIBUTE_CACHE_URI, contentValuesFromAttribute(attribute), AttributeCacheEntity
                  .ATTRIBUTE_NAME + "=?",
              new String[] { attribute.getName() });
      if (updateCount > 0) {
        Logger.v("Attribute cache updated, count: " + updateCount);
      } else {
        Logger.v("Unable to update attribute cache");
      }
    } catch (Exception e) {
      Logger.f( TAG + " updateAttributeCache() : Exception: ", e);
    }
    return updateCount;
  }

  /**
   * Add attribute {@link MoEAttribute} to Attribute Cache.
   *
   * @param attribute Attribute to be added.
   * @return {@link Uri} of the added attribute.
   */
  @Nullable private Uri addAttributeToCache(@NonNull MoEAttribute attribute) {
    try {
      Uri newRecord = context.getContentResolver()
          .insert(ATTRIBUTE_CACHE_URI, contentValuesFromAttribute(attribute));
      if (null != newRecord) {
        Logger.v("New attribute added to cache with Uri: " + newRecord.toString());
      } else {
        Logger.v("Unable to add attribute to cache");
      }
      return newRecord;
    } catch (Exception e) {
      Logger.f(TAG + " addAttributeToCache() : Exception: ", e);
    }
    return null;
  }

  private ContentValues contentValuesFromAttribute(MoEAttribute attribute){
    ContentValues contentValue = new ContentValues();
    contentValue.put(AttributeCacheEntity.ATTRIBUTE_NAME, attribute.getName());
    contentValue.put(AttributeCacheEntity.ATTRIBUTE_VALUE, attribute.getValue());
    contentValue.put(AttributeCacheEntity.LAST_TRACKED_TIME, attribute.getLastTrackedTime());
    contentValue.put(AttributeCacheEntity.DATA_TYPE, attribute.getDataType());
    return contentValue;
  }

  private MoEAttribute cachedAttributeFromCursor(Cursor cursor){
    return new MoEAttribute(
        cursor.getString(AttributeCacheEntity.COLUMN_INDEX_ATTRIBUTE_NAME),
        cursor.getString(AttributeCacheEntity.COLUMN_INDEX_ATTRIBUTE_VALUE),
        cursor.getLong(AttributeCacheEntity.COLUMN_INDEX_LAST_TRACKED_TIME),
        cursor.getString(AttributeCacheEntity.COLUMN_INDEX_DATATYPE)
    );
  }

  /**
   * Checks whether attribute is present in cache or not.
   *
   * @param attributeName Attribute name which needs to be checked.
   * @return true if given attribute is present in cache else false.
   */
  boolean isAttributePresentInCache(String attributeName){
    Cursor cursor = null;
    try {
      cursor = context.getContentResolver()
          .query(ATTRIBUTE_CACHE_URI, AttributeCacheEntity.PROJECTION,
              AttributeCacheEntity.ATTRIBUTE_NAME + "=?", new String[] {
                  attributeName
              }, null);
      if (cursor != null && cursor.moveToFirst()){
        return true;
      }
    }catch (Exception e){
      Logger.f( TAG + " isAttributePresentInCache() : Exception ", e);
    }finally {
      closeCursor(cursor);
    }
    return false;
  }

  /**
   * Clear all cached values.
   */
  void clearAttributeCache() {
    try {
      Logger.v(TAG + " clearAttributeCache() : Clearing all cached attributes");
      context.getContentResolver().delete(ATTRIBUTE_CACHE_URI, null, null);
    } catch (Exception e) {
      Logger.f( TAG + " clearAttributeCache() : Exception: ", e);
    }
  }

  /**
   * Save remote configuration to storage.
   *
   * configuration.
   */
  @RestrictTo(Scope.LIBRARY) public void addOrUpdateRemoteConfiguration(String configurationString) {
    Logger.v(TAG + " addOrUpdateRemoteConfiguration(): Saving or updating remote configuration.");
    ConfigurationProvider.getInstance(context).setRemoteConfiguration(configurationString);
  }

  @RestrictTo(Scope.LIBRARY)
  public void updateConfigApiSyncTime(long time){
    ConfigurationProvider.getInstance(context).setLastConfigSyncTime(time);
  }

  public void saveUserAttributeUniqueId(@NonNull MoEAttribute attribute){
    Logger.v(TAG + " saveUserAttributeUniqueId(): Will save USER_ATTRIBUTE_UNIQUE_ID in cache "
        + "table and shared preference.");
    ConfigurationProvider.getInstance(context).saveUserAttributeUniqueId(attribute.getValue());
    addOrUpdateAttributeToCache(attribute);
  }

  public void saveUserSession(UserSession userSession){
    try {
      JSONObject sessionJson = UserSession.toJson(userSession);
      if (sessionJson == null) {
        Logger.e( TAG + " saveUserSession() : Could not serialise session about. Will not save "
            + "session.");
        return;
      }
      ConfigurationProvider.getInstance(context).saveUserSession(sessionJson.toString());
    } catch (Exception e) {
      Logger.e( TAG + " saveUserSession() : Exception: ", e);
    }
  }

  @Nullable public UserSession getLastSavedSession(){
    String sessionString = ConfigurationProvider.getInstance(context).getUserSession();
    if (sessionString == null) return null;
    return UserSession.fromJsonString(sessionString);
  }

  public SDKIdentifiers getSDKIdentifiers() {
    ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
    return new SDKIdentifiers(MoEUtils.getUserAttributeUniqueId(context),
        provider.getSegmentAnonymousId()
        , provider.getCurrentUserId());
  }

  public DevicePreferences getDevicePreferences() {
    ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
    return new DevicePreferences(provider.isDataTrackingOptedOut(),
        provider.isPushNotificationOptedOut(), provider.isInAppOptedOut());
  }

  public int updateBatch(BatchData batchData){
    int updateCount = -1;
    try {
      if (batchData._id == -1) return -1;
      Uri updateRec =
          BATCHED_DATA_URI.buildUpon().appendPath(String.valueOf(batchData._id)).build();
      ContentValues values = new ContentValues();
      values.put(BatchDataEntity.BATCHED_DATA, batchData.batchDataJson.toString());
      updateCount = context.getContentResolver().update(updateRec, values, null, null);
    }catch (Exception e){
      Logger.f( TAG + " updateBatch() : Exception ", e);
    }
    return updateCount;
  }

  public void deleteUserSession(){
    ConfigurationProvider.getInstance(context).saveUserSession(null);
  }

  void clearDataOnLogout(){
    // clear stored data in tables
    context.getContentResolver()
        .delete(MoEDataContract.DatapointEntity.getContentUri(context), null, null);
    context.getContentResolver()
        .delete(MoEDataContract.MessageEntity.getContentUri(context), null, null);
    context.getContentResolver()
        .delete(MoEDataContract.InAppMessageEntity.getContentUri(context), null, null);
    context.getContentResolver()
        .delete(MoEDataContract.UserAttributeEntity.getContentUri(context), null, null);
    context.getContentResolver().delete(MoEDataContract.CampaignListEntity.getContentUri
        (context), null, null);
    context.getContentResolver().delete(MoEDataContract.BatchDataEntity.getContentUri(context),
        null, null);
    context.getContentResolver().delete(MoEDataContract.DTEntity.getContentUri(context), null,
        null);
    clearAttributeCache();
    // clear shared preference
    ConfigurationProvider.getInstance(context).removeUserConfigurationOnLogout();
  }

  @RestrictTo(Scope.LIBRARY_GROUP)
  public PushTokens getPushTokens() {
    ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
    return new PushTokens(provider.getString(ConfigurationProvider.FCM_PUSH_TOKEN, ""),
        provider.getString(ConfigurationProvider.OEM_PUSH_TOKEN, ""));
  }

  @RestrictTo(Scope.LIBRARY_GROUP)
  public boolean isDeviceRegistered(){
    return ConfigurationProvider.getInstance(context).getBoolean(ConfigurationProvider.IS_DEVICE_REGISTERED,
        ConfigDefault.IS_DEVICE_REGISTERED);
  }

  void setDeviceRegistrationState(boolean state){
    ConfigurationProvider.getInstance(context).putBoolean(ConfigurationProvider.IS_DEVICE_REGISTERED, state);
  }
}

