/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.push;

import android.content.Context;
import android.support.annotation.RestrictTo;
import android.support.annotation.RestrictTo.Scope;
import android.text.TextUtils;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.PayloadBuilder;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.Logger;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEDispatcher;
import com.moengage.push.PushManager.OnTokenReceivedListener;
import org.json.JSONObject;

/**
 * Class to handle push token processing, includes sending token to the server, provide callback
 * if required.
 *
 * @author Umang Chamaria
 * Date: 28/03/19
 */
public class TokenHandler {

  public static final String TOKEN_BY_MOE = "MoE";
  public static final String REQ_REGISTRATION = "MOE_REG_REQ";

  private final String ATTR_REGISTRATION_BY = "registered_by";
  private final Object lock = new Object();
  private OnTokenReceivedListener tokenListener;
  private static final String TAG = "TokenHandler";
  private static TokenHandler instance = null;

  private TokenHandler(){}

  public static TokenHandler getInstance(){
    if (instance == null){
      synchronized (TokenHandler.class){
        if (instance == null) instance = new TokenHandler();
      }
    }
    return instance;
  }

  /**
   * Process Push Token
   *
   * @param context instance of {@link Context}
   * @param token Push Token
   * @param pushRegisteredBy Push registration is handled by App or MoEngage
   */
  @RestrictTo(Scope.LIBRARY_GROUP)
  public void processToken(Context context, String token, String pushRegisteredBy) {
    if (TextUtils.isEmpty(token)) return;
    Logger.v(TAG + " processToken() : Will try to process push token. Token: " + token + "\n"
        + "registered by: " + pushRegisteredBy);
    try {
      synchronized (lock) {
        token = ripMultiplexingExtras(token);
        if (tokenListener != null) tokenListener.onTokenReceived(token);
        String oldRegId = ConfigurationProvider.getInstance(context).getFcmToken();
        boolean update = tokenRefreshRequired(context, token);
        if (update) {
          PayloadBuilder payloadBuilder = new PayloadBuilder();
          payloadBuilder.putAttrString(ATTR_REGISTRATION_BY, pushRegisteredBy);
          payloadBuilder.setNonInteractive();
          MoEHelper.getInstance(context)
              .trackEvent(MoEHelperConstants.TOKEN_EVENT, payloadBuilder);
          trackDeviceAttributeForRegistration(context, pushRegisteredBy);
        }
        if (update) {
          ConfigurationProvider.getInstance(context).setFcmToken(token);
          MoEDispatcher.getInstance(context).getDeviceAddManager().registerFcmToken(context);
        }
        Logger.v(TAG + " processToken() oldId: = "
            + oldRegId
            + " token = "
            + token
            + " --updating[true/false]: "
            + update);
      }
    } catch (Exception e) {
      Logger.f(TAG + " processToken() : Exception ", e);
    }
  }

  private static final String ID_PREFIX = "|ID|";

  private String ripMultiplexingExtras(String token) {
    return !TextUtils.isEmpty(token) && token.startsWith(ID_PREFIX) ? token.substring(7) : token;
  }

  private boolean tokenRefreshRequired(Context context, String newToken) {
    if (!TextUtils.isEmpty(newToken)) {
      String oldToken = ConfigurationProvider.getInstance(context).getFcmToken();
      return TextUtils.isEmpty(oldToken) || !newToken.equals(oldToken);
    } else {
      return false;
    }
  }

  private void trackDeviceAttributeForRegistration(Context context, String pushRegisteredBy) {
    try {
      JSONObject attribute = new JSONObject();
      attribute.put(MoEConstants.PUSH_REGISTRATION_ATTRIBUTE, pushRegisteredBy);
      MoEDispatcher.getInstance(context).setDeviceAttribute(attribute);
    } catch (Exception e) {
      Logger.f("PushManager: trackDeviceAttributeForRegistration() : ", e);
    }
  }

  void setTokenListener(OnTokenReceivedListener listener){
    tokenListener = listener;
  }
}
