/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.rest;

import android.support.annotation.Nullable;
import com.moengage.core.Logger;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.ProtocolException;
import java.net.URL;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import javax.net.ssl.HttpsURLConnection;
import org.json.JSONObject;

/**
 * @author Umang Chamaria
 */
public class RestClient {

  private String TAG = "";

  private Request request;

  public RestClient(Request request) {
    this.request = request;
    TAG = "RestClient" + " " + request.uri.getEncodedPath() + " " + request.requestType;
  }

  @Nullable public Response executeRequest() {
    return execute();
  }

  private Response execute() {
    HttpURLConnection urlConnection = null;
    Response response = null;
    try {
      String urlString = request.uri.toString();
      URL url = new URL(urlString);
      Logger.v(TAG + " execute(): Request url: " + urlString);
      if (RestConstants.SCHEME_HTTPS.equals(request.uri.getScheme())) {
        urlConnection = (HttpsURLConnection) url.openConnection();
      } else {
        urlConnection = (HttpURLConnection) url.openConnection();
      }
      addHeaders(urlConnection, request.headersMap);
      setContentType(urlConnection, request.contentType);
      setRequestType(urlConnection, request.requestType);
      addConnectionTimeOut(urlConnection, request.timeOut);
      if (request.requestBody != null && request.requestBody.length() > 0) {
        addBody(urlConnection, request.requestBody);
      }
      int responseCode = urlConnection.getResponseCode();
      Logger.v(TAG + " execute(): Response Code: " + responseCode);
      if (responseCode == 200) {
        String responseBody = convertStreamToString(urlConnection.getInputStream());
        Logger.v(TAG + " execute() : Response Body: " + responseBody);
        response = new Response(responseCode, responseBody);
      } else {
        String errorResponse = convertStreamToString(urlConnection.getErrorStream());
        Logger.f(TAG + " Response: API Failed: "
            + " response code :"
            + responseCode
            + "reason : "
            + errorResponse);
        Logger.e(TAG + " execute() : Error message: " + errorResponse);
        response = new Response(responseCode, errorResponse);
      }
    } catch (Exception e) {
      Logger.e( TAG + " execute() : Exception: ", e);
    }finally {
      if (urlConnection != null) {
        urlConnection.disconnect(); 
      }
    }
    return response;
  }

  private void addConnectionTimeOut(HttpURLConnection urlConnection, int timeOut) {
    urlConnection.setConnectTimeout(timeOut * 1000);
    urlConnection.setReadTimeout(timeOut * 1000);
  }

  private void setContentType(HttpURLConnection urlConnection, String contentType) {
    urlConnection.setRequestProperty(CONTENT_TYPE, contentType);
  }

  private void setRequestType(HttpURLConnection urlConnection,
      RequestBuilder.RequestType requestType) throws ProtocolException {
    urlConnection.setRequestMethod(requestType.toString());
  }

  private void addBody(HttpURLConnection urlConnection, JSONObject requestBody) throws IOException {
    urlConnection.setDoOutput(true);
    urlConnection.setRequestProperty(CHARACTER_SET, ENCODING_CHARSET_UTF8);
    urlConnection.setRequestProperty(CONTENT_TYPE, RestConstants.DEFAULT_CONTENT_TYPE);
    OutputStream output = urlConnection.getOutputStream();
    if (requestBody != null) {
      Logger.v(TAG + " addBody(): Request Body: " + requestBody.toString());
      output.write(requestBody.toString().getBytes(ENCODING_CHARSET_UTF8));
    }
    output.close();
  }

  private void addHeaders(HttpURLConnection urlConnection, Map<String, String> headers) {
    Set<Entry<String, String>> headerMap = headers.entrySet();
    for (Map.Entry<String, String> header : headerMap) {
      Logger.v(TAG + " addHeaders(): " + header.getKey() + " : " + header.getValue());
      urlConnection.addRequestProperty(header.getKey(), header.getValue());
    }
  }

  private String convertStreamToString(InputStream inputStream) throws IOException {
    if (inputStream == null) return null;
    BufferedReader reader = new BufferedReader(new InputStreamReader(inputStream));
    StringBuilder stringBuilder = new StringBuilder();

    String line;
    try {
      while ((line = reader.readLine()) != null) {
        stringBuilder.append(line);
      }
    } finally {
      inputStream.close();
    }
    return stringBuilder.toString();
  }

  private static final String ENCODING_CHARSET_UTF8 = "UTF-8";
  private static final String CHARACTER_SET = "Accept-Charset";
  private static final String CONTENT_TYPE = "Content-type";
}