/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 * All rights reserved.
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core;

import android.app.Activity;
import android.app.job.JobParameters;
import android.content.Context;
import android.content.Intent;
import android.database.Cursor;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.support.annotation.RestrictTo;
import android.support.annotation.RestrictTo.Scope;
import android.support.annotation.WorkerThread;
import android.text.TextUtils;
import com.moe.pushlibrary.MoEHelper;
import com.moe.pushlibrary.PayloadBuilder;
import com.moe.pushlibrary.models.Event;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moe.pushlibrary.utils.MoEHelperUtils;
import com.moengage.core.analytics.AnalyticsHelper;
import com.moengage.core.events.MoEEventManager;
import com.moengage.core.executor.ITask;
import com.moengage.core.executor.OnTaskCompleteListener;
import com.moengage.core.executor.SDKTask;
import com.moengage.core.executor.TaskProcessor;
import com.moengage.core.executor.TaskResult;
import com.moengage.core.inapp.InAppManager;
import com.moengage.core.listeners.OnAppBackgroundListener;
import com.moengage.core.listeners.OnLogoutCompleteListener;
import com.moengage.core.mipush.MiPushManager;
import com.moengage.core.model.RemoteConfig;
import com.moengage.core.pushamp.PushAmpManager;
import com.moengage.core.remoteconfig.ConfigApiNetworkTask;
import com.moengage.core.reports.ReportsBatchHelper;
import com.moengage.core.userattributes.MoEAttributeManager;
import com.moengage.location.GeoManager;
import com.moengage.push.PushHandler;
import com.moengage.push.PushManager;
import com.moengage.push.TokenHandler;
import java.util.HashMap;
import java.util.Locale;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import org.json.JSONObject;

import static com.moe.pushlibrary.MoEHelper.isAppInForeground;

/**
 * @author Umang Chamaria
 */
public class MoEDispatcher implements OnTaskCompleteListener {

  private static final String TAG = "MoEDispatcher";

  private Context context;
  private TaskProcessor taskProcessor;

  private static MoEDispatcher instance;

  //flag to delete data after interaction data is sent.
  boolean shouldClearData = false;

  /**
   * List of all the running tasks
   */
  private HashMap<String, Boolean> runningTaskList;

  private boolean shouldTrackUniqueId = false;

  private JSONObject uniqueIdAttribute = null;

  private ScheduledExecutorService scheduler;

  private OnLogoutCompleteListener logoutCompleteListener;

  private MoEAttributeManager attributeManager = null;

  private DeviceAddManager deviceAddManager = null;

  private MoECoreEvaluator coreEvaluator = null;

  private ReportsBatchHelper batchHelper = null;

  /**
   * MoEDispatcher constructor. There should be only one instance of
   * MoEDispatcher per task instance of an APP
   *
   * @param context Application Context
   */
  private MoEDispatcher(Context context) {
    if (context != null) {
      this.context = context;
      taskProcessor = TaskProcessor.getInstance();
      runningTaskList = new HashMap<>();
      taskProcessor.setOnTaskCompleteListener(this);
      attributeManager = new MoEAttributeManager(context);
    } else {
      Logger.e("MoEDispatcher  : context is null");
    }
  }

  public static MoEDispatcher getInstance(Context context) {
    if (instance == null) {
      synchronized (MoEDispatcher.class) {
        if (instance == null) instance = new MoEDispatcher(context);
      }
    }
    return instance;
  }

  /**
   * Forcefully try to show in-apps
   *
   * @param force if true tries to show in-app
   */
  public void checkForInAppMessages(boolean force) {
    Logger.v("MoEDispatcher: showInAppIfPossible: Check in app messages");
    if (force) {
      //InAppController.getInstance().showInAppIfEligible(mContext);
    }
  }

  /**
   * Lifecycle callback of an activity lifecycle
   *
   * @param activity activity reference {@link Activity}
   * @param intent intent calling that activity
   */
  public void onStart(Activity activity, Intent intent) {
    if (!RemoteConfig.getConfig().isAppEnabled) return;
    if (null == activity) {
      Logger.e("MoEDispatcher:onStart activity instance is null");
      return;
    }
    if (null == intent) {
      intent = activity.getIntent();
    }
    context = activity.getApplicationContext();
      Logger.v("MoEDispatcher:onStart ----");
      MoEHelperUtils.dumpIntentExtras(intent);
      addTaskToQueue(
          new ActivityStartTask(activity));
    InAppManager.getInstance().showInAppIfRequired(context);

    if (MoEHelper.getActivityCounter() == 1) {
      pushTokenFallBack();
    }
    MoEUtils.updateTestDeviceState(this.context);
  }

  private void pushTokenFallBack() {
    PushHandler pushHandler = PushManager.getInstance().getPushHandler();
    if (pushHandler != null) {
      pushHandler.setPushRegistrationFallback(context);
    }
  }

  /**
   * Logs impression for notification click
   *
   * @param gtime notification clicked time
   */
  public void trackNotificationClicked(long gtime) {
    if (!RemoteConfig.getConfig().isAppEnabled) return;
    addTaskToQueue(new NotificationClickedTask(context, gtime));
  }

  /**
   * Saves a user attribute
   *
   * @param userJson user attribute in json format
   */
  public void setUserAttribute(JSONObject userJson) {
    attributeManager.setUserAttribute(userJson);
  }

  public void setCustomUserAttribute(JSONObject userJson) {
    attributeManager.setCustomUserAttribute(userJson);
  }

  /**
   * Cancel gcm registration fallback.
   */
  void cancelRegistrationFallback() {
    MoEUtils.setRegistrationScheduled(context, false);
    MoEUtils.saveCurrentExponentialCounter(context, 1);
  }

  public void onResume(Activity activity, boolean isRestoring) {
    if (!RemoteConfig.getConfig().isAppEnabled) return;
    if (!isRestoring) {
      showDialogAfterPushClick(activity);
    }
  }

  /**
   * Lifecycle callback of activity lifecycle
   *
   * @param activity activity reference {@link Activity}
   */
  public void onStop(Activity activity) {
    if (!RemoteConfig.getConfig().isAppEnabled) return;
    if (null == activity) return;
    addTaskToQueue(new ActivityStopTask(context, activity.getClass().getName()));
  }

  /**
   * Logs out the user. Clears the data and tries to re-register for push in case push
   * registration is handled by MoEngage.
   */
  @RestrictTo(Scope.LIBRARY)
  @WorkerThread public void handleLogout(boolean isForcedLogout) {
    Logger.i("Started logout process");
    if (!RemoteConfig.getConfig().isAppEnabled) return;
    trackLogoutEvent(isForcedLogout);
    addTaskToQueueBeginning(new SendInteractionDataTask(context));
    shouldClearData = true;
  }

  @WorkerThread private void trackLogoutEvent(boolean isForcedLogout) {
    try {
      PayloadBuilder eventAttributes = new PayloadBuilder();
      if (isForcedLogout) eventAttributes.putAttrString("type", "forced");
      eventAttributes.setNonInteractive();
      Event event =
          new Event(MoEConstants.LOGOUT_EVENT, eventAttributes.build());
      MoEDAO.getInstance(context).addEvent(event);
    } catch (Exception e) {
      Logger.f("MoEDispatcher: trackLogoutEvent(): ", e);
    }
  }


  @WorkerThread void clearDataOnLogout() {
    shouldClearData = false;
    Logger.i("Completed logout process");
  }

  /**
   * Sends all the tracked events to the server.
   */
  public void sendInteractionData() {
    startTask(new SendInteractionDataTask(context));
  }

  public void sendInteractionData(OnJobComplete jobComplete, JobParameters parameters){
    startTask(new SendInteractionDataTask(context, jobComplete, parameters));
  }

  /**
   * Updates the local DB that the specified message has been clicked<br>
   * <b>Note : Don't call on UI Thread</b>.
   *
   * @param id The id associated with the inbox message that was clicked
   */
  @WorkerThread public void setInboxMessageClicked(long id) {
    MoEDAO.getInstance(context).setMessageClicked(id);
  }

  /**
   * Handles the app update event, registers,and tries to get new GCM registration ID<br>
   * <b>Note : Don't call on UI Thread</b>
   */
  @WorkerThread void handleAppUpdateEvent() {
    //Logging an update event
    try {
      if (!RemoteConfig.getConfig().isAppEnabled) return;
      int prevVersion = ConfigurationProvider.getInstance(context).getStoredAppVersion();
      PayloadBuilder eventObj = new PayloadBuilder();
      eventObj.putAttrInt(MoEHelperConstants.FROM_VERSION, prevVersion);
      eventObj.putAttrInt(MoEHelperConstants.TO_VERSION, ConfigurationProvider.getInstance(context).getAppVersion());
      //update event
      Logger.i("Adding an update event");
      MoEEventManager.getInstance(context).trackEvent(MoEHelperConstants.EVENT_APP_UPD, eventObj);
      if (!isAppInForeground()) {
        sendInteractionData();
      }
    } catch (Exception e) {
      Logger.f("Adding update event", e);
    }
  }

  /**
   * Get a all inbox messages<br>
   * <b>Note : Don't call on UI Thread</b>
   *
   * @return A populated cursor with the inbox messages or null
   */
  @Nullable @WorkerThread public Cursor getAllMessages() {
    return MoEDAO.getInstance(context).getMessages(context);
  }

  /**
   * Returns the number of unread messages. Unread messages are counted based on clicks<br>
   * <b>Note : Don't call on UI Thread</b>
   *
   * @return The Count of unread messages
   */
  @WorkerThread public int getUnreadMessageCount() {
    return MoEDAO.getInstance(context).getUnreadMessageCount();
  }

  /**
   * Initializes the MoEngage SDK
   *
   * @param senderId GCM Project's Sender ID
   * @param appId Application Id from the MoEngage Dashboard
   */

  public void initialize(final String senderId, final String appId) {
    if (!TextUtils.isEmpty(appId)) {
      SdkConfig.getConfig().appId = appId;
      if (!MoEUtils.isEmptyString(senderId)) SdkConfig.getConfig().senderId = senderId;
      //force registration for push
      dispatchPushTask(TokenHandler.REQ_REGISTRATION);
    } else {
      Logger.e("MoEDispatcher: initialize : AppId is null");
    }
  }

  /**
   * Show a dialog after push was clicked
   *
   * @param activity The instance of the activity where the dialog needs to be created
   */
  private void showDialogAfterPushClick(Activity activity) {
    if (null == activity) return;
    try {
      Intent intent = activity.getIntent();
      if (intent != null) {
        Bundle extras = intent.getExtras();
        if (extras != null && extras.containsKey(MoEHelperConstants.GCM_EXTRA_SHOW_DIALOG)) {
          //remove the extra
          intent.removeExtra(MoEHelperConstants.GCM_EXTRA_SHOW_DIALOG);
          if (extras.containsKey(MoEHelperConstants.GCM_EXTRA_COUPON_CODE)) {
            MoEUtils.showCouponDialog(extras.getString(MoEHelperConstants.GCM_EXTRA_CONTENT),
                extras.getString(MoEHelperConstants.GCM_EXTRA_COUPON_CODE), activity);
            intent.removeExtra(MoEHelperConstants.GCM_EXTRA_CONTENT);
            intent.removeExtra(MoEHelperConstants.GCM_EXTRA_COUPON_CODE);
          } else {
            MoEUtils.showNormalDialogWithOk(extras.getString(MoEHelperConstants.GCM_EXTRA_CONTENT),
                activity);
            intent.removeExtra(MoEHelperConstants.GCM_EXTRA_CONTENT);
          }
        }
      }
    } catch (Exception e) {
      Logger.f("MoEDispatcher: showDialogAfterPushClick : ", e);
    }
  }

  /**
   * Checks whether the task is synchronous or not and adds to the processing queue accordingly.
   * If task is synchronous and is in runningTaskList do not add to processing queue else add
   * in processing queue
   *
   * @param task task to be add to the processor
   */
  public void addTaskToQueue(ITask task) {
    Logger.v("Trying to add " + task.getTaskTag() + " to the queue");
    if (canAddTaskToQueue(task)){
      Logger.v(task.getTaskTag() + " added to queue");
      runningTaskList.put(task.getTaskTag(), task.isSynchronous());
      taskProcessor.addTask(task);
    }else {
      Logger.v(TAG + " addTaskToQueue() : Task is already queued. Cannot add it to queue. Task : " + task.getTaskTag());
    }
  }

  /**
   * Checks whether the task is synchronous or not and adds to the processing queue accordingly.
   * If task is synchronous and is in runningTaskList do not add to processing queue else add
   * to front of processing queue
   *
   * @param task task to be add to the processor
   */
  public void addTaskToQueueBeginning(ITask task) {
    Logger.v("Trying to add " + task.getTaskTag() + " to the queue");
    if (canAddTaskToQueue(task)){
      Logger.v(task.getTaskTag() + " added to beginning of queue");
      runningTaskList.put(task.getTaskTag(), task.isSynchronous());
      taskProcessor.addTaskToFront(task);
    }else {
      Logger.v(TAG + " addTaskToQueueBeginning() : Task is already queued. Cannot add it to queue.");
    }
  }

  public void startTask(ITask task){
    Logger.v(TAG + " startTask() : Try to start task " + task.getTaskTag());
    if (canAddTaskToQueue(task)) {
      Logger.v(TAG + " Starting task " + task.getTaskTag());
      runningTaskList.put(task.getTaskTag(), task.isSynchronous());
      taskProcessor.startTask(task);
    } else {
      Logger.v(
          TAG + " startTask() : Cannot start task. Task is already in progress or queued. " + task.getTaskTag());
    }
  }

  @Override public void onTaskComplete(String tag, TaskResult taskResult) {
    // removing task from running list
    Logger.v("Task completed : " + tag);
    if (runningTaskList.containsKey(tag)) {
      runningTaskList.remove(tag);
    }
    switch (tag) {
      case SDKTask.TAG_SEND_INTERACTION_DATA:
        if (shouldClearData) {
          addTaskToQueueBeginning(new LogoutTask(context));
        }
        break;
      case SDKTask.TAG_TRACK_ATTRIBUTE:
        if (!taskResult.isSuccess()) {
          shouldTrackUniqueId = true;
          uniqueIdAttribute = (JSONObject) taskResult.getPayload();
        }
        break;
      case SDKTask.TAG_DEVICE_ADD:
        deviceAddManager.processTaskResult(context, taskResult);
        break;
      case SDKTask.TAG_SYNC_CONFIG_API:
        onConfigApiSyncComplete(taskResult);
        break;
      case SDKTask.TAG_LOGOUT_TASK:
        if (shouldTrackUniqueId) trackChangedUniqueId();
    }
  }

  private void dispatchPushTask(String extra) {
    PushHandler pushHandler = PushManager.getInstance().getPushHandler();
    if (pushHandler != null) {
      pushHandler.offLoadToWorker(context, extra);
    }
  }

  public void logPushFailureEvent(Context context, String e) {
    // intentionally not removed. Can cause crashes when main sdk and add on not updated in tandem.
  }

  @RestrictTo(Scope.LIBRARY_GROUP)
  public void syncConfigIfRequired() {
    if (ConfigurationProvider.getInstance(context).getLastConfigSyncTime()
        + MoEConstants.CONFIG_API_SYNC_DELAY < MoEUtils.currentTime()) {
      startTask(new ConfigApiNetworkTask(context));
    }
  }

  public void trackDeviceLocale() {
    try {
      if (!RemoteConfig.getConfig().isAppEnabled) return;
      trackDeviceAndUserAttribute("LOCALE_COUNTRY", Locale.getDefault().getCountry());;
      trackDeviceAndUserAttribute("LOCALE_COUNTRY_DISPLAY", Locale.getDefault().getDisplayCountry());
      trackDeviceAndUserAttribute("LOCALE_LANGUAGE", Locale.getDefault().getLanguage());
      trackDeviceAndUserAttribute("LOCALE_LANGUAGE_DISPLAY",
          Locale.getDefault().getDisplayLanguage());
      trackDeviceAndUserAttribute("LOCALE_DISPLAY", Locale.getDefault().getDisplayName());
      trackDeviceAndUserAttribute("LOCALE_COUNTRY_ ISO3", Locale.getDefault().getISO3Country());
      trackDeviceAndUserAttribute("LOCALE_LANGUAGE_ISO3", Locale.getDefault().getISO3Language());
    } catch (Exception e) {
      Logger.f("MoEDispatcher : trackDeviceLocale", e);
    }
  }

  private void trackDeviceAndUserAttribute(String attrName, String attrValue) {
    try {
      JSONObject attribute = new JSONObject();
      attribute.put(attrName, attrValue);
      setUserAttribute(attribute);
      //setDeviceAttribute(attribute);
    } catch (Exception e) {
      Logger.f("MoEDispatcher: trackDeviceAndUserAttribute() ", e);
    }
  }

  public void logoutUser(final boolean isForcedLogout) {
    try {
      Bundle extras = new Bundle();
      extras.putBoolean(MoEConstants.SERVICE_LOGOUT_TYPE, isForcedLogout);
      addTaskToQueue(new MoEWorkerTask(context, MoEConstants.SERVICE_TYPE_LOGOUT, extras));
    } catch(Exception e){
      Logger.f("MoEDispatcher: logoutUser() ", e);
    }
  }

  private void trackChangedUniqueId() {
    if (uniqueIdAttribute != null) {
      setUserAttribute(uniqueIdAttribute);
      uniqueIdAttribute = null;
      shouldTrackUniqueId = false;
    }
  }

  public void setAlias(JSONObject aliasJSON){
   if (!RemoteConfig.getConfig().isAppEnabled) return;
    addTaskToQueue(new SetAliasTask(context, aliasJSON));
  }

  public void setDeviceAttribute(JSONObject deviceAttribute){
    addTaskToQueue(new SetDeviceAttributeTask(context, deviceAttribute));
  }

  private void schedulePeriodicFlushIfRequired() {
    try {
      Logger.v(TAG + " schedulePeriodicFlushIfRequired() : Will try to schedule periodic flush if"
          + " enabled.");
      if (RemoteConfig.getConfig().isPeriodicFlushEnabled
          && SdkConfig.getConfig().isPeriodicFlushEnabled) {
        Runnable syncRunnable = new Runnable() {
          @Override public void run() {
            Logger.v("MoEDispatcher: schedulePeriodicFlushIfRequired() inside runnable, will sync "
                + "now");
            sendInteractionData();
          }
        };
        long timeDelay = RemoteConfig.getConfig().periodicFlushTime;
        if (SdkConfig.getConfig().flushInterval > timeDelay){
          timeDelay = SdkConfig.getConfig().flushInterval;
        }
        Logger.v("MoEDispatcher: schedulePeriodicFlushIfRequired() scheduling periodic sync");
        scheduler = Executors.newScheduledThreadPool(1);
        scheduler.scheduleWithFixedDelay(syncRunnable, timeDelay, timeDelay, TimeUnit.SECONDS);
      }
    } catch (Exception e) {
      Logger.e("MoEDispatcher: schedulePeriodicFlushIfRequired() ", e);
    }
  }

  private void shutDownPeriodicFlush() {
    try {
      if (RemoteConfig.getConfig().isPeriodicFlushEnabled
          && SdkConfig.getConfig().isPeriodicFlushEnabled
          && scheduler != null) {
        Logger.v("MoEDispatcher: shutDownPeriodicFlush() shutting down periodic flush");
        scheduler.shutdownNow();
      }
    } catch (Exception e) {
      Logger.f("MoEDispatcher: shutDownPeriodicFlush() ", e);
    }
  }

  public void onAppOpen() {
    try {
      syncConfigIfRequired();
      updateFeatureConfigForOptOutIfRequired();
      GeoManager.getInstance().updateFenceAndLocation(context);
      InAppManager.getInstance().syncInAppsIfRequired(context);

      PushAmpManager.getInstance().forceServerSync(context, true);
      PushHandler pushHandler = PushManager.getInstance().getPushHandler();
      if (pushHandler != null){
        pushHandler.offLoadToWorker(context, TokenHandler.REG_ON_APP_OPEN);
      }
      schedulePeriodicFlushIfRequired();
      MoEDTManager.getInstance().forceSyncDeviceTriggers(context);
      MiPushManager.getInstance().initMiPush(MoEHelper.getInstance(context).getApplication());
    } catch (Exception e) {
      Logger.f("MoEDispatcher: onAppOpen() ", e);
    }
  }

  void updateFeatureConfigForOptOutIfRequired(){
    ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
    if (provider.isDataTrackingOptedOut()){
      // gaid optOut
      SdkConfig.getConfig().isGaidTrackingOptedOut = true;
      //androidId optOut
      SdkConfig.getConfig().isAndroidIdTrackingOptedOut = true;
      //location optOut
      SdkConfig.getConfig().isLocationTrackingOptedOut = true;
      //geo-fence optOut
      SdkConfig.getConfig().isGeofenceTrackingOptedOut = true;
      //device attributes optOut
      SdkConfig.getConfig().isDeviceAttributeTrackingOptedOut = true;
      //setting location services state
      SdkConfig.getConfig().isLocationServiceEnabled = false;
    }
    if (provider.isPushNotificationOptedOut()) {
      provider.clearPushToken();
    }
  }

  public void setOnLogoutCompleteListener(OnLogoutCompleteListener listener) {
    logoutCompleteListener = listener;
  }

  public void removeOnLogoutCompleteListener() {
    logoutCompleteListener = null;
  }

  void notifyLogoutCompleteListener() {
    Logger.v(TAG + " notifyLogoutCompleteListener() : Notifying listeners");
    if (logoutCompleteListener != null){
      logoutCompleteListener.logoutComplete();
    }
  }

  void onAppClose() {
    Logger.v(TAG + " onAppClose(): Application going to background.");
    // notify when app goes to background
    notifyOnAppBackground();
    // retry device add if token and app id is available
    getDeviceAddManager().retryDeviceRegistrationIfRequired(context);
    // shutdown periodic flush
    shutDownPeriodicFlush();
    //schedule device triggers
    MoEDTManager.getInstance().scheduleBackgroundSync(context);
    // save sent screens
    ConfigurationProvider.getInstance(context).saveSentScreenNames(ConfigurationCache.getInstance().getSentScreenNames());
    //schedule message pull
    PushAmpManager.getInstance().scheduleServerSync(context);
    GeoManager.getInstance().scheduleBackgroundSync(context);
    // track exit intent
    trackAppExit();
    // update last active time
    AnalyticsHelper.getInstance(context).onAppClose(context);
    InAppManager.getInstance().onAppClose(context);
  }

  private void notifyOnAppBackground(){
    OnAppBackgroundListener listener = MoEHelper.getInstance(context).getOnAppBackgroundListener();
    if (listener != null){
      listener.goingToBackground();
    } else {
      Logger.v(TAG + " execute() : on app background listener not set cannot provide callback");
    }
  }

  private void trackAppExit(){
    MoEHelper.getInstance(context).trackEvent(MoEConstants.MOE_APP_EXIT_EVENT, new PayloadBuilder());
  }

  public MoEAttributeManager getAttributeManager() {
    return attributeManager;
  }

  public DeviceAddManager getDeviceAddManager() {
    if (deviceAddManager == null) {
      deviceAddManager = new DeviceAddManager();
    }
    return deviceAddManager;
  }

  public MoECoreEvaluator getCoreEvaluator() {
    if (coreEvaluator == null) {
      coreEvaluator = new MoECoreEvaluator();
    }
    return coreEvaluator;
  }

  public ReportsBatchHelper getBatchHelper(){
    if (batchHelper == null){
      batchHelper = new ReportsBatchHelper();
    }
    return batchHelper;
  }

  private boolean canAddTaskToQueue(ITask task) {
    if (!task.isSynchronous()) return true;
    return !runningTaskList.containsKey(task.getTaskTag());
  }

  private void onConfigApiSyncComplete(TaskResult taskResult) {
    if (taskResult == null || !taskResult.isSuccess()) return;
    if (MoEUtils.canEnableMiPush(RemoteConfig.getConfig())){
      ConfigurationProvider.getInstance(context).savePushService(MoEConstants.PUSH_SERVICE_MI_PUSH);
      MiPushManager.getInstance().initMiPush(MoEHelper.getInstance(context).getApplication());
    }else {
      ConfigurationProvider.getInstance(context).saveMiPushToken("");
      ConfigurationProvider.getInstance(context).setMiTokenServerSentState(false);
      ConfigurationProvider.getInstance(context).savePushService(MoEConstants.PUSH_SERVICE_FCM);
    }
  }

  public void showInAppFromPush(Bundle bundle){
    try{
      InAppManager.getInstance().showInAppFromPush(context, bundle);
    }catch (Exception e){
      Logger.e( TAG + " showInAppFromPush() : ", e);
    }
  }
}
