/*
 * Copyright (c) 2014-2020 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.core.utils;

import android.content.Context;
import android.net.Uri;
import android.text.TextUtils;
import com.moe.pushlibrary.exceptions.SDKNotInitializedException;
import com.moe.pushlibrary.utils.MoEHelperConstants;
import com.moengage.core.AdvertisingIdClient;
import com.moengage.core.ConfigurationCache;
import com.moengage.core.ConfigurationProvider;
import com.moengage.core.MoEConstants;
import com.moengage.core.MoEUtils;
import com.moengage.core.SdkConfig;
import com.moengage.core.model.BaseRequest;
import com.moengage.core.rest.RequestBuilder;
import com.moengage.core.rest.RequestBuilder.RequestType;
import com.moengage.core.rest.RestConstants;
import java.util.TimeZone;
import org.json.JSONException;

public class RestUtils {

  public static String getAuthority() {
    switch (SdkConfig.getConfig().dataRegion) {
      case REGION_EU:
        return MoEConstants.AUTHORITY_API_V2_EU;
      case REGION_SERV3:
        return MoEConstants.AUTHORITY_SOUTH_EAST_ONE;
      default:
        return MoEConstants.AUTHORITY_API_V2_US;
    }
  }

  public static RequestBuilder getBaseRequestBuilder(Uri uri, RequestType
      requestType, String appId) throws SDKNotInitializedException {
    if (TextUtils.isEmpty(appId)) throw new SDKNotInitializedException("App ID has not been set");
    return new RequestBuilder(uri, requestType).addHeader(
        MoEConstants.REQUEST_HEADER_APP_KEY, appId);
  }


  public static JsonBuilder getDefaultParams(Context context) throws JSONException {
    JsonBuilder params = new JsonBuilder();
    ConfigurationProvider provider = ConfigurationProvider.getInstance(context);
    long currentTime = MoEUtils.currentTime();
    params.putString(MoEConstants.GENERIC_PARAM_V2_KEY_OS, MoEConstants.GENERIC_PARAM_V2_VALUE_OS)
        .putString(MoEConstants.GENERIC_PARAM_V2_KEY_APP_ID, MoEUtils.getAppId())
        .putString(MoEConstants.GENERIC_PARAM_V2_KEY_SDK_VERSION,
            String.valueOf(MoEHelperConstants.LIB_VERSION))
        .putString(MoEConstants.GENERIC_PARAM_V2_KEY_UUID,
            provider.getCurrentUserId())
        .putString(MoEConstants.GENERIC_PARAM_V2_KEY_TIMESTAMP, String.valueOf(currentTime))
        .putString(MoEConstants.GENERIC_PARAM_V2_KEY_TIMEZONE_OFFSET,
            String.valueOf(TimeZone.getDefault().getOffset(currentTime)))
        .putString(MoEConstants.GENERIC_PARAM_V2_KEY_APP_VERSION, Integer.toString
            (provider.getAppVersion()));

    String integrationType = ConfigurationCache.getInstance().getIntegrationType();
    String integrationVersion = ConfigurationCache.getInstance().getIntegrationVersion();
    if (!MoEUtils.isEmptyString(integrationType) && !MoEUtils.isEmptyString(integrationVersion)) {
      params.putString(MoEConstants.GENERIC_PARAM_V2_KEY_INTEGRATION_TYPE, integrationType)
          .putString(MoEConstants.GENERIC_PARAM_V2_KEY_INTEGRATION_VERSION, integrationVersion);

    }

    if (!provider.isDataTrackingOptedOut()) {
      params.putString(MoEConstants.GENERIC_PARAM_V2_KEY_APP_VERSION_NAME,
          provider.getAppVersionName());
      if (!SdkConfig.getConfig().isGaidTrackingOptedOut) {
        String gaid = provider.getStoredGAID();
        if (TextUtils.isEmpty(gaid)) {
          AdvertisingIdClient.AdInfo adInfo = MoEUtils.getAdvertisementInfo(context);
          if (adInfo != null) {
            gaid = adInfo.getId();
            provider.storeGAID(gaid);
          }
        }
        if (!TextUtils.isEmpty(gaid)) {
          params.putString(MoEConstants.GENERIC_PARAM_V2_KEY_GAID, gaid);
        }
      }
    }

    params.putString(MoEConstants.GENERIC_PARAM_V2_KEY_PUSH_SERVER, provider.getPushService());
    return params;
  }

  public static Uri.Builder getBaseUriBuilder(){
    return new Uri.Builder()
        .scheme(RestConstants.SCHEME_HTTPS)
        .encodedAuthority(RestUtils.getAuthority());
  }

  public static BaseRequest getBaseRequest(Context context) throws JSONException {
    return new BaseRequest(MoEUtils.getAppId(), RestUtils.getDefaultParams(context),
        ConfigurationProvider.getInstance(context).getCurrentUserId());
  }
}